;;; jde.el -- Integrated Development Environment for Java.

;; Author: Paul Kinnucan <paulk@mathworks.com>
;; Maintainer: Paul Kinnucan
;; Version 1.2
;; Keywords: java, tools

;; Copyright (C) 1997 Free Software Foundation, Inc.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; This package turns Emacs into an integrated development
;; environment for Java. It provides the following
;; capabilities:

;;   -- Specialized editing mode for Java source code
;;      (via cc-mode.el)

;;   -- Java tools commands

;;   -- Compile (compiles source in current buffer)

;;   -- Run App (runs Java app in current buffer)

;;   -- Debug  (jdb, runs debugger)

;;   -- Browse Doc (runs browser on JDK 1.x API doc)

;;   -- Java debug mode
     
;; This mode displays source file being debugged
;; with pointer to current line.

;; This package requires:

;;    -- jde-run.el (included)

;;    -- jde-db.el (included)

;;    -- latest version of cc-mode.el

;;       (available from
;;	 ftp://ftp.python.org/pub/emacs/cc-mode.tar.gz)

;;    -- andersl-java-font-lock.el

;;       (available from 
;;        http:\\www.csd.uu.se\~andersl\emacs.shtml
;;        ftp:\\ftp.csd.uu.se\pub\users\andersl\emacs\)


;;       This is required for syntax highlighting.

;;    -- Java Development Kit (jdk) 
     
;;       (available from
;;	 http://www.javasoft.com/products/jdk/1.1/index.html)

;;       -- Web browser

;;       (e.g., Netscape or Internet Explorer)

;;    -- Unix-style shell for Windows 95 or Windows/NT

;;    I recommend bash, available from www.cygnus.com.

;; To install this package:

;; 1. Copy jde.el, jde-db.el, and jde-run.el files
;; to a directory in your emacs lisp file load
;; path.

;; 2. Byte-compile these files if you like.

;; 3. Add the following code to your .emacs file:

;;    (load "jde")
;;    (setq jde-web-browser "BROWSER")
;;    (setq jde-doc-dir "JDK DIRECTORY")

;; For example, suppose that Netscape is your
;; browser and is in your shell command path
;; and you have installed the JDK API doc at
;; c:/JDK1.1/docs. Then, you would add

;;    (load "jde")
;;    (setq jde-web-browser "netscape")
;;    (setq jde-doc-dir "c:/jdk1.1/docs/")

;; to your .emacs file.

;; If you are using the bash shell, you should 
;; specify this either by setting the SHELL
;; environment variable or the elisp variable
;; shell-file-name to bash. For example, for
;; Win95, enter
   
;;    set shell=bash

;; in your autoexec.bat file or

;;    (setq shell-file-name "bash")

;; in your .emacs file.

;; If you are using bash on Windows 95 or Windows/NT, you should
;; also put 

;;    (setq shell-command-switch "-c")

;; in your .emacs file.


;; 4. Add the JDK bin directory to your shell's
;; command path, if necessary

;; 5. Set your shell's Java CLASSPATH environment variable so
;; that it includes the path to your Java
;; app code.
;; Please send any comments, bugs, or upgrade requests to
;; Paul Kinnucan at paulk@mathworks.com.


;;; Code:

(require 'jde-db)
(require 'jde-run)
(require 'compile)

(defvar jde-web-browser  "netscape"
  "Browser for viewing Java doc")

(defvar jde-doc-dir "d:\\jdk1.1\\docs\\"
  "Path to directory containing the index.html for the JDK API doc.")

(defvar jde-compile-options ""
  "javac compile options, e.g., -depend -debug")

(defvar jde-hook nil
  "Hook called by Java IDE's java-mode-hook function.")


(defun jde-browse-jdk-doc ()
  "Displays JDK doc in a web browser. See jde-web-browser
and jde-doc-dir for more information."
  (interactive)
  (let ((cmd (bashify-command
	      (concat jde-web-browser " '"
		       jde-doc-dir "index.html'"))))
  (start-process-shell-command "jdkdoc" nil cmd)))

(defun jde-make-compile-command ()
  "Constructs the java compile command as: javac + options + buffer file name."
  (bashify-command
   (concat "javac " 
	   jde-compile-options " "
	   (file-name-nondirectory buffer-file-name))))

(defun jde-set-compile-options (options)
  "Sets the compile options.
Enter the options as you would on the command line, e.g.,
-depend -verbose."
  (interactive
   "sEnter options: ")
  (setq jde-compile-options options)
;;  (setq compile-command (jde-make-compile-command))
  )

(defun jde-compile (cmd)
  "Compile the Java program in the current buffer."
  (interactive "P")
  (setq compile-command (jde-make-compile-command))

  ;; Force save-some-buffers to use the minibuffer
  ;; to query user about whether to save modified buffers.
  ;; Otherwise, when user invokes jed-compile from
  ;; menu, save-some-buffers tries to popup a menu
  ;; which seems not to be supported--at least on
  ;; the PC.
  (if (eq system-type 'windows-nt)
      (let ((temp last-nonmenu-event))
	;; The next line makes emacs think that jde-compile
	;; was invoked from the minibuffer, even when it
	;; is actually invoked from the menu-bar.
	(setq last-nonmenu-event t)
	(save-some-buffers (not compilation-ask-about-save) nil)
	(setq last-nonmenu-event temp))
    (save-some-buffers (not compilation-ask-about-save) nil))

  (compile-internal compile-command "No more errors"))

(defvar jde-menu-map (make-sparse-keymap "Java") nil)
(define-key jde-menu-map [jdk-doc] '("Browse JDK Doc" . jde-browse-jdk-doc))
(define-key jde-menu-map [debug-java] '("Debug App" . jde-db))
(define-key jde-menu-map [run-java] '("Run App" . jde-run))
(define-key jde-menu-map [compile-options]
	     '("Compile Options" . jde-set-compile-options))
(define-key jde-menu-map [compile] '("Compile" . jde-compile))

(defvar jde-xemacs-menu
  '(["Compile"           jde-compile t]
    ["Compile Options"   jde-set-compile-options t]
    ["Run App"           jde-run t]
    ["Debug App"         jde-db t]
    ["Browse JDK Doc"    jde-browse-jdk-doc t]
    )
  "XEmacs 19 menu for JDE.")

(defun jde-xemacs-menu ()
  (cons "JDE" jde-xemacs-menu))

(defun jde-set-menubar()
  (if (and (not (memq 'infodoc c-emacs-features))
	   (boundp 'current-menubar)
	   current-menubar)
      (if (fboundp 'add-submenu)
	  (add-submenu nil (jde-xemacs-menu))
	(add-menu nil "JDE" jde-xemacs-menu))
    (define-key java-mode-map [menu-bar java]
      (nconc (list "Java") jde-menu-map)))    
)

(defun jde-java-mode-hook ()
  "Java IDE's java-mode-hook function.
This function sets  up the IDE menu and turns on font locking for Java syntax
highlighting. The last thing it does is call the jde-hook 
function, offering the user an opportunity to perform customizations."
  (cond (window-system
	 (jde-set-menubar)
	 (require 'andersl-java-font-lock)
         (turn-on-font-lock)
	 (run-hooks 'jde-hook))))

(add-hook 'java-mode-hook 'jde-java-mode-hook)

(provide 'jde)

;;; jde.el ends here.







