module type BIT_PARSER =
sig
  (** A value of type [(a, b) parser] is a parser which is parameterised
      by a value of type [a] and which builds a value of type [b]. *)
  type ('a, 'b) parser

  (** [bits] is a parser which receives a number of bits to read and
      reads and returns the bits as an int64 *)
  val bits : (int, int64) parser

  (** [parse p v arr] passes the parameter [v] to the parser [p] and
      reads values from the array [arr]. *)
  val parse : ('a, 'b) parser -> 'a -> int64 array -> 'b
end

(** [extract_bits (x, y) arr] returns an int64 built from the bits
    located between positions [x] and [y] in the array [arr]. *)
let extract_bits : int * int -> int64 array -> int64 =
  let read_bits word (lo, hi) =
    if lo >= hi then 0L
    else Int64.shift_right_logical (Int64.shift_left word lo) (64 + lo - hi)
  in
  fun (lo, hi) arr ->
    if lo > hi then invalid_arg "lo > hi"
    else if hi - lo > 64 then invalid_arg "hi - lo > 64"
    else
      let lword = lo  /  64 and hword = hi  /  64
      and lbit  = lo mod 64 and hbit  = hi mod 64 in
      if lword = hword then
        read_bits arr.(lword) (lbit, hbit)
      else if hbit = 0 then
        read_bits arr.(lword) (lbit, 64)
    else
      let x = read_bits arr.(lword) (lbit, 64) in
      let y = read_bits arr.(hword) (0, hbit) in
      Int64.logor (Int64.shift_left (x) hbit) y


module type APPLICATIVE =
sig
  type 'a t
  val pure : 'a -> 'a t
  val (<*>) : ('a -> 'b) t -> 'a t -> 'b t
end

module type MONAD =
sig
  type 'a t
  val return : 'a -> 'a t
  val (>>=) : 'a t -> ('a -> 'b t) -> 'b t
end

module type ARROW =
sig
  type ('a, 'b) t
  val arr : ('a -> 'b) -> ('a, 'b) t
  val (>>>) : ('a, 'b) t -> ('b, 'c) t -> ('a, 'c) t
  val first : ('a, 'b) t -> ('a * 'c, 'b * 'c) t
end

module Arrow_of_monad (M: MONAD) :
  ARROW with type ('a, 'b) t = 'a -> 'b M.t =
struct
  open M
  type ('a, 'b) t = 'a -> 'b M.t
  let arr f x = return (f x)
  let (>>>) f g x = f x >>= fun y -> g y
  let first f (x,y) =
    f x >>= fun x' -> return (x', y)
end

module Applicative_of_monad (M:MONAD) :
  APPLICATIVE with type 'a t = 'a M.t =
struct
  type 'a t = 'a M.t
  let pure = M.return
  let (<*>) l r = M.(
      l >>= fun l' ->
      r >>= fun r' ->
      pure (l' r')
    )
end

module Parser_monad :
sig
  include MONAD
  include BIT_PARSER with type ('a, 'b) parser = 'a -> 'b t
end =
struct
  type state = int * int64 array
  type 'a t = state -> ('a * state)
  type ('a, 'b) parser = 'a -> 'b t
  let return v state = (v, state)
  let (>>=) m k s = let a, s' = m s in k a s'
  let bits n (offset, arr) =
    (extract_bits (offset, offset + n) arr, (offset + n, arr))
  let parse parser v arr = fst (parser v (0, arr))
end

module Parser_applicative :
sig
  include APPLICATIVE
  include BIT_PARSER with type ('a, 'b) parser = 'a -> 'b t
end =
struct
  include Applicative_of_monad(Parser_monad)
  type ('a, 'b) parser = ('a, 'b) Parser_monad.parser
  let bits = Parser_monad.bits
  let parse = Parser_monad.parse
end

module Parser_arrow :
sig
  include ARROW
  include BIT_PARSER with type ('a, 'b) parser = ('a, 'b) t
end =
struct
  include Arrow_of_monad(Parser_monad)
  type ('a, 'b) parser = ('a, 'b) Parser_monad.parser
  let bits = Parser_monad.bits
  let parse = Parser_monad.parse
end

let () = (* Some tests *)
  begin
    assert (extract_bits (0,1) [| 0x8000000000000000L |]
            =
            0x1L);

    assert (extract_bits (0,1) [| 0xFFFFFFFFFFFFFFFFL |]
            =
            0x1L);

    assert (extract_bits (1,2) [| 0x8000000000000000L |]
            =
            0x0L);

    assert (extract_bits (1,2) [| 0xF000000000000000L |]
            =
            0x1L);

    assert (extract_bits (63,64) [| 0x0000000000000001L |]
            =
            0x1L);

    assert (extract_bits (54,64) [| 0x0000000000000301L |]
            =
           0x301L);

    assert (extract_bits (62,65) [| 0x0000000000000302L;
                                    0xFFFFFFFFFFFFFFFFL; |]
            =
            0x5L);


  assert (extract_bits (0,64) [| 0x343753387fc21383L;
                                 0xFFFFFFFFFFFFFFFFL; |]
            =
            0x343753387fc21383L);

  assert (extract_bits (0,0) [| 0x343753387fc21383L;
                                0xFFFFFFFFFFFFFFFFL; |]
            =
            0x0L);

  assert (extract_bits (0,0) [| 0x343753387fc21383L;
                                0xFFFFFFFFFFFFFFFFL; |]
            =
            0x0L);
  end
 
