(**************************************
Compiler Construction 2015
Computer Laboratory 
University of Cambridge 
Timothy G. Griffin (tgg22@cam.ac.uk) 
*****************************************) 
(* 
   This is a defunctionalized (dfc-ed) 
   version of Interpreter 1. 
 
   NOTE: for the sake of simplicity, we 
   postpone the dfc of environments until 
   a (future) transition. 
*) 


open Ast 
open Common 

let verbose = ref false 

type value = 
     | CON of Common.constant 
     | PAIR of value * value 
     | INL of value 
     | INR of value 
     | FUN of closure  (* defunctionalisation of FUN of ((continuation * value) -> value) *) 

and closure = Past.var * expr * env 

and continuation = 
  | ID 
  | C_UNARY of Past.unary_oper     (* fun v -> k (do_unary(op, v))      *) 
               * continuation
  | C_OPER of Past.oper            (* fun v2 -> k (do_oper(op, v1, v2)) *) 
              * value 
              * continuation
  | C_OPER_FST of expr             (* fun v1 -> eval_cps(e2, env, fun v2 -> k (do_oper(op, v1, v2))) *) 
                * env 
                * Past.oper 
                * continuation
  | C_IF of expr                   (* fun v -> if bool_of_value v then .... *) 
           * expr 
           * env 
           * continuation
  | C_PAIR of value                (* fun v2 -> k (PAIR(v1, v2)      *) 
              * continuation
  | C_PAIR_FST of expr             (* fun v1 -> eval_cps(e2, env, fun v2 -> k (PAIR(v1, v2))) *) 
                  * env 
                  * continuation
  | C_FST of continuation          (* fun v -> match v with | PAIR (v1, _) ... *) 
  | C_SND of continuation          (* fun v -> match v with | PAIR (_, v2) ... *) 
  | C_INL of continuation          (* fun v -> k (INL v) *) 
  | C_INR of continuation          (* fun v -> k (INR v) *) 
  | C_CASE of Past.var             (* fun v -> match v with | INL v .... *) 
              * expr 
              * Past.var 
              * expr 
              * env 
              * continuation
  | C_APPLY of closure             (* fun v2 -> f(k, v2) *) 
             * continuation
  | C_ARG of expr                  (* fun v1 -> match v1 with | FUN f .... *) 
             * env 
             * continuation

and env = Past.var -> value 

type state = env * expr 

type binding = Past.var * value

type bindings = binding list

(* update : (env * binding) -> env *) 
let update(env, (x, v)) = fun y -> if x = y then v else env y

let do_unary = function 
  | (Past.NOT, CON (BOOL m)) -> CON(BOOL (not m))
  | (Past.NEG, CON (INT m))  -> CON(INT (-m))
  | (Past.READ, CON(UNIT))   -> CON(INT (readint()))
  | (op, _) -> complain ("malformed unary operator: " ^ (Pp_past.string_of_unary_oper op))

let do_oper = function 
  | (Past.AND, CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m && n))
  | (Past.OR,  CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m || n))
  | (Past.EQB, CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m = n))
  | (Past.LT, CON (INT m), CON (INT n)) -> CON(BOOL (m < n))
  | (Past.EQI, CON (INT m), CON (INT n)) -> CON(BOOL (m = n))
  | (Past.ADD, CON (INT m), CON (INT n)) -> CON(INT (m + n))
  | (Past.SUB, CON (INT m), CON (INT n)) -> CON(INT (m - n))
  | (Past.MUL, CON (INT m), CON (INT n)) -> CON(INT (m * n))
  | (op, _, _)  -> complain ("malformed binary operator: " ^ (Pp_past.string_of_oper op))


(* 
   For each continuation (fun x -> e) that we introduced in interp_1.ml 
   we introduce the associated constructor here.  For example, 

   interp_1.ml: 

    | Pair(e1, e2)     -> 
         eval_cps(e1, env, fun v1 -> eval_cps(e2, env, fun v2 -> k (PAIR(v1, v2))))

   interp_2.ml: 

    | Pair(e1, e2)     -> eval_dfc(e1, env, C_PAIR_FST(e2, env, k))
*) 

let rec eval_dfc (e, env, k) = 
    match e with 
    | Unit             -> apply_cnt(k, CON UNIT) 
    | Var x            -> apply_cnt(k, env x)
    | Integer n        -> apply_cnt(k, CON(INT n))
    | Boolean b        -> apply_cnt(k, CON (BOOL b)) 
    | UnaryOp(op, e)   -> eval_dfc(e, env,  C_UNARY(op, k))
    | Op(e1, op, e2)   -> eval_dfc(e1, env, C_OPER_FST (e2, env, op, k))
    | If(e1, e2, e3)   -> eval_dfc(e1, env, C_IF (e2, e3, env, k))
    | Pair(e1, e2)     -> eval_dfc(e1, env, C_PAIR_FST(e2, env, k))
    | Fst e            -> eval_dfc(e, env, C_FST k)
    | Snd e            -> eval_dfc(e, env, C_SND k) 
    | Inl e            -> eval_dfc(e, env, C_INL k) 
    | Inr e            -> eval_dfc(e, env, C_INR k) 
    | Case(e, (x1, e1), (x2, e2)) -> eval_dfc(e, env, C_CASE(x1, e1, x1, e2, env, k))
    | Lambda(x, body)  -> apply_cnt(k, FUN(x, body, env))
    | App(e1, e2)      -> eval_dfc(e1, env, C_ARG(e2, env, k))
    | LetFun(f, (x, body), e) -> 
       let new_env = update(env, (f, FUN(x, body, env)))
       in eval_dfc(e, new_env, k) 
    | LetRecFun(f, (x, body), e) -> 
       let rec new_env g = (* a recursive environment! *) 
           if g = f then FUN(x, body, new_env) else env g
       in eval_dfc(e, new_env, k) 

(* in interp_1.ml we have 

     | FUN of ((continuation * value) -> value)
 
and we build functions like this 

    | Lambda(x, e)     -> k (FUN (fun (c, v) -> eval_cps(e, update(env, (x, v)), c)))

and apply them like this 

    | FUN f -> eval_cps(e2, env, fun v2 -> f(k, v2)). 

So the important thing is that when we write the apply 
for the definctionalized representation of a Slang.2 
function, then we must expand it to something 
that repersents 

    ( ????, (c, v)) -> eval_cps(body, update(env, (x, v)), c)))

Thus, all we need is a triple of the form (x, body, env), 
which we call a closure. 

Here's the apply function for this representation. 

*) 
and apply_closure = function 
  | ((x, body, env), (c, v)) -> eval_dfc(body, update(env, (x, v)), c)

(* We represent each application of a continiaton "k e" in interp_1.ml 
   as a function call "apply_cnt(k', e')", where k' and e' are interp_2.ml
   representations. 

   For each continuation (fun x -> e) that we introduced in interp_1.ml 
   we have a constructor C so that 

   apply_cnt (C(..), x) -> e'

   where e' is the representation of e. For example 

   interp_1.ml : 

       fun v2 -> k (PAIR(v1, v2)

   results in 

   apply_cnt(C_PAIR (v1, k), v2) = apply_cnt(k, PAIR(v1, v2)). 

   interp_1.ml : 

       fun v1 -> eval_cps(e2, env, fun v2 -> k (PAIR(v1, v2)))

   results in 

   apply_cnt(C_PAIR_FST (e2, env, k),v1) = eval_dfc(e2, env, C_PAIR(v1, k))

*) 
and apply_cnt = function 
  | (ID,                                  v) -> v 
  | (C_UNARY(op, k),                      v) -> apply_cnt(k ,(do_unary(op, v)))
  | (C_OPER(op, v1, k),                  v2) -> apply_cnt(k, do_oper(op, v1, v2))
  | (C_OPER_FST (e2, env, op, k),        v1) -> eval_dfc(e2, env, C_OPER (op, v1, k))
  | (C_PAIR (v1, k),                     v2) -> apply_cnt(k, PAIR(v1, v2))
  | (C_PAIR_FST (e2, env, k),            v1) -> eval_dfc(e2, env, C_PAIR(v1, k))
  | (C_FST k,                    PAIR(v, _)) -> apply_cnt(k, v)
  | (C_SND k,                    PAIR(_, v)) -> apply_cnt(k, v)
  | (C_INL k,                             v) -> apply_cnt(k, (INL v))
  | (C_INR k,                             v) -> apply_cnt(k, (INR v))
  | (C_CASE (x1, e1, x2, e2, env, k), INL v) -> eval_dfc(e1, update(env, (x1, v)), k) 
  | (C_CASE (x1, e1, x2, e2, env, k), INR v) -> eval_dfc(e2, update(env, (x2, v)), k) 
  | (C_APPLY(f, k),                      v2) -> apply_closure(f, (k, v2))
  | (C_ARG (e2, env, k),             FUN cl) -> eval_dfc(e2, env, C_APPLY(cl, k))
  | (C_IF (e2, e3, env, k), CON(BOOL true))  -> eval_dfc(e2, env, k) 
  | (C_IF (e2, e3, env, k), CON(BOOL false)) -> eval_dfc(e3, env, k) 
  | (_, _) -> complain "apply_cnt : run-time error"

let eval(e, env) = eval_dfc(e, env, ID) 

(* env_empty : env *) 
let env_empty = fun x -> complain (x ^ " is not defined!\n")

(* interpret : expr -> value *) 
let interpret e = eval(e, env_empty)

let rec string_of_value = function 
     | CON c -> string_of_constant c
     | PAIR(v1, v2) -> "(" ^ (string_of_value v1) ^ ", " ^ (string_of_value v2) ^ ")"
     | INL v -> "inl(" ^ (string_of_value v) ^ ")"
     | INR  v -> "inr(" ^ (string_of_value v) ^ ")"
     | FUN _ -> "FUNCTION( ... )" 
    

      
    
    
