(**************************************
Compiler Construction 2015
Computer Laboratory 
University of Cambridge 
Timothy G. Griffin (tgg22@cam.ac.uk) 
*****************************************) 

(* 
This is the main file.
*) 

(* 
   parse command line options and args 
*) 
let infile         = ref ""
let verbose        = ref false
let verbose_front  = ref false
let run_tests      = ref false
let use_i0         = ref false
let use_i1         = ref false
let use_i2         = ref false
let use_i3         = ref false
let use_i4         = ref false
let use_i5         = ref false
let use_i6         = ref false
let use_i7         = ref false
let use_i8         = ref false
let use_i9         = ref false
let use_all        = ref false
let set_infile f   = infile := f 

let option_spec = [
     ("-V",    Arg.Set verbose_front, "verbose front end"); 
     ("-v",    Arg.Set verbose,       "verbose interpreter(s)"); 
     ("-i0",   Arg.Set use_i0,        "Interpreter 0 (definitional interpreter)"); 
     ("-i1",   Arg.Set use_i1,        "Interpreter 1 (cps(Interpreter 0))" ); 
     ("-i2",   Arg.Set use_i2,        "Interpreter 2 (dfc(Interpreter 1))" ); 
     ("-i3",   Arg.Set use_i3,        "Interpreter 3 (stackify(Interpreter 2))" ); 
     ("-i4",   Arg.Set use_i4,        "Interpreter 4 (split_stacks(Interpreter 3))" ); 
     ("-i5",   Arg.Set use_i5,        "Interpreter 5 (refactor(Interpreter 4))" ); 
     ("-i6",   Arg.Set use_i6,        "Interpreter 6 (defunctionalise environments of Interpreter 3))" ); 
     ("-i7",   Arg.Set use_i7,        "Interpreter 7 (stackify(Interpreter 6))" ); 
     ("-i8",   Arg.Set use_i8,        "Interpreter 8 (refactor Interpreter 6 to compile + eval)" ); 
     ("-i9",   Arg.Set use_i9,        "Interpreter 9 (make code of Interpreter 8 linear, with addresses)" ); 
     ("-all",  Arg.Set use_all,       "all interpreters"); 
     ("-t",    Arg.Set run_tests,     "run all test/*.slang with each selected interpreter, report unexpected outputs (silent otherwise)")
    ] 
let usage_msg = "Usage: slang.byte [options] [<file>]\nOptions are:"

let _ = Arg.parse option_spec set_infile usage_msg

let _ = if !use_all 
        then (use_i0     := true; 
	      use_i1     := true; 
	      use_i2     := true; 
	      use_i3     := true; 
	      use_i4     := true; 
	      use_i5     := true; 
	      use_i6     := true; 
	      use_i7     := true; 
	      use_i8     := true; 	      
	      use_i9     := true;
             )
        else ()

(* set all verbosity flags *) 
let _ = if !verbose_front then Front_end.verbose := true else () 

let _ = if !verbose 
        then (
              Interp_0.verbose := true; 
              Interp_1.verbose := true; 
              Interp_2.verbose := true; 
              Interp_3.verbose := true; 
              Interp_4.verbose := true; 
              Interp_5.verbose := true; 
              Interp_6.verbose := true; 
              Interp_7.verbose := true; 
              Interp_8.verbose := true; 
              Interp_9.verbose := true; 
             )
        else () 

let error file action s = print_string ("\nERROR in " ^ file ^ " with " ^ action ^ " : " ^ s ^ "\n")

let fatal_error file action s = let _ = error file action s in exit(-1) 

         
(* bind interpreters *) 
(* each interpreter i_ has type "(string * Ast.expr) -> string option" *) 

let wrap file e interpret msg = 
    try Some(interpret e)
    with Common.Error s -> let _ = error file msg s in None 
       | exc -> fatal_error file msg ("Exception: " ^ (Printexc.to_string exc))

let i0 (file, e)   = wrap file e (fun x -> Interp_0.string_of_value (Interp_0.interpret x)) "Interpreter 0" 
let i1 (file, e)   = wrap file e (fun x -> Interp_1.string_of_value (Interp_1.interpret x)) "Interpreter 1" 
let i2 (file, e)   = wrap file e (fun x -> Interp_2.string_of_value (Interp_2.interpret x)) "Interpreter 2" 
let i3 (file, e)   = wrap file e (fun x -> Interp_3.string_of_value (Interp_3.interpret x)) "Interpreter 3" 
let i4 (file, e)   = wrap file e (fun x -> Interp_4.string_of_value (Interp_4.interpret x)) "Interpreter 4" 
let i5 (file, e)   = wrap file e (fun x -> Interp_5.string_of_value (Interp_5.interpret x)) "Interpreter 5" 
let i6 (file, e)   = wrap file e (fun x -> Interp_6.string_of_value (Interp_6.interpret x)) "Interpreter 6" 
let i7 (file, e)   = wrap file e (fun x -> Interp_7.string_of_value (Interp_7.interpret x)) "Interpreter 7" 
let i8 (file, e)   = wrap file e (fun x -> Interp_8.string_of_value (Interp_8.interpret x)) "Interpreter 8" 
let i9 (file, e)   = wrap file e (fun x -> Interp_9.string_of_value (Interp_9.interpret x)) "Interpreter 9" 

let interpreters = [
    (* use-flag, the interpreter, a description string *) 
    (!use_i0,       i0,   "Interpreter 0");
    (!use_i1,       i1,   "Interpreter 1");
    (!use_i2,       i2,   "Interpreter 2");
    (!use_i3,       i3,   "Interpreter 3");
    (!use_i4,       i4,   "Interpreter 4");
    (!use_i5,       i5,   "Interpreter 5");
    (!use_i6,       i6,   "Interpreter 6"); 
    (!use_i7,       i7,   "Interpreter 7");
    (!use_i8,       i8,   "Interpreter 8");
    (!use_i9,       i9,   "Interpreter 9")
] 

let show_output describe string_out = 
    if !run_tests
    then () 
    else let _ = if !verbose then print_string ("\n" ^ describe ^ " : \n") else ()
         in print_string ("output> " ^ string_out ^ "\n")

(* used for -t option *) 
let check_expected describe file string_out = function 
  | None -> () 
  | Some expected -> 
      if string_out = expected 
      then () 
      else print_string (
	      "\nTEST ERROR with " 
              ^ describe 
              ^ " in "  
	      ^ file 
	      ^ " : expected " 
              ^ expected 
	      ^ " but computed " 
              ^ string_out  
	      ^ "\n" )

(* runs all interpreters with input e : Ast.expr. 
   If expected_option = Some (expected), then 
   checks if the correct result was computed
   (silent if OK). 
*) 
let rec run_interpreters file e expected_option = function 
    | [] -> () 
    | (use, interp, describe) :: rest -> 
      if use 
      then (match interp(file, e) with 
           | None -> run_interpreters file e expected_option rest 
           | Some string_out -> 
              let _ = show_output describe string_out 
              in let _ = check_expected describe file string_out expected_option 
              in run_interpreters file e expected_option rest )
      else run_interpreters file e expected_option rest 
   
(* process_inputs : runs all (flagged) interpreters on all inputs *) 
let rec process_inputs = function 
       | [] -> () 
       | (file, expected) :: rest -> 
          let in_chan = try 
                           open_in file 
                        with _ -> fatal_error file "slang" ("can't open file " ^ file) 
          in let e = try 
                       Front_end.get_ast(file, in_chan)  
                     with Common.Error s -> let _ = error file "Front End" s in Ast.Unit 
          in let _ = run_interpreters file e expected interpreters 
          in process_inputs rest 
 
let _ = process_inputs 
        (if !run_tests 
        then (try Tests.get_inputs () with Common.Error s -> fatal_error "tests/" "Test.get_inputs" s)
        else [(!infile, None)])
            




