open Format
open Past

(*
   Documentation of Format can be found here: 
   http://caml.inria.fr/resources/doc/guides/format.en.html
   http://caml.inria.fr/pub/docs/manual-ocaml/libref/Format.html
*) 

let rec string_of_type = function 
  | TEint -> "int" 
  | TEbool -> "bool" 
  | TEunit -> "unit" 
  | TEarrow(t1, t2)   -> "(" ^ (string_of_type t1) ^ " -> " ^ (string_of_type t2) ^ ")" 
  | TEproduct(t1, t2) -> "(" ^ (string_of_type t1) ^ " * " ^ (string_of_type t2) ^ ")"  
  | TEunion(t1, t2)   -> "(" ^ (string_of_type t1) ^ " + " ^ (string_of_type t2) ^ ")"  
  | TEwildcard -> " _ "


let string_of_unary_oper = function 
  | NEG -> "-" 
  | NOT -> "~" 
  | READ -> "read" 


let string_of_oper = function 
  | ADD -> "+" 
  | MUL  -> "*" 
  | SUB -> "-" 
  | LT   -> "<" 
  | EQ   -> "=" 
  | EQI   -> "=" 
  | EQB   -> "=" 
  | AND   -> "&&" 
  | OR   -> "||" 

let fstring ppf s = fprintf ppf "%s" s

let pp_type ppf t = fstring ppf (string_of_type t) 

let pp_unary ppf t = fstring ppf (string_of_unary_oper t) 

let pp_binary ppf t = fstring ppf (string_of_oper t) 

(* ignore locations *) 
let rec pp_expr ppf = function 
    | Unit _              -> fstring ppf "()" 
    | What _              -> fstring ppf "?" 
    | Var (_, x)          -> fstring ppf x 
    | Integer (_, n)      -> fstring ppf (string_of_int n)
    | Boolean (_, b)      -> fstring ppf (string_of_bool b)
    | UnaryOp(_, op, e)   -> fprintf ppf "%a(%a)" pp_unary op pp_expr e 
    | Op(_, e1, op, e2)   -> fprintf ppf "(%a %a %a)" pp_expr e1  pp_binary op pp_expr e2 
    | If(_, e1, e2, e3)   -> fprintf ppf "@[if %a then %a else %a @]" 
                                      pp_expr e1 pp_expr e2 pp_expr e3
    | Pair(_, e1, e2)     -> fprintf ppf "(%a, %a)" pp_expr e1 pp_expr e2
    | Fst(_, e)           -> fprintf ppf "fst(%a)" pp_expr e
    | Snd(_, e)           -> fprintf ppf "snd(%a)" pp_expr e
    | Inl(_, e)           -> fprintf ppf "inl(%a)" pp_expr e
    | Inr(_, e)           -> fprintf ppf "inr(%a)" pp_expr e
    | Case(_, e, (x1, e1), (x2, e2)) -> 
        fprintf ppf "@[<2>case %a of@ | inl %a -> %a @ | inr %a -> %a end@]" 
                     pp_expr e fstring x1 pp_expr e1 fstring x2 pp_expr e2 
    | Lambda(_, (x, t, e)) -> 
         fprintf ppf "(fun %a : %a -> %a)" fstring x pp_type t  pp_expr e
    | App(_, e1, e2)      -> fprintf ppf "%a %a" pp_expr e1 pp_expr e2
    | Let(_, bl, e)       -> fprintf ppf "@[<2>let %a in %a end@]" pp_binding_list bl pp_expr e 
    | LetFun(_, f, (x, t1, e1), t2, e2)     -> 
         fprintf ppf "@[let %a(%a : %a) : %a =@ %a @ in %a @ end@]" 
                     fstring f fstring x  pp_type t1 pp_type t2 pp_expr e1 pp_expr e2
    | LetRecFun(_, f, (x, t1, e1), t2, e2)     -> 
         fprintf ppf "@[letrec %a(%a : %a) : %a =@ %a @ in %a @ end@]" 
                     fstring f fstring x  pp_type t1 pp_type t2 pp_expr e1 pp_expr e2

and pp_binding_list ppf = function 
  | [] -> fstring ppf "" 
  | [(x, t, e)] -> fprintf ppf "%a : %a = %a" fstring x pp_type t pp_expr e
  | (x, t, e):: rest -> 
      fprintf ppf "%a : %a = %a @ and %a" fstring x pp_type t pp_expr e pp_binding_list rest

let print_expr e = 
    let _ = pp_expr std_formatter e
    in print_flush () 

let eprint_expr e = 
    let _ = pp_expr err_formatter e
    in print_flush () 

let string_of_expr e = 
     let _ = pp_expr std_formatter e 
     in flush_str_formatter () 
