open Format
open Ast 

(*
   Documentation of Format can be found here: 
   http://caml.inria.fr/resources/doc/guides/format.en.html
   http://caml.inria.fr/pub/docs/manual-ocaml/libref/Format.html
*) 


let fstring ppf s = fprintf ppf "%s" s

let pp_unary ppf t = fstring ppf (Pp_past.string_of_unary_oper t) 

let pp_binary ppf t = fstring ppf (Pp_past.string_of_oper t) 

let rec pp_expr ppf = function 
    | Unit             -> fstring ppf "()" 
    | Var x            -> fstring ppf x 
    | Integer n        -> fstring ppf (string_of_int n)
    | Boolean b        -> fstring ppf (string_of_bool b)
    | UnaryOp(op, e)   -> fprintf ppf "%a(%a)" pp_unary op pp_expr e 
    | Op(e1, op, e2)   -> fprintf ppf "(%a %a %a)" pp_expr e1  pp_binary op pp_expr e2 
    | If(e1, e2, e3)   -> fprintf ppf "@[if %a then %a else %a @]" 
                                      pp_expr e1 pp_expr e2 pp_expr e3
    | Pair(e1, e2)     -> fprintf ppf "(%a, %a)" pp_expr e1 pp_expr e2
    | Fst e            -> fprintf ppf "fst(%a)" pp_expr e
    | Snd e            -> fprintf ppf "snd(%a)" pp_expr e
    | Inl e            -> fprintf ppf "inl(%a)" pp_expr e
    | Inr e            -> fprintf ppf "inr(%a)" pp_expr e
    | Case(e, (x1, e1), (x2, e2)) -> 
        fprintf ppf "@[<2>case %a of@ | inl %a -> %a @ | inr %a -> %a end@]" 
                     pp_expr e fstring x1 pp_expr e1 fstring x2 pp_expr e2 
    | Lambda(x, e) -> 
         fprintf ppf "(fun %a -> %a)" fstring x pp_expr e
    | App(e1, e2)      -> fprintf ppf "%a %a" pp_expr e1 pp_expr e2
    | LetFun(f, (x, e1), e2)     -> 
         fprintf ppf "@[let %a(%a) =@ %a @ in %a @ end@]" 
                     fstring f fstring x  pp_expr e1 pp_expr e2
    | LetRecFun(f, (x, e1), e2)  -> 
         fprintf ppf "@[letrec %a(%a) =@ %a @ in %a @ end@]" 
                     fstring f fstring x  pp_expr e1 pp_expr e2

let print_expr e = 
    let _ = pp_expr std_formatter e
    in print_flush () 

let eprint_expr e = 
    let _ = pp_expr err_formatter e
    in pp_print_flush err_formatter () 

let string_of_expr e = 
     let _ = pp_expr str_formatter e 
     in flush_str_formatter () 
