(**************************************
Compiler Construction 2015
Computer Laboratory 
University of Cambridge 
Timothy G. Griffin (tgg22@cam.ac.uk) 
*****************************************) 
(* Interpreter 9. 

   Make instructions linear by introducing 
   labels and goto.  labels will be translated
   to numeric addresses. 
*) 


open Ast 
open Common 

let verbose = ref false 

type address = int 

type label = string 

type location = label * (address option) 

type value = 
     | CON of Common.constant 
     | PAIR of value * value 
     | INL of value 
     | INR of value 
     | FUN of bool * closure    (* the bool is a flag : true means recursive environment! *) 

and closure =  location * env 

and instruction = 
  | PUSH of value 
  | UNARY of Past.unary_oper 
  | OPER of Past.oper 
  | TEST of location 
  | CASE of location
  | GOTO of location
  | IPAIR 
  | FST
  | SND
  | IINL
  | IINR
  | APPLY
  | RETURN 
  | SWAP_POP 
  | LOOKUP of Past.var 
  | BIND of Past.var 
  | MK_CLOSURE of location * (Past.var list) 
  | MK_REC of location  * (Past.var list) 
  | SKIP 
  | HALT 

and l_instruction  = NotLabelled of instruction  | Labelled of label * instruction 

and code = l_instruction list 

and binding = Past.var * value

and env = binding list

type env_or_value = 
  | EV of env        (* an environment on the run-time stack *) 
  | V of value       (* a value on the run-time stack *) 
  | RA of address    (* a return address on the run-time stack *) 

type env_value_stack = env_or_value list 

type state = address * env_value_stack 

let installed : (l_instruction array) ref = ref (Array.of_list [NotLabelled HALT])

(* update : (env * binding) -> env *) 
let update(env, (x, v)) = (x, v) :: env 


let mk_fun(loc, env) = FUN(false, (loc, env)) 
let mk_rec(loc, env) = FUN(true, (loc, env)) 
(* 
let mk_rec(loc, env) = match loc with (f, _) -> FUN(true, (loc, (f, FUN(true, (loc, []))) :: env))
*) 
(* 
      for a recursive function f we want 

      lookup (env, f) = FUN(true, (x, body, env))  
*) 

let lookup_opt (env, x) = 
    let rec aux = function 
      | [] -> None 
      | (y, v) :: rest -> 
          if x = y 
          then Some(match v with 
               | FUN(true, (loc, _)) -> FUN(true, (loc, (y,v) :: rest))
               | _ -> v)
          else aux rest  
      in aux env 

let lookup (env, x) = 
    match lookup_opt (env, x) with 
    | None -> complain (x ^ " is not defined!\n")
    | Some v -> v 

let rec search (evs, x) = 
  match evs with 
  | [] -> complain (x ^ " is not defined!\n")
  | (V _) :: rest -> search (rest, x) 
  | (RA _) :: rest -> search (rest, x) 
  | (EV env) :: rest -> 
    (match lookup_opt(env, x) with 
    | None -> search (rest, x) 
    | Some v -> v 
    ) 

let env_of_evs(fvars, evs) = 
 let rec inlist x = function 
   | [] -> false 
   | y :: rest -> (x = y) || (inlist x rest) 
 in let rec filter seen = function 
   | [] -> ([], seen) 
   | (x, v) :: rest -> 
      if (inlist x fvars) && (not (inlist x seen))
      then let (env', seen') = filter (x :: seen) rest in 
           ((x, v) :: env', seen') 
      else (filter seen rest)
 in let rec aux seen= function 
  | [] -> []
  | (V _) :: rest -> aux seen rest 
  | (RA _) :: rest -> aux seen rest 
  | (EV env) :: rest -> 
     let (env', seen') = filter seen env in env' @ (aux seen' rest)
 in aux [] evs 
    
let do_unary = function 
  | (Past.NOT, CON (BOOL m)) -> CON(BOOL (not m))
  | (Past.NEG, CON (INT m))  -> CON(INT (-m))
  | (Past.READ, CON(UNIT))   -> CON(INT (readint()))
  | (op, _) -> complain ("malformed unary operator: " ^ (Pp_past.string_of_unary_oper op))

let do_oper = function 
  | (Past.AND, CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m && n))
  | (Past.OR,  CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m || n))
  | (Past.EQB, CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m = n))
  | (Past.LT, CON (INT m), CON (INT n)) -> CON(BOOL (m < n))
  | (Past.EQI, CON (INT m), CON (INT n)) -> CON(BOOL (m = n))
  | (Past.ADD, CON (INT m), CON (INT n)) -> CON(INT (m + n))
  | (Past.SUB, CON (INT m), CON (INT n)) -> CON(INT (m - n))
  | (Past.MUL, CON (INT m), CON (INT n)) -> CON(INT (m * n))
  | (op, _, _)  -> complain ("malformed binary operator: " ^ (Pp_past.string_of_oper op))


let rec string_of_value = function 
     | CON c           -> string_of_constant c
     | PAIR(v1, v2)    -> "(" ^ (string_of_value v1) ^ ", " ^ (string_of_value v2) ^ ")"
     | INL v           -> "inl(" ^ (string_of_value v) ^ ")"
     | INR  v          -> "inr(" ^ (string_of_value v) ^ ")"
     | FUN (true, cl)  -> "FUN(true, " ^ (string_of_closure cl) ^ ")"
     | FUN (false, cl) -> "FUN(false, " ^ (string_of_closure cl) ^ ")"

and string_of_closure (loc, env) = 
   "(" ^ (string_of_location loc) ^ ", " ^ (string_of_env env) ^ ")"

and string_of_env env = Common.string_of_list ",\n " string_of_binding env 

and string_of_binding (x, v) =    "(" ^ x ^ ", " ^ (string_of_value v) ^ ")"

and string_of_location (l, _) = l 

and string_of_instruction = function 
 | UNARY op     -> "UNARY " ^ (Pp_past.string_of_unary_oper op) 
 | OPER op      -> "OPER " ^ (Pp_past.string_of_oper op) 
 | IPAIR        -> "IPAIR"
 | FST          -> "FST"
 | SND          -> "SND"
 | IINL         -> "IINL"
 | IINR         -> "IINR"
 | SKIP         -> "SKIP"
 | PUSH v       -> "PUSH " ^ (string_of_value v) 
 | LOOKUP x     -> "LOOKUP " ^ x
 | TEST l       -> "TEST " ^ (string_of_location l)
 | CASE l       -> "CASE " ^ (string_of_location l)
 | GOTO l       -> "GOTO " ^ (string_of_location l)
 | APPLY        -> "APPLY"
 | RETURN       -> "RETURN"
 | HALT         -> "HALT"
 | BIND x       -> "BIND " ^ x
 | SWAP_POP     -> "SWAP_POP"
 | MK_CLOSURE(loc, fvars) -> "MK_CLOSURE(" ^ (string_of_location loc) ^ ", [" ^ (Common.string_of_list ", " (fun x -> x) fvars) ^ "])"
 | MK_REC(loc, fvars) -> "MK_REC(" ^ (string_of_location loc) ^ ", [" ^ (Common.string_of_list ", " (fun x -> x) fvars) ^ "])"

and string_of_linstruction = function 
  | NotLabelled i      -> string_of_instruction i
  | Labelled (lab, i) -> lab ^ " : " ^ (string_of_instruction i)

and string_of_code c = Common.string_of_list "\n " string_of_linstruction c 

let string_of_env_or_value = function 
  | EV env -> "EV " ^ (string_of_env env)
  | V v -> "V " ^ (string_of_value v)
  | RA i -> "RA" ^ (string_of_int i)

let string_of_env_value_stack = Common.string_of_list ";\n " string_of_env_or_value 

let get_instruction = function NotLabelled inst -> inst | Labelled(_, inst) -> inst 

let string_of_state (cp, evs) = 
    "\n" ^ (string_of_int cp) ^ " -> " 
         ^ (string_of_linstruction  (Array.get !installed cp)) ^ "\n" 
         ^ (string_of_env_value_stack evs) 
         ^ "\n" 

(* Look Mom, no heap! 
   Oh, wait, we are using Ocaml's heap to implement our stack .... 
*) 
let step (cp, evs) = 
 match (get_instruction (Array.get !installed cp), evs) with 
 | (UNARY op,           (V v) :: evs) -> (cp + 1, V(do_unary(op, v)) :: evs) 
 | (OPER op, (V v2) :: (V v1) :: evs) -> (cp + 1, V(do_oper(op, v1, v2)) :: evs)
 | (IPAIR,   (V v2) :: (V v1) :: evs) -> (cp + 1, V(PAIR(v1, v2)) :: evs)
 | (FST,       V(PAIR (v, _)) :: evs) -> (cp + 1, (V v) :: evs)
 | (SND,       V(PAIR (_, v)) :: evs) -> (cp + 1, (V v) :: evs)
 | (IINL,               (V v) :: evs) -> (cp + 1, V(INL v) :: evs)
 | (IINR,               (V v) :: evs) -> (cp + 1, V(INR v) :: evs)
 | (PUSH v,                      evs) -> (cp + 1, (V v) :: evs)
 | (LOOKUP x,                    evs) -> (cp + 1, V(search(evs, x)) :: evs)
 | (MK_CLOSURE(loc, fvars),      evs) -> (cp + 1, V(mk_fun(loc, env_of_evs (fvars, evs))) :: evs)
 | (MK_REC (loc, fvars),         evs) -> (cp + 1, V(mk_rec(loc, env_of_evs(fvars, evs))) :: evs)
 | (BIND x,             (V v) :: evs) -> (cp + 1, EV([(x, v)]) :: evs) 
 | (SWAP_POP,      (V v) :: _ :: evs) -> (cp + 1, (V v) :: evs) 
 | (SKIP,                        evs) -> (cp + 1, evs) 
 | (HALT,                        evs) -> (cp, evs) 
 | (TEST (_, Some _),  V(CON (BOOL true)) :: evs) -> (cp + 1, evs) 
 | (TEST (_, Some i), V(CON (BOOL false)) :: evs) -> (i,     evs) 
 | (CASE (_, Some _),              V(INL v)::evs) -> (cp + 1, (V v) :: evs) 
 | (CASE (_, Some i),              V(INR v)::evs) -> (i,      (V v) :: evs) 
 | (GOTO (_, Some i),                        evs) -> (i, evs) 
 | (RETURN,          (V v) :: (RA i) :: _ :: evs) -> (i, (V v) :: evs) 
 | (APPLY,  (V v) :: V(FUN (_, ((_, Some i), env))) :: evs) -> 
        (i, (V v) :: (RA (cp + 1)) :: (EV env) :: evs)
 | _ -> complain ("step : bad state = " ^ (string_of_state (cp, evs)) ^ "\n")

(* COMPILE *) 

let nl inst = NotLabelled inst 

let al lab inst = Labelled(lab, inst) 

let new_label = 
    let i = ref 0 in 
    let get () = let v = !i in (i := (!i) + 1; "L"^ (string_of_int v))
    in get 

let rec comp = function 
  | Unit           -> ([], [nl (PUSH (CON UNIT))]) 
  | Boolean b      -> ([], [nl (PUSH (CON (BOOL b)))])
  | Integer n      -> ([], [nl (PUSH (CON(INT n)))]) 
  | Var x          -> ([], [nl (LOOKUP x)]) 
  | UnaryOp(op, e) -> let (defs, c) = comp e in  
                         (defs, c @ [nl (UNARY op)])
  | Op(e1, op, e2) -> let (defs1, c1) = comp e1 in  
                      let (defs2, c2) = comp e2 in  
                       (defs1 @ defs2, c1 @ c2 @ [nl (OPER op)])
  | Pair(e1, e2)   -> let (defs1, c1) = comp e1 in  
                      let (defs2, c2) = comp e2 in  
                          (defs1 @ defs2, c1 @ c2 @ [nl IPAIR]) 
  | App(e1, e2)    -> let (defs1, c1) = comp e1 in  
                      let (defs2, c2) = comp e2 in  
                          (defs1 @ defs2, c1 @ c2 @ [nl APPLY]) 
  | Fst e          -> let (defs, c) = comp e in (defs, c @ [nl FST])
  | Snd e          -> let (defs, c) = comp e in (defs, c @ [nl SND])
  | Inl e          -> let (defs, c) = comp e in (defs, c @ [nl IINL])
  | Inr e          -> let (defs, c) = comp e in (defs, c @ [nl IINR])
  | If(e1, e2, e3) -> let else_label = new_label () in 
                      let after_else_label = new_label () in 
                      let (defs1, c1) = comp e1 in  
                      let (defs2, c2) = comp e2 in  
                      let (defs3, c3) = comp e3 in  
                      (* simplify away the SKIPs someday at a later phase ... *) 
                         (defs1 @ defs2 @ defs3, 
                          (c1 
   		           @ [nl(TEST(else_label, None))] 
                           @ c2 
		           @ [nl(GOTO (after_else_label, None)); al else_label SKIP] 
                           @ c3 
		           @ [al after_else_label SKIP]))   
 | Case(e1, (x1, e2), (x2, e3)) -> 
                      let inr_label = new_label () in 
                      let after_inr_label = new_label () in 
                      let (defs1, c1) = comp e1 in  
                      let (defs2, c2) = comp e2 in  
                      let (defs3, c3) = comp e3 in  
                      (* simplify away the SKIPs someday at a later phase ... *) 
                         (defs1 @ defs2 @ defs3, 
                          (c1 
   		           @ [nl(CASE(inr_label, None))] 
                           @ ((nl (BIND x1)) :: c2 @ [nl SWAP_POP])
		           @ [nl(GOTO (after_inr_label, None)); al inr_label SKIP] 
                           @ ((nl (BIND x2)) :: c3 @ [nl SWAP_POP])
		           @ [al after_inr_label SKIP]))
 | Lambda(x, e)    -> let (defs, c) = comp e in  
                      let fvars = Free_vars.free_vars ([x], e)  in 
                      let f = new_label () in 
                      let def = (al f (BIND x)) :: c @ [nl SWAP_POP; nl RETURN] in 
                          (def @ defs, [nl (MK_CLOSURE((f, None), fvars))])
 | LetFun(f, (x, e1), e2) -> 
                      let (defs1, c1) = comp e1 in  
                      let (defs2, c2) = comp e2 in  
                      let fvars = Free_vars.free_vars ([x], e1)  in 
                      let def = (al f (BIND x)) :: c1 @ [nl SWAP_POP; nl RETURN] in 
                          (def @ defs1 @ defs2, 
                           (nl (MK_CLOSURE((f, None), fvars))) :: 
                           (nl (BIND f)) :: c2 @ [nl SWAP_POP])
 | LetRecFun(f, (x, e1), e2) -> 
                      let (defs1, c1) = comp e1 in  
                      let (defs2, c2) = comp e2 in  
                      let fvars = Free_vars.free_vars ([f; x], e1)  in 
                      let def = (al f (BIND x)) :: c1 @ [nl SWAP_POP; nl RETURN] in 
                          (def @ defs1 @ defs2, 
                           (nl (MK_REC((f, None), fvars))) 
                           :: (nl (BIND f)) :: c2 @ [nl SWAP_POP])
let compile e = 
    let (defs, c) = comp e in 
    let result = c @              (* body of program *) 
                   [nl SWAP_POP;  (* get rid of initial environment *) 
                   nl HALT]       (* stop the interpreter *) 
                   @ defs in      (* the function definitions *) 
    let _ = if !verbose 
            then print_string ("\ncompiled code = \n" ^ (string_of_code result))
            else () 
    in result 

let rec driver n state = 
  let _ = if !verbose 
          then print_string ("\nstate " ^ (string_of_int n) 
                             ^ " = \n" ^ (string_of_state state) ^ "\n")
          else () 
  in match state with 
     | (cp, evs) -> 
       if HALT = get_instruction (Array.get !installed cp)
       then (match evs with 
             | [V v] -> v 
             | _ -> complain ("driver : bad halted state = " ^ (string_of_state state) ^ "\n"))
       else driver (n + 1) (step state) 


(* put code listing into an array, associate an array index to each label *) 
let code_to_array l = 
   let map_l_instruction f = function 
     | NotLabelled inst -> NotLabelled (f inst) 
     | Labelled (lab, inst) -> Labelled (lab, f inst) 
   in let rec find lab = function 
     | [] -> complain ("find : " ^ lab ^ " is not found")
     | (x, v) :: rest -> if x = lab then v else find lab rest 
    (* insert array index for each label *) 
   in let apply_label_map_to_instruction m = function 
     | GOTO (lab, _) -> GOTO(lab, Some(find lab m))
     | TEST (lab, _) -> TEST(lab, Some(find lab m))
     | CASE (lab, _) -> CASE(lab, Some(find lab m))
     | MK_CLOSURE ((lab, _), fvars) -> MK_CLOSURE((lab, Some(find lab m)), fvars)
     | MK_REC ((lab, _), fvars) -> MK_REC((lab, Some(find lab m)), fvars)
     | inst -> inst 
   (* find array index for each label *) 
   in let listing_to_label_map l = 
       let rec aux carry k = function 
         | [] -> carry 
         | (NotLabelled _) :: rest -> aux carry (k+1) rest 
         | (Labelled (lab, _)) :: rest -> aux ((lab, k) :: carry) (k+1) rest 
       in aux [] 0 l 
    in let l_map = listing_to_label_map l 
    in Array.of_list (List.map (map_l_instruction (apply_label_map_to_instruction l_map)) l) 

let eval(e, env) = 
    let c = compile e in 
    let _ = installed := code_to_array c in 
    (* set the code pointer to 0, install initial environment *) 
      driver 1 (0 , [EV env])

(* env_empty : env *) 
let env_empty = [] 

(* interpret : expr -> value *) 
let interpret e = eval(e, env_empty)

    

      
    
    
