(**************************************
Compiler Construction 2015
Computer Laboratory 
University of Cambridge 
Timothy G. Griffin (tgg22@cam.ac.uk) 
*****************************************) 
(* Interpreter 8. 

   This is similar to the transformation of interp_4.ml to interp_5.ml. 
   However, here we are in a "better" situation in the sense that 
   environments are now shared on the stack and do not have to be 
   explicitly duplicated. 

   Note that the directives of interp_7.ml contain no run-time 
   objects such as values or environments (the exception is a closure). 
   In addition, we have rules in interp_7.ml that "unfold" expressions, such as 

   (E(Pair(e1, e2)) :: ds,   evs) 
    -> ((E e1) :: C_PAIR_FST(e2) :: ds, evs)

   and 

 | (C_PAIR_FST (e) :: ds, evs) -> ((E e) :: C_PAIR :: ds, evs)

 In Interpreter 8, defined below, the idea 
 is to "compile" an expression into "code" *before* evaluating it. 
 So for example, we replace "E e" above with "compile e", where 
 compile returns a list of instructions. To capture the first rule above
 we introduce an instruction for duplicating the environment. So combining 
 the above two rules gives 

 compile(Pair(e1, e2)) =  (compile e1) @ (compile e2) @ [IPAIR] 

 where IPAIR is an instruction (defined below). 

 In addition, several "optimizations" are applied.  For example, the step 
 function of interp_7.ml might produce a sequence like 

    (ARG (e) :: ds, V(FUN (x, body, env) ::evs) 
 -> ((E e) :: (APPLY (x, body, env) :: ds, evs)
 -> ... 
 -> (APPLY (x, body, env) :: ds, es, (V v) :: evs)
 -> ((E body) :: ds, EV(update(env, (x, v))) :: evs)

 The machine defined below will simply leave the FUN construct on 
 the value stack: 

 compile(App(e1, e2)) = (compile e1) @ (compile e2) @ [APPLY] 

 and 

 step(APPLY :: ds,  (V v) :: V(FUN (c, env)) :: evs) = (c @ ds, EV((x, v) :: env):: vs)

 This choice is really forced on us by the fact that code does not 
 contain run-time values.  But how do we make a function? 

 In interp_7.ml: 

 step(E(Lambda(x, e)) :: ds, env :: es, vs) = (ds, es, FUN(x, e, env) ::vs)

 step(E(Lambda(x, e)) :: ds,   evs) = (ds, V(mk_fun(x, e, env_of_evs evs)) :: evs)

 Below we have 

 compile(Lambda(x, e)) = [MK_CLOSURE((BIND x) :: (compile e)); POP_ENV]

 and 

 step(MK_CLOSURE(c) :: ds, env :: es, vs) = (ds,  env ::es, (FUN(c, env)) :: vs)
 
 step((BIND x) :: ds, env :: es, v :: vs) = (ds, (update(env, (x, v))) :: es, vs) 

   
*) 


open Ast 
open Common 

let verbose = ref false 

type value = 
     | CON of Common.constant 
     | PAIR of value * value 
     | INL of value 
     | INR of value 
     | FUN of bool * closure    (* the bool is a flag : true means recursive environment! *) 

and closure = code * env 

and instruction = 
  | PUSH of value 
  | UNARY of Past.unary_oper 
  | OPER of Past.oper 
  | TEST of code * code
  | CASE of code * code
  | IPAIR 
  | FST
  | SND
  | IINL
  | IINR
  | APPLY
  | LOOKUP of Past.var 
  | BIND of Past.var 
  | MK_CLOSURE of code * (Past.var list) 
  | MK_REC of Past.var * code * (Past.var list) 
  | SWAP_POP 

and code = instruction list 

and binding = Past.var * value

and env = binding list

type env_or_value = EV of env | V of value 

type env_value_stack = env_or_value list 

type state = code * env_value_stack 

(* update : (env * binding) -> env *) 
let update(env, (x, v)) = (x, v) :: env 

let mk_fun(c, env) = FUN(false, (c, env)) 
let mk_rec(f, c, env) = FUN(true, (c, (f, FUN(true, (c, []))) :: env))

(* 
      for a recursive function f we want 

      lookup (env, f) = FUN(true, (x, body, env))  
*) 

let lookup_opt (env, x) = 
    let rec aux = function 
      | [] -> None 
      | (y, v) :: rest -> 
          if x = y 
          then Some(match v with 
               | FUN(true, (body, _)) -> FUN(true, (body, (x, FUN(true, (body, []))) :: rest))
               | _ -> v)
          else aux rest  
      in aux env 

let lookup (env, x) = 
    match lookup_opt (env, x) with 
    | None -> complain (x ^ " is not defined!\n")
    | Some v -> v 

let rec search (evs, x) = 
  match evs with 
  | [] -> complain (x ^ " is not defined!\n")
  | (V _) :: rest -> search (rest, x) 
  | (EV env) :: rest -> 
    (match lookup_opt(env, x) with 
    | None -> search (rest, x) 
    | Some v -> v 
    ) 

let env_of_evs(fvars, evs) = 
 let rec inlist x = function 
   | [] -> false 
   | y :: rest -> (x = y) || (inlist x rest) 
 in let rec filter seen = function 
   | [] -> ([], seen) 
   | (x, v) :: rest -> 
      if (inlist x fvars) && (not (inlist x seen))
      then let (env', seen') = filter (x :: seen) rest in 
           ((x, v) :: env', seen') 
      else (filter seen rest)
 in let rec aux seen= function 
  | [] -> []
  | (V _) :: rest -> aux seen rest 
  | (EV env) :: rest -> 
     let (env', seen') = filter seen env in env' @ (aux seen' rest)
 in aux [] evs 

    
let do_unary = function 
  | (Past.NOT, CON (BOOL m)) -> CON(BOOL (not m))
  | (Past.NEG, CON (INT m))  -> CON(INT (-m))
  | (Past.READ, CON(UNIT))   -> CON(INT (readint()))
  | (op, _) -> complain ("malformed unary operator: " ^ (Pp_past.string_of_unary_oper op))

let do_oper = function 
  | (Past.AND, CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m && n))
  | (Past.OR,  CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m || n))
  | (Past.EQB, CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m = n))
  | (Past.LT, CON (INT m), CON (INT n)) -> CON(BOOL (m < n))
  | (Past.EQI, CON (INT m), CON (INT n)) -> CON(BOOL (m = n))
  | (Past.ADD, CON (INT m), CON (INT n)) -> CON(INT (m + n))
  | (Past.SUB, CON (INT m), CON (INT n)) -> CON(INT (m - n))
  | (Past.MUL, CON (INT m), CON (INT n)) -> CON(INT (m * n))
  | (op, _, _)  -> complain ("malformed binary operator: " ^ (Pp_past.string_of_oper op))

let rec string_of_value = function 
     | CON c           -> string_of_constant c
     | PAIR(v1, v2)    -> "(" ^ (string_of_value v1) ^ ", " ^ (string_of_value v2) ^ ")"
     | INL v           -> "inl(" ^ (string_of_value v) ^ ")"
     | INR  v          -> "inr(" ^ (string_of_value v) ^ ")"
     | FUN (true, cl)  -> "FUN(true, " ^ (string_of_closure cl) ^ ")"
     | FUN (false, cl) -> "FUN(false, " ^ (string_of_closure cl) ^ ")"

and string_of_closure (c, env) = 
   "(" ^ (string_of_code c) ^ ", " ^ (string_of_env env) ^ ")"

and string_of_env env = Common.string_of_list ",\n " string_of_binding env 

and string_of_binding (x, v) =    "(" ^ x ^ ", " ^ (string_of_value v) ^ ")"

and string_of_instruction = function 
 | UNARY op     -> "UNARY " ^ (Pp_past.string_of_unary_oper op) 
 | OPER op      -> "OPER " ^ (Pp_past.string_of_oper op) 
 | IPAIR        -> "IPAIR"
 | FST          -> "FST"
 | SND          -> "SND"
 | IINL         -> "IINL"
 | IINR         -> "IINR"
 | PUSH v       -> "PUSH " ^ (string_of_value v) 
 | LOOKUP x     -> "LOOKUP " ^ x
 | TEST(c1, c2) -> "TEST(" ^ (string_of_code c1) ^ ", " ^ (string_of_code c1) ^ ")"
 | CASE(c1, c2) -> "CASE(" ^ (string_of_code c1) ^ ", " ^ (string_of_code c1) ^ ")"
 | APPLY        -> "APPLY"
 | BIND x       -> "BIND " ^ x
 | SWAP_POP     -> "SWAP_POP"
 | MK_CLOSURE (c, fvars) -> "MK_CLOSURE(" ^ (string_of_code c) ^ ", [" ^ (Common.string_of_list ", " (fun x -> x) fvars) ^ "])" 
 | MK_REC(f, c, fvars) -> "MK_REC(" ^ f ^ ", " ^ (string_of_code c) ^ ", [" ^ (Common.string_of_list ", " (fun x -> x) fvars) ^ "])" 

and string_of_code c = Common.string_of_list "\n " string_of_instruction c 

let string_of_env_or_value = function 
  | EV env -> "EV " ^ (string_of_env env)
  | V v -> "V " ^ (string_of_value v)

let string_of_env_value_stack = Common.string_of_list ";\n " string_of_env_or_value 

let string_of_state (c, evs) = "\n" ^ (string_of_code c) ^ "\n" ^ (string_of_env_value_stack evs) ^ "\n" 

let step = function 
 | ((UNARY op) :: ds,            (V v) :: evs) -> (ds, V(do_unary(op, v)) :: evs) 
 | ((OPER op) :: ds,  (V v2) :: (V v1) :: evs) -> (ds, V(do_oper(op, v1, v2)) :: evs)
 | (IPAIR :: ds,      (V v2) :: (V v1) :: evs) -> (ds, V(PAIR(v1, v2)) :: evs)
 | (FST :: ds,          V(PAIR (v, _)) :: evs) -> (ds, (V v) :: evs)
 | (SND :: ds,          V(PAIR (_, v)) :: evs) -> (ds, (V v) :: evs)
 | (IINL :: ds,                  (V v) :: evs) -> (ds, V(INL v) :: evs)
 | (IINR :: ds,                  (V v) :: evs) -> (ds, V(INR v) :: evs)
 | ((PUSH v) :: ds,            evs) -> (ds, (V v) :: evs)
 | ((LOOKUP x) :: ds, evs) -> (ds,  V(search(evs, x)) :: evs)
 | ((TEST(c1, c2)) :: ds, V(CON (BOOL true)) :: evs) -> (c1 @ ds, evs) 
 | ((TEST(c1, c2)) :: ds, V(CON (BOOL false)) :: evs) -> (c2 @ ds, evs) 
 | (CASE (c1,  _) :: ds,  V(INL v)::evs) -> (c1 @ ds, (V v) :: evs) 
 | (CASE ( _, c2) :: ds,  V(INR v)::evs) -> (c2 @ ds, (V v) :: evs) 
 | (APPLY :: ds,  (V v) :: V(FUN (_, (c, env))) :: evs) -> (c @ ds, (V v) :: (EV env) :: evs)
 | (MK_CLOSURE(c, fvars) :: ds,  evs) -> (ds,  V(mk_fun(c, env_of_evs(fvars, evs))) :: evs)
 | (MK_REC(f, c, fvars) :: ds,   evs) -> (ds,  V(mk_rec(f, c, env_of_evs(fvars, evs))) :: evs)
 | ((BIND x) :: ds,     (V v) :: evs) -> (ds, EV([(x, v)]) :: evs) 
 | (SWAP_POP :: ds,     (V v) :: _ :: evs) -> (ds, (V v) :: evs) 
 | state -> complain ("step : bad state = " ^ (string_of_state state) ^ "\n")


let rec compile = function 
 | Unit           -> [PUSH (CON UNIT)] 
 | Integer n      -> [PUSH (CON (INT n))] 
 | Boolean b      -> [PUSH (CON (BOOL b))] 
 | Var x          -> [LOOKUP x] 
 | Lambda(x, e)   -> [MK_CLOSURE((BIND x) :: (compile e) @ [SWAP_POP], 
                                 Free_vars.free_vars ([x], e))]
 | UnaryOp(op, e) -> (compile e) @ [UNARY op]
 | Op(e1, op, e2) -> (compile e1) @ (compile e2) @ [OPER op] 
 | If(e1, e2, e3) -> (compile e1) @ [TEST(compile e2, compile e3)]
 | Pair(e1, e2)   -> (compile e1) @ (compile e2) @ [IPAIR] 
 | App(e1, e2)    -> (compile e1) @ (compile e2) @ [APPLY; SWAP_POP] (* get rid of arg *) 
 | Fst e          -> (compile e) @ [FST] 
 | Snd e          -> (compile e) @ [SND] 
 | Inl e          -> (compile e) @ [IINL] 
 | Inr e          -> (compile e) @ [IINR] 
 | Case(e, (x1, e1), (x2, e2)) -> 
       (compile e)
       @ [CASE((BIND x1) :: (compile e1) @ [SWAP_POP], 
               (BIND x2) :: (compile e2) @ [SWAP_POP])]
 | LetFun(f, (x, body), e)    -> 
     let fvars = Free_vars.free_vars ([x], body)  in 
       (MK_CLOSURE((BIND x) :: (compile body) @ [SWAP_POP], fvars)) :: 
       (BIND f) :: 
       (compile e) @ [SWAP_POP]
 | LetRecFun(f, (x, body), e) -> 
     let fvars = Free_vars.free_vars ([f; x], body)  in 
       (MK_REC(f, (BIND x) :: (compile body) @ [SWAP_POP], fvars)) ::  
       (BIND f) :: 
       (compile e) @ [SWAP_POP]

let rec driver n state = 
  let _ = if !verbose 
          then print_string ("\nstate " ^ (string_of_int n) 
                             ^ " = \n" ^ (string_of_state state) ^ "\n")
          else () 
  in match state with 
     | ([], [V v]) -> v 
     | _ -> driver (n + 1) (step state) 

let eval(e, env) = 
    let c = compile e in 
    let _ = if !verbose 
            then print_string("Compile code =\n" ^ (string_of_list "\n" string_of_instruction c))
            else () 
    in driver 1 (c @ [SWAP_POP], [EV env])


(* env_empty : env *) 
let env_empty = [] 

(* interpret : expr -> value *) 
let interpret e = eval(e, env_empty)

    

      
    
    
