(**************************************
Compiler Construction 2015
Computer Laboratory 
University of Cambridge 
Timothy G. Griffin (tgg22@cam.ac.uk) 
*****************************************) 
(* Interpreter 7. 

  Split stack of Interpreter 6 into *two* stacks.
  The "directive stack" contains bits of 
  the program text, but no run-time values. 
  The "run-time stack" contains values and 
  environments. 

*) 


open Ast 
open Common 

let verbose = ref false 

(* "semenatic" domain *) 

type value = 
     | CON of Common.constant 
     | PAIR of value * value 
     | INL of value 
     | INR of value 
     | FUN of bool * closure    (* the bool is a flag : true means recursive environment! *) 

and closure = Past.var * expr * env 

and directive = 
  | E of expr 
  | C_UNARY of Past.unary_oper 
  | C_OPER of Past.oper 
  | C_OPER_FST of expr * Past.oper 
  | C_IF of expr * expr 
  | C_PAIR 
  | C_PAIR_FST of expr 
  | C_FST
  | C_SND
  | C_INL
  | C_INR
  | C_CASE of Past.var * expr * Past.var * expr 
  | C_APPLY of closure 
  | C_ARG of expr 
  | SWAP_POP 

and directive_stack = directive  list

and binding = Past.var * value

and env = binding list

type env_or_value = EV of env | V of value 

type env_value_stack = env_or_value list 

type state = directive_stack * env_value_stack 

(* update : (env * binding) -> env *) 
let update(env, (x, v)) = (x, v) :: env 

(* When making a closure, only include bindings that 
   are needed. 
*) 

let rec inlist x = function 
  | [] -> false 
  | y :: rest -> (x = y) || (inlist x rest) 

let rec remove x = function 
  | [] -> []
  | y :: rest -> if x = y then rest else y :: (remove x rest) 

let rec filter_env fvars = function 
  | [] -> [] 
  | (x, v) :: rest -> 
       if inlist x fvars 
       (* now remove x, since we filter env's derived from 
          the stack and there may be multiple instances of x, 
          so just pick up the first one *) 
       then (x, v) :: (filter_env (remove x fvars) rest) 
       else (filter_env fvars rest)

let mk_fun(x, body, env) = 
    let fvars = Free_vars.free_vars ([x], body) in 
    let smaller_env = filter_env fvars env in 
      FUN(false, (x, body, smaller_env)) 

let mk_rec_fun(f, x, body, env) = 
    let fvars = Free_vars.free_vars ([f; x], body) in 
    let smaller_env = filter_env fvars env in 
     FUN(true, (x, body, (f, FUN(true, (x, body, []))) :: smaller_env))
(* 
      for a recursive function f we want 

      lookup (env, f) = FUN(true, (x, body, env))  
*) 
let lookup_opt (env, x) = 
    let rec aux = function 
      | [] -> None 
      | (y, v) :: rest -> 
          if x = y 
          then Some(match v with 
               | FUN(true, (y, body, _)) -> FUN(true, (y, body, (x, FUN(true, (y, body, []))) :: rest))
               | _ -> v)
          else aux rest  
      in aux env 

let lookup (env, x) = 
    match lookup_opt (env, x) with 
    | None -> complain (x ^ " is not defined!\n")
    | Some v -> v 

let rec search (evs, x) = 
  match evs with 
  | [] -> complain (x ^ " is not defined!\n")
  | (V _) :: rest -> search (rest, x) 
  | (EV env) :: rest -> 
    (match lookup_opt(env, x) with 
    | None -> search (rest, x) 
    | Some v -> v 
    ) 
    
let rec env_of_evs = function 
  | [] -> []
  | (V _) :: rest -> env_of_evs rest 
  | (EV env) :: rest -> env @ (env_of_evs rest)
    


let do_unary = function 
  | (Past.NOT, CON (BOOL m)) -> CON(BOOL (not m))
  | (Past.NEG, CON (INT m))  -> CON(INT (-m))
  | (Past.READ, CON(UNIT))   -> CON(INT (readint()))
  | (op, _) -> complain ("malformed unary operator: " ^ (Pp_past.string_of_unary_oper op))

let do_oper = function 
  | (Past.AND, CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m && n))
  | (Past.OR,  CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m || n))
  | (Past.EQB, CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m = n))
  | (Past.LT, CON (INT m), CON (INT n)) -> CON(BOOL (m < n))
  | (Past.EQI, CON (INT m), CON (INT n)) -> CON(BOOL (m = n))
  | (Past.ADD, CON (INT m), CON (INT n)) -> CON(INT (m + n))
  | (Past.SUB, CON (INT m), CON (INT n)) -> CON(INT (m - n))
  | (Past.MUL, CON (INT m), CON (INT n)) -> CON(INT (m * n))
  | (op, _, _)  -> complain ("malformed binary operator: " ^ (Pp_past.string_of_oper op))

let rec string_of_value = function 
     | CON c           -> string_of_constant c
     | PAIR(v1, v2)    -> "(" ^ (string_of_value v1) ^ ", " ^ (string_of_value v2) ^ ")"
     | INL v           -> "inl(" ^ (string_of_value v) ^ ")"
     | INR  v          -> "inr(" ^ (string_of_value v) ^ ")"
     | FUN (true, cl)  -> "FUN(true, " ^ (string_of_closure cl) ^ ")"
     | FUN (false, cl) -> "FUN(false, " ^ (string_of_closure cl) ^ ")"

and string_of_closure (x, e, env) = 
   "(" ^ x ^ ", " ^ (Pp_ast.string_of_expr e) ^ ", " ^ (string_of_env env) ^ ")"

and string_of_env env = Common.string_of_list ",\n " string_of_binding env 

and string_of_binding (x, v) =    "(" ^ x ^ ", " ^ (string_of_value v) ^ ")"

let string_of_directive = function 
  | E e            -> "E(" ^ (Pp_ast.string_of_expr e) ^ ")"
  | C_UNARY op     -> "C_UNARY " ^ (Pp_past.string_of_unary_oper op) 
  | C_PAIR         -> "C_PAIR " 
  | C_FST          -> "C_FST"
  | C_SND          -> "C_SND"
  | C_INL          -> "C_INL"
  | C_INR          -> "C_INR"
  | SWAP_POP       -> "SWAP_POP"
  | C_APPLY cl     -> "C_APPLY " ^ (string_of_closure cl) 
  | C_ARG e        -> "C_ARG("  ^ (Pp_ast.string_of_expr e)   ^ ")"
  | C_OPER op      -> "C_OPER(" ^ (Pp_past.string_of_oper op) ^ ")"
  | C_IF (e1, e2)  -> "C_IF(" ^ (Pp_ast.string_of_expr e1) ^ ", " ^ (Pp_ast.string_of_expr e2) ^ ")"
  | C_PAIR_FST e   -> "C_PAIR_FST(" ^ (Pp_ast.string_of_expr e) ^ ")"
  | C_OPER_FST(e, op) ->  "C_OPER_FST(" ^ (Pp_ast.string_of_expr e) ^ ", " ^ (Pp_past.string_of_oper op) ^ ")"
  | C_CASE(x1, e1, x2, e2) -> 
      "C_CASE(" ^ x1 ^ ", " 
                ^ (Pp_ast.string_of_expr e1) ^ ", " 
                ^ x2 ^ ", " 
                ^ (Pp_ast.string_of_expr e2) ^ ")" 

let string_of_env_or_value = function 
  | EV env -> "EV " ^ (string_of_env env)
  | V v -> "V " ^ (string_of_value v)

let string_of_env_value_stack = Common.string_of_list ";\n " string_of_env_or_value 

let string_of_directive_stack = Common.string_of_list ";\n " string_of_directive

let string_of_state (ds, evs) = "\n" ^ (string_of_directive_stack ds) ^ "\n" ^ (string_of_env_value_stack evs) ^ "\n" 


(*      
   R(EVAL(e, env, cnt))  = ((E e) :: ds(cnt), evs(cnt))    where env_of_evs(evs(cnt)) = env 
   R(APPLY(cnt, v))      = (ds(cnt), (V v) :: evs(cnt))
*) 

let step = function 
 (* EVAL -> EVAL *) 
 | (E(UnaryOp(op, e)) :: ds,              evs) -> ((E e)  :: (C_UNARY op) :: ds,          evs)
 | (E(Op(e1, op, e2)) :: ds,              evs) -> ((E e1) :: (C_OPER_FST(e2, op)) :: ds,  evs)
 | (E(If(e1, e2, e3)) :: ds,              evs) -> ((E e1) :: (C_IF(e2, e3)) :: ds,        evs)
 | (E(Pair(e1, e2)) :: ds,                evs) -> ((E e1) :: C_PAIR_FST(e2) :: ds,        evs)
 | (E(App(e1, e2)) :: ds,                 evs) -> ((E e1) :: C_ARG(e2) :: ds,             evs)
 | (E(Fst e) :: ds,                       evs) -> ((E e)  :: C_FST :: ds,                 evs)
 | (E(Snd e) :: ds,                       evs) -> ((E e)  :: C_SND :: ds,                 evs) 
 | (E(Inl e) :: ds,                       evs) -> ((E e)  :: C_INL :: ds,                 evs)
 | (E(Inr e) :: ds,                       evs) -> ((E e)  :: C_INR :: ds,                 evs) 
 | (E(Case(e, (x1, e1), (x2, e2))) :: ds, evs) -> ((E e)  :: C_CASE(x1, e1, x2, e2) :: ds, evs)
 | (E(LetFun(f, (x, body), e)) :: ds,     evs) -> ((E e)  :: SWAP_POP :: ds, (EV[(f, mk_fun(x, body, env_of_evs evs))]) :: evs)
 | (E(LetRecFun(f, (x, body), e)) :: ds,  evs) -> ((E e)  :: SWAP_POP :: ds, (EV[(f, mk_rec_fun(f, x, body, env_of_evs evs))]) :: evs)

 (* EVAL -> APPLY *) 
 | ((E Unit) :: ds,          evs) -> (ds, V(CON UNIT) :: evs) 
 | (E(Integer n) :: ds,      evs) -> (ds, V(CON(INT n)) :: evs)
 | (E(Boolean b) :: ds,      evs) -> (ds, V(CON (BOOL b)) :: evs)
 | (E(Var x) :: ds,          evs) -> (ds, V(search(evs, x)) :: evs)
 | (E(Lambda(x, e)) :: ds,   evs) -> (ds, V(mk_fun(x, e, env_of_evs evs)) :: evs)

 (* APPLY -> APPLY *) 
 | (SWAP_POP :: ds,              v :: _ :: evs) -> (ds, v :: evs) 
 | ((C_UNARY op) :: ds,             (V v)::evs) -> (ds, V(do_unary(op, v)) :: evs) 
 | ((C_OPER op) :: ds, (V v2) :: (V v1) :: evs) -> (ds, V (do_oper(op, v1, v2)) :: evs)
 | (C_PAIR :: ds,      (V v2) :: (V v1) :: evs) -> (ds, V(PAIR(v1, v2)) :: evs)
 | (C_FST :: ds,         V(PAIR (v, _)) :: evs) -> (ds, (V v) :: evs)
 | (C_SND :: ds,         V(PAIR (_, v)) :: evs) -> (ds, (V v) :: evs)
 | (C_INL :: ds,                  (V v) :: evs) -> (ds, V(INL v) :: evs)
 | (C_INR :: ds,                  (V v) :: evs) -> (ds, V(INR v) :: evs)
 (* APPLY -> EVAL *) 
 | ((C_PAIR_FST e) :: ds,                       evs) -> ((E e) :: C_PAIR :: ds, evs)
 | (C_OPER_FST (e, op) :: ds,                   evs) -> ((E e) :: (C_OPER op) :: ds, evs)
 | ((C_APPLY (x, e, env)) :: ds,        (V v):: evs) -> ((E e) :: SWAP_POP :: ds, (EV ((x, v) :: env)) :: evs)
 | (C_ARG (e) :: ds,           V(FUN(_, cl)) :: evs) -> ((E e) :: (C_APPLY cl) :: ds, evs)
 | (C_IF (e2, e3) :: ds,  V(CON (BOOL true)) :: evs) -> ((E e2) :: ds, evs) 
 | (C_IF (e2, e3) :: ds, V(CON (BOOL false)) :: evs) -> ((E e3) :: ds, evs) 
 | (C_CASE (x1, e1,  _,  _) :: ds,  V(INL v) :: evs) -> ((E e1) :: SWAP_POP :: ds, (EV [(x1, v)]) :: evs)
 | (C_CASE ( _,  _, x2, e2) :: ds,  V(INR v) :: evs) -> ((E e2) :: SWAP_POP :: ds, (EV [(x2, v)]) :: evs)
 | state -> complain "step : bad state" 


(*
      (E(App(Lambda(x, body), e)) :: ds, evs) 
   -> ((E (Lambda(x, body)) :: C_ARG(e) :: ds,  evs)
   -> (C_ARG(e) :: ds, VALUE(FUN(x, body, NEW_ENV)) :: evs)
   -> ((E e) :: (C_APPLY (x, body, NEW_ENV)) :: ds, evs)
   -> ... 
   -> ((C_APPLY (x, body, NEW_ENV)) :: ds,  (VALUE v) :: evs)
   -> ((E body) :: SWAP_POP :: ds, (ENV (x, v) :: NEW_ENV) :: evs)
   -> ... 
   -> (SWAP_POP :: ds, result :: (ENV ((x, v) :: NEW_ENV)) :: evs)
   -> (ds, result :: evs)
*) 

let rec driver n state = 
  let _ = if !verbose 
          then print_string ("\nstate " ^ (string_of_int n) ^ " = \n" ^ (string_of_state state) ^ "\n")
          else () 
  in match state with 
     | ([], [V v]) -> v 
     | _ -> driver (n + 1) (step state) 

let eval(e, env) = driver 1 ([E e; SWAP_POP], [EV env])

(* env_empty : env *) 
let env_empty = [] 

(* interpret : expr -> value *) 
let interpret e = eval(e, env_empty)

    

      
    
    
