(**************************************
Compiler Construction 2015
Computer Laboratory 
University of Cambridge 
Timothy G. Griffin (tgg22@cam.ac.uk) 
*****************************************) 
(* Interpreter 6. 

   We went down the wrong path from interp_3.ml to interp_4.ml! 

   Our taget is a JARGON-like VM, where environments are implemented 
   on the value stack!  Once these are split into two distinct 
   stacks (as is done in interp_3.ml --> interp_4.ml), then it is very 
   hard to put them back together!  

   Here, we will transform interp_3.ml to interp_6.ml (this file), 
   with only this step: defunctionalizing environments. 

  In the process, we "optimize" closures (x, body, env) 
  so that the (defunctionalised) environment env only contains 
  the bindings needed for the body of the closure (that is, the free vars 
  of body non including x). To do this the file free_vars.ml has been added. 

  Environments (var -> expr) are defunctionalised in the way suggested
  by Problem_Set_1.txt. 


  type binding = var * value

  type env = binding list

   The function lookup : (env * var) - value 
   is the "apply function" for the defunctionalised environments. 

*) 


open Ast 
open Common 

let verbose = ref false 

(* "semenatic" domain *) 

type value = 
     | CON of Common.constant 
     | PAIR of value * value 
     | INL of value 
     | INR of value 
     | FUN of bool * closure    (* the bool is a flag : true means recursive environment! *) 

and closure = Past.var * expr * env 

and continuation_item = 
  | C_UNARY of Past.unary_oper 
  | C_OPER of Past.oper * value 
  | C_OPER_FST of expr * env * Past.oper 
  | C_IF of expr * expr * env 
  | C_PAIR of value 
  | C_PAIR_FST of expr * env 
  | C_FST
  | C_SND
  | C_INL
  | C_INR
  | C_CASE of Past.var * expr * Past.var * expr * env 
  | C_APPLY of closure 
  | C_ARG of expr * env 

and continuation = continuation_item  list

and binding = Past.var * value

and env = binding list


(* update : (env * binding) -> env *) 
let update(env, (x, v)) = (x, v) :: env 

(* When making a closure, only include bindings that 
   are needed. 
*) 

let rec inlist x = function 
  | [] -> false 
  | y :: rest -> (x = y) || (inlist x rest) 

let rec filter_env fvars = function 
  | [] -> [] 
  | (x, v) :: rest -> if inlist x fvars then (x, v) :: (filter_env fvars rest) else (filter_env fvars rest)

let mk_fun(x, body, env) = 
    let fvars = Free_vars.free_vars ([x], body) in 
    let smaller_env = filter_env fvars env in 
      FUN(false, (x, body, smaller_env)) 

let mk_rec_fun(f, x, body, env) = 
    let fvars = Free_vars.free_vars ([f; x], body) in 
    let smaller_env = filter_env fvars env in 
       FUN(true, (x, body, (f, FUN(true, (x, body, []))) ::  smaller_env))  
(* 
      for a recursive function f we want 

      lookup (env, f) = FUN(true, (x, body, env))  
*) 
let lookup (env, x) = 
    let rec aux = function 
      | [] -> complain (x ^ " is not defined!\n")
      | (y, v) :: rest -> 
          if x = y 
          then match v with 
             | FUN(true, (y, body, _)) -> FUN(true, (y, body, (x, FUN(true, (y, body, []))) :: rest))
             | _ -> v 
          else aux rest  
      in aux env 


let do_unary = function 
  | (Past.NOT, CON (BOOL m)) -> CON(BOOL (not m))
  | (Past.NEG, CON (INT m))  -> CON(INT (-m))
  | (Past.READ, CON(UNIT))   -> CON(INT (readint()))
  | (op, _) -> complain ("malformed unary operator: " ^ (Pp_past.string_of_unary_oper op))

let do_oper = function 
  | (Past.AND, CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m && n))
  | (Past.OR,  CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m || n))
  | (Past.EQB, CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m = n))
  | (Past.LT, CON (INT m), CON (INT n)) -> CON(BOOL (m < n))
  | (Past.EQI, CON (INT m), CON (INT n)) -> CON(BOOL (m = n))
  | (Past.ADD, CON (INT m), CON (INT n)) -> CON(INT (m + n))
  | (Past.SUB, CON (INT m), CON (INT n)) -> CON(INT (m - n))
  | (Past.MUL, CON (INT m), CON (INT n)) -> CON(INT (m * n))
  | (op, _, _)  -> complain ("malformed binary operator: " ^ (Pp_past.string_of_oper op))

type state = 
   | EVAL of expr * env * continuation 
   | APPLY of continuation * value 

let rec string_of_value = function 
     | CON c           -> string_of_constant c
     | PAIR(v1, v2)    -> "(" ^ (string_of_value v1) ^ ", " ^ (string_of_value v2) ^ ")"
     | INL v           -> "inl(" ^ (string_of_value v) ^ ")"
     | INR  v          -> "inr(" ^ (string_of_value v) ^ ")"
     | FUN (true, cl)  -> "FUN(true, " ^ (string_of_closure cl) ^ ")"
     | FUN (false, cl) -> "FUN(false, " ^ (string_of_closure cl) ^ ")"

and string_of_closure (x, e, env) = 
   "(" ^ x ^ ", " ^ (Pp_ast.string_of_expr e) ^ ", " ^ (string_of_env env) ^ ")"

and string_of_env env = Common.string_of_list ",\n " string_of_binding env 

and string_of_binding (x, v) =    "(" ^ x ^ ", " ^ (string_of_value v) ^ ")"

let string_of_continuation_item = function 
  | C_UNARY op     -> "C_UNARY " ^ (Pp_past.string_of_unary_oper op) 
  | C_PAIR v       -> "C_PAIR " ^ (string_of_value v)
  | C_FST          -> "C_FST"
  | C_SND          -> "C_SND"
  | C_INL          -> "C_INL"
  | C_INR          -> "C_INR"
  | C_APPLY cl     -> "C_APPLY " ^ (string_of_closure cl) 
  | C_ARG (e, env) -> "C_ARG("  ^ (Pp_ast.string_of_expr e)   ^ ", " ^ (string_of_env env) ^ ")"
  | C_OPER (op, v) -> "C_OPER(" ^ (Pp_past.string_of_oper op) ^ ", " ^ (string_of_value v) ^ ")"
  | C_CASE(x1, e1, x2, e2, env) -> 
      "C_CASE(" ^ x1 ^ ", " 
                ^ (Pp_ast.string_of_expr e1) ^ ", " 
                ^ x2 ^ ", " 
                ^ (Pp_ast.string_of_expr e2) ^ ", " 
                ^ (string_of_env env) ^ ")"
  | C_PAIR_FST (e, env)    -> 
      "C_PAIR_FST(" ^ (Pp_ast.string_of_expr e) ^ ", " ^ (string_of_env env) ^ ")"
  | C_OPER_FST(e, env, op) ->  
      "C_OPER_FST(" ^ (Pp_ast.string_of_expr e) ^ ", " ^ (string_of_env env) ^ ", " ^ (Pp_past.string_of_oper op) ^ ")"
  | C_IF (e1, e2, env) -> 
      "C_IF(" ^ (Pp_ast.string_of_expr e1) ^ ", " ^ (Pp_ast.string_of_expr e2) ^ ", " ^ (string_of_env env) ^ ")"

let string_of_continuation = Common.string_of_list ";\n " string_of_continuation_item

let string_of_state = function 
   | EVAL(e, env, cnt) -> 
      "EVAL(" ^ (Pp_ast.string_of_expr e) ^ ", " 
              ^ (string_of_env env) ^ ", " 
              ^ (string_of_continuation cnt) ^ ")" 
   | APPLY(cnt, v)     -> 
      "APPLY(" ^ (string_of_continuation cnt) ^ ", " 
               ^ (string_of_value v) ^ ")"

(* Examples of evaluation 

    EVAL(App(Lambda(x, body), e), env, k) 
 -> EVAL(Lambda(x, body), env, C_ARG(e, env) :: k)
 -> APPLY(C_ARG(e, env) :: k, FUN(false, (x, body, env)))
 -> EVAL(e, env, (C_APPLY (x, body, env)) :: k)
 -> ... 
 -> ... 
 -> APPLY((C_APPLY (x, body, env)) :: k, v) 
 -> EVAL(body, update(env, (x, v)), k)
 -> ... 
 -> APPLY(k, result) 


     EVAL(Pair(e1, e2), env,   k) 
  -> EVAL(e1, env, C_PAIR_FST(e2, env) :: k)
  -> ... 
  -> APPLY(C_PAIR_FST(e2, env) :: k, v1) 
  -> EVAL(e2, env, (C_PAIR v1) :: k)
  -> ... 
  -> APPLY((C_PAIR v1) :: k, v2)
  -> APPLY(k, PAIR(v1, v2))
  -> ... 

*) 
 
let step = function 
 (* EVAL --> EVAL *) 
 | EVAL(UnaryOp(op, e),              env, k) -> EVAL(e,  env, (C_UNARY op) :: k)
 | EVAL(Op(e1, op, e2),              env, k) -> EVAL(e1, env, C_OPER_FST(e2, env, op) :: k)
 | EVAL(If(e1, e2, e3),              env, k) -> EVAL(e1, env, C_IF(e2, e3, env) :: k)
 | EVAL(Pair(e1, e2),                env, k) -> EVAL(e1, env, C_PAIR_FST(e2, env) :: k)
 | EVAL(Fst e,                       env, k) -> EVAL(e,  env, C_FST :: k)
 | EVAL(Snd e,                       env, k) -> EVAL(e,  env, C_SND :: k) 
 | EVAL(Inl e,                       env, k) -> EVAL(e,  env, C_INL :: k) 
 | EVAL(Inr e,                       env, k) -> EVAL(e,  env, C_INR :: k) 
 | EVAL(Case(e, (x1, e1), (x2, e2)), env, k) -> EVAL(e,  env, C_CASE(x1, e1, x2, e2, env) :: k)
 | EVAL(App(e1, e2),                 env, k) -> EVAL(e1, env, C_ARG(e2, env) :: k)
 | EVAL(LetFun(f, (x, body), e),     env, k) -> EVAL(e, update(env, (f, mk_fun(x, body, env))) , k) 
 | EVAL(LetRecFun(f, (x, body), e),  env, k) -> EVAL(e, update(env, (f, mk_rec_fun(f, x, body, env))) , k) 
 (* EVAL --> APPLY *) 
 | EVAL(Unit,              _, k) -> APPLY(k, CON UNIT) 
 | EVAL(Var x,           env, k) -> APPLY(k, lookup (env, x))
 | EVAL(Integer n,         _, k) -> APPLY(k, CON(INT n))
 | EVAL(Boolean b,         _, k) -> APPLY(k, CON (BOOL b)) 
 | EVAL(Lambda(x, body), env, k) -> APPLY(k, mk_fun(x, body, env))
 (* APPLY --> APPLY *) 
 | APPLY((C_UNARY op) :: k,    v) -> APPLY(k ,(do_unary(op, v)))
 | APPLY(C_OPER(op, v1) :: k, v2) -> APPLY(k, do_oper(op, v1, v2))
 | APPLY((C_PAIR v1) :: k,    v2) -> APPLY(k, PAIR(v1, v2))
 | APPLY(C_FST :: k,  PAIR(v, _)) -> APPLY(k, v)
 | APPLY(C_SND :: k,  PAIR(_, v)) -> APPLY(k, v)
 | APPLY(C_INL :: k,           v) -> APPLY(k, (INL v))
 | APPLY(C_INR :: k,           v) -> APPLY(k, (INR v))
 (* APPLY --> EVAL *) 
 | APPLY(C_OPER_FST (e2, env, op) :: k,         v1)  -> EVAL(e2, env, C_OPER (op, v1) :: k)
 | APPLY((C_APPLY (x, body, env)) :: k,         v2)  -> EVAL(body, update(env, (x, v2)), k)
 | APPLY(C_ARG (e2, env) :: k,          FUN(_, cl))  -> EVAL(e2, env, (C_APPLY cl) :: k)
 | APPLY(C_PAIR_FST (e2, env) :: k,             v1)  -> EVAL(e2, env, (C_PAIR v1) :: k)
 | APPLY(C_CASE (x1, e1, x2, e2, env) :: k,  INL v)  -> EVAL(e1, update(env, (x1, v)), k) 
 | APPLY(C_CASE (x1, e1, x2, e2, env) :: k,  INR v)  -> EVAL(e2, update(env, (x2, v)), k)
 | APPLY(C_IF (e2, e3, env) :: k,   CON(BOOL true))  -> EVAL(e2, env, k)
 | APPLY(C_IF (e2, e3, env) :: k,  CON(BOOL false))  -> EVAL(e3, env, k)
 | state -> complain ("step : malformed state = " ^ (string_of_state state) ^ "\n")

let rec driver n state = 
  let _ = if !verbose 
          then print_string ("\nstate " ^ (string_of_int n) ^ " = \n" ^ (string_of_state state) ^ "\n")
          else () 
  in match state with 
     | APPLY([], v) -> v 
     | _ -> driver (n + 1) (step state) 

let eval(e, env) = driver 1 (EVAL(e, env, []))

(* env_empty : env *) 
let env_empty = [] 

(* interpret : expr -> value *) 
let interpret e = eval(e, env_empty)

    

      
    
    
