(**************************************
Compiler Construction 2015
Computer Laboratory 
University of Cambridge 
Timothy G. Griffin (tgg22@cam.ac.uk) 
*****************************************) 
(* 
   This is a CPS version of Interpeter 0. 
*) 

open Ast 
open Common 

let verbose = ref false 

type value = 
     | CON of Common.constant 
     | PAIR of value * value 
     | INL of value 
     | INR of value 
     | FUN of ((continuation * value) -> value)

and continuation = value -> value 

type env = Past.var -> value 

type binding = Past.var * value

type bindings = binding list

(* auxiliary functions *) 

(* constant_of_value :value -> costant *) 
let constant_of_value = function 
    | CON v -> v 
    | _ -> complain "constant_of_value : expecting a constant!"

let bool_of_value v = bool_of_constant(constant_of_value v)

(* update : (env * binding) -> env *) 
let update(env, (x, v)) = fun y -> if x = y then v else env y

let do_unary = function 
  | (Past.NOT, CON (BOOL m)) -> CON(BOOL (not m))
  | (Past.NEG, CON (INT m))  -> CON(INT (-m))
  | (Past.READ, CON(UNIT))   -> CON(INT (readint()))
  | (op, _) -> complain ("malformed unary operator: " ^ (Pp_past.string_of_unary_oper op))

let do_oper = function 
  | (Past.AND, CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m && n))
  | (Past.OR,  CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m || n))
  | (Past.EQB, CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m = n))
  | (Past.LT, CON (INT m), CON (INT n)) -> CON(BOOL (m < n))
  | (Past.EQI, CON (INT m), CON (INT n)) -> CON(BOOL (m = n))
  | (Past.ADD, CON (INT m), CON (INT n)) -> CON(INT (m + n))
  | (Past.SUB, CON (INT m), CON (INT n)) -> CON(INT (m - n))
  | (Past.MUL, CON (INT m), CON (INT n)) -> CON(INT (m * n))
  | (op, _, _)  -> complain ("malformed binary operator: " ^ (Pp_past.string_of_oper op))

let rec eval_cps (e, env, k) = 
    match e with 
    | Unit             -> k (CON UNIT) 
    | Var x            -> k (env x)
    | Integer n        -> k (CON(INT n))
    | Boolean b        -> k( CON (BOOL b)) 
    | UnaryOp(op, e)   -> 
         eval_cps(e, env, fun v -> k (do_unary(op, v)))
    | Op(e1, op, e2)   -> 
         eval_cps(e1, env, fun v1 -> eval_cps(e2, env, fun v2 -> k (do_oper(op, v1, v2))))
    | If(e1, e2, e3)   -> 
         eval_cps(e1, env, fun v -> if bool_of_value v then eval_cps(e2, env, k) else eval_cps(e3, env, k))
    | Pair(e1, e2)     -> 
         eval_cps(e1, env, fun v1 -> eval_cps(e2, env, fun v2 -> k (PAIR(v1, v2))))
    | Fst e            -> 
         eval_cps(e, env, 
                 fun v -> match v with 
                         | PAIR (v1, _) -> k v1 
                         | v -> complain "eval_cps : run-time error.  Expecting a pair!")
    | Snd e            -> 
         eval_cps(e, env, 
                  fun v -> match v with 
                           | PAIR (_, v2) -> k v2 
                           | v -> complain "eval_cps : run-time error.  Expecting a pair!")
    | Inl e            -> eval_cps(e, env, fun v -> k (INL v))
    | Inr e            -> eval_cps(e, env, fun v -> k (INR v))
    | Case(e, (x1, e1), (x2, e2)) -> 
       eval_cps(e, env, fun v -> 
         (match v with 
         | INL v -> eval_cps(e1, update(env, (x1, v)), k) 
         | INR v -> eval_cps(e2, update(env, (x2, v)), k)
         | v -> complain "eval_cps : run-time error.  Expecting inl or inr!"
         ))
    | Lambda(x, e)     -> k (FUN (fun (c, v) -> eval_cps(e, update(env, (x, v)), c)))
    | App(e1, e2)      -> 
        eval_cps(e1, env, fun v1 -> 
                  match v1 with 
                  | FUN f -> eval_cps(e2, env, fun v2 -> f(k, v2)) 
                  | v -> complain "eval_cps : run-time error.  Expecting a function!")
    | LetFun(f, (x, body), e) -> 
       let new_env = update(env, (f, FUN (fun (c, v) -> eval_cps(body, update(env, (x, v)), c))))
       in eval_cps(e, new_env, k) 
    | LetRecFun(f, (x, body), e) -> 
       let rec new_env g = (* a recursive environment! *) 
           if g = f then FUN (fun (c, v) -> eval_cps(body, update(new_env, (x, v)), c)) else env g
       in eval_cps(e, new_env, k) 

let eval(e, env) = eval_cps(e, env, fun x -> x) 

(* env_empty : env *) 
let env_empty = fun x -> complain (x ^ " is not defined!\n")

(* interpret : expr -> value *) 
let interpret e = eval(e, env_empty)

let rec string_of_value = function 
     | CON c -> string_of_constant c
     | PAIR(v1, v2) -> "(" ^ (string_of_value v1) ^ ", " ^ (string_of_value v2) ^ ")"
     | INL v -> "inl(" ^ (string_of_value v) ^ ")"
     | INR  v -> "inr(" ^ (string_of_value v) ^ ")"
     | FUN _ -> "FUNCTION( ... )" 
    

      
    
    
