(**************************************
Compiler Construction 2015
Computer Laboratory 
University of Cambridge 
Timothy G. Griffin (tgg22@cam.ac.uk) 
*****************************************) 

(*  Interpreter 0 for Slang.2 

    This is a "definitional" interpreter for  for Slang.2 (the defined language) 
    using high-level constructs of Ocaml (the defining language). 
    For examples, Slang.2 functions are represented as Ocaml functions 
    of type 
           value -> value
           
    Slang conditionals are translated to Ocaml conditionals, etc. 
    The most interesting (and tricky) case is the "let rec" construct of 
    Slang --- this is translated using the "lec rec" construct of Ocaml.
    Not with the defined function itself, but with the definition of 
    a recursive environment! (Because when a recursive function 
    calls itself, it must find its own definition in the environment...) 

    Note that some of the functions can fail.  However, 
    if the input expressin has passed static analysis, then such "run time" 
    errors should never happen! (Can you prove that?) 
*) 

open Ast 
open Common 

let verbose = ref false 

type value = 
     | CON of Common.constant 
     | PAIR of value * value 
     | INL of value 
     | INR of value 
     | FUN of (value -> value)


type env = Past.var -> value 

type binding = Past.var * value

type bindings = binding list

(* auxiliary functions *) 

(* constant_of_value :value -> costant *) 
let constant_of_value = function 
    | CON v -> v 
    | _ -> complain "constant_of_value : expecting a constant!"

let bool_of_value v = bool_of_constant(constant_of_value v)

(* update : (env * binding) -> env *) 
let update(env, (x, v)) = fun y -> if x = y then v else env y

let do_unary = function 
  | (Past.NOT, CON (BOOL m)) -> CON(BOOL (not m))
  | (Past.NEG, CON (INT m))  -> CON(INT (-m))
  | (Past.READ, CON(UNIT))   -> CON(INT (readint()))
  | (op, _) -> complain ("malformed unary operator: " ^ (Pp_past.string_of_unary_oper op))

let do_oper = function 
  | (Past.AND, CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m && n))
  | (Past.OR,  CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m || n))
  | (Past.EQB, CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m = n))
  | (Past.LT, CON (INT m), CON (INT n)) -> CON(BOOL (m < n))
  | (Past.EQI, CON (INT m), CON (INT n)) -> CON(BOOL (m = n))
  | (Past.ADD, CON (INT m), CON (INT n)) -> CON(INT (m + n))
  | (Past.SUB, CON (INT m), CON (INT n)) -> CON(INT (m - n))
  | (Past.MUL, CON (INT m), CON (INT n)) -> CON(INT (m * n))
  | (op, _, _)  -> complain ("malformed binary operator: " ^ (Pp_past.string_of_oper op))

(*
    eval = (expr * env) -> value
         = (expr * (var -> value)) -> value
*) 
let rec eval (e, env) = 
    match e with 
    | Unit             -> CON UNIT 
    | Var x            -> env x
    | Integer n        -> CON(INT n)
    | Boolean b        -> CON (BOOL b)
    | UnaryOp(op, e)   -> do_unary(op, eval(e, env))
    | Op(e1, op, e2)   -> do_oper(op, eval(e1, env), eval(e2, env))
    | If(e1, e2, e3)   -> if bool_of_value(eval(e1, env)) then eval(e2, env) else eval(e3, env)
    | Pair(e1, e2)     -> PAIR(eval(e1, env), eval(e2, env))
    | Fst e            -> (match eval(e, env) with 
                           | PAIR (v1, _) -> v1 
                           | v -> complain "eval : run-time error.  Expecting a pair!"
                          )
    | Snd e            -> (match eval(e, env) with 
                           | PAIR (_, v2) -> v2 
                           | v -> complain "eval : run-time error.  Expecting a pair!"
                          )
    | Inl e            -> INL (eval(e, env))
    | Inr e            -> INR (eval(e, env))
    | Case(e, (x1, e1), (x2, e2)) -> 
       (match (eval(e, env)) with 
       | INL v -> eval(e1, update(env, (x1, v))) 
       | INR v -> eval(e2, update(env, (x2, v)))
       | v -> complain "eval : run-time error.  Expecting inl or inr!"
       )
    | Lambda(x, e)     -> FUN (fun v -> eval(e, update(env, (x, v))))
    | App(e1, e2)      -> (match eval(e1, env) with 
                           | FUN f -> f (eval(e2, env))
                           | v -> complain "eval : run-time error.  Expecting a function!"
                          )
    | LetFun(f, (x, body), e) -> 
       let new_env = update(env, (f, FUN (fun v -> eval(body, update(env, (x, v))))))
       in eval(e, new_env) 
    | LetRecFun(f, (x, body), e) -> 
       let rec new_env g = (* a recursive environment! *) 
           if g = f then FUN (fun v -> eval(body, update(new_env, (x, v)))) else env g
       in eval(e, new_env) 

(* env_empty : env *) 
let env_empty = fun x -> complain (x ^ " is not defined!\n")

(* interpret : expr -> value *) 
let interpret e = eval(e, env_empty)

let rec string_of_value = function 
     | CON c -> string_of_constant c
     | PAIR(v1, v2) -> "(" ^ (string_of_value v1) ^ ", " ^ (string_of_value v2) ^ ")"
     | INL v -> "inl(" ^ (string_of_value v) ^ ")"
     | INR  v -> "inr(" ^ (string_of_value v) ^ ")"
     | FUN _ -> "FUNCTION( ... )" 
    

      
    
    
