
open Ast 

(*  If e' = convert e
    then every bound variable in e' is unique. 
    This is done n a lazy way, so that if 
    e has no two bound vars with the same name, 
    then e' = e. Also, if name "f" is changed, 
    it will be changed to something of the form 
    "_f" or "__f" or "___f",  and so on. 
    This appraoch is taken in order to make it 
    easier for users to see their input programs   
    move through multiple interpreter/compiler phases. 
*) 

let seen = ref [] 

let rec inlist x = function 
  | [] ->  false
  | y :: rest -> if x = y then true else (inlist x rest)

let rec get_new_var f = 
    let g = "_" ^ f
    in  if inlist g (!seen) 
        then  get_new_var g 
        else  (seen := g :: (!seen); g) 

let rec lookup x = function 
  | [] ->  x
  | (y, z) :: rest -> if x = y then z else (lookup x rest)

let check_bound_var env f = 
    if inlist f (!seen) 
    then let g = get_new_var f in ((f, g) :: env, g) 
    else (seen := f :: (!seen); (env, f))

let rec aux env e = 
    match e with 
    | Unit           -> e
    | Integer _      -> e  
    | Boolean _      -> e 
    | Var x          -> Var (lookup x env)
    | UnaryOp(op, e) -> UnaryOp(op, aux env e)
    | Op(e1, op, e2) -> Op(aux env e1, op, aux env e2) 
    | If(e1, e2, e3) -> If(aux env e1, aux env e2,  aux env e3)
    | Pair(e1, e2)   -> Pair(aux env e1, aux env e2) 
    | Fst e          -> Fst(aux env e) 
    | Snd e          -> Snd(aux env e) 
    | Inl e          -> Inl(aux env e) 
    | Inr e          -> Inr(aux env e) 
    | Case(e, (x1, e1), (x2, e2)) -> 
      let (env1, y1) = check_bound_var env x1 in 
      let (env2, y2) = check_bound_var env x2 in 
          Case(aux env e, (y1, aux env1 e1), (y2, aux env2 e2))
    | Lambda(x, e) -> let (env1, y) = check_bound_var env x in Lambda(y, aux env1 e)
    | App(e1, e2)    -> App(aux env e1, aux env e2) 
    | LetFun(f, (x, e1), e2) -> 
      let (env1, g) = check_bound_var env f in 
      let (env2, y) = check_bound_var env x in 
         LetFun(g, (y,  aux env2 e1), aux env1 e2)
    | LetRecFun(f, (x, e1), e2) -> 
      let (env1, g) = check_bound_var env f in 
      let (env2, y) = check_bound_var env1 x in 
         LetRecFun(g, (y,  aux env2 e1), aux env1 e2)

let convert e = 
    let _ = seen := [] (* start fresh with each new e *) 
    in aux [] e 
