open Past 

exception Error of string 

let complain s = raise (Error s) 

let  get_loc = function 
    | Unit loc                      -> loc 
    | What loc                      -> loc 
    | Var (loc, _)                  -> loc 
    | Integer (loc, _)              -> loc 
    | Boolean (loc, _)              -> loc 
    | UnaryOp(loc, _, _)            -> loc 
    | Op(loc, _, _, _)              -> loc 
    | If(loc, _, _, _)              -> loc 
    | Pair(loc, _, _)               -> loc 
    | Fst(loc, _)                   -> loc 
    | Snd(loc, _)                   -> loc 
    | Inr(loc, _)                   -> loc 
    | Inl(loc, _)                   -> loc 
    | Case(loc, _, _, _)            -> loc 
    | Lambda(loc, _)                -> loc 
    | App(loc, _, _)                -> loc 
    | Let(loc, _, _)                -> loc 
    | LetFun(loc, _, _, _, _)       -> loc 
    | LetRecFun(loc, _, _, _, _)    -> loc 

let string_of_loc loc = 
    "line " ^ (string_of_int (loc.Lexing.pos_lnum)) ^ ", " ^ 
    "position " ^ (string_of_int ((loc.Lexing.pos_cnum - loc.Lexing.pos_bol) + 1))

let rec find loc x = function 
  | [] -> complain (x ^ " is not defined at " ^ (string_of_loc loc)) 
  | (y, v) :: rest -> if x = y then v else find loc x rest


let rec match_types = function 
  | (TEwildcard, _) -> true 
  | (TEunit, TEunit) -> true
  | (TEint, TEint) -> true
  | (TEbool, TEbool) -> true
  | (TEarrow(t1, t2), TEarrow(t3, t4)) -> (match_types(t1, t3)) && (match_types(t2, t4)) 
  | (TEproduct(t1, t2), TEproduct(t3, t4)) -> (match_types(t1, t3)) && (match_types(t2, t4)) 
  | (TEunion(t1, t2), TEunion(t3, t4)) -> (match_types(t1, t3)) && (match_types(t2, t4)) 
  | (_, _) -> false 


let make_if loc (e1, t1) (e2, t2) (e3, t3) = 
     match t1 with 
     | TEbool -> 
          if t2 = t3 
          then (If(loc, e1, e2, e3), t2) 
          else complain ("conditional at " 
                         ^ (string_of_loc loc) 
                         ^ " has then-branch of type " 
			 ^ (Pp_past.string_of_type t2) 
                         ^ " and else -branch of type " 
			 ^ (Pp_past.string_of_type t3) 
			)
      | ty -> complain ("condition at " 
			^ (string_of_loc loc)
			^ " is of type " 
	                ^ (Pp_past.string_of_type t1) 
			^ " --- expecting boolean!")

(* we want this to work for 

   (fun (d : int + int) -> case d of inl x -> x + 2 | inr y -> 0 end) (inl 99) 

*) 

let make_app loc (e1, t1) (e2, t2) = 
    match t1 with 
    | TEarrow(t3, t4) -> 
         if match_types(t2, t3) 
         then (App(loc, e1, e2), t4)
         else complain "(1)"
    | _ -> complain ("expression at " ^ (string_of_loc loc) ^ " is not a function!")

let make_pair loc (e1, t1) (e2, t2) = (Pair(loc, e1, e2), TEproduct(t1, t2))


(* we want this to work for 

let add(p : (int * (int * (int * int)))) : int =
    (fst(p) + (fst(snd(p)) + fst(snd(snd(p)))))  
in add (1, (88, (12, 10000))) end
*) 

let make_fst loc = function 
  | (e, TEproduct(t, _)) -> (Fst(loc, e), t) 
  | (e, TEwildcard) -> (Fst(loc, e), TEwildcard) 
  | (e, t) -> complain "(2)" 

let make_snd loc = function 
  | (e, TEproduct(_, t)) -> (Snd(loc, e), t) 
  | (e, TEwildcard) -> (Snd(loc, e), TEwildcard) 
  | (e, t) -> complain "(3)" 

let make_inl loc t2 (e, t1)          = (Inl(loc, e), TEunion(t1, t2))

let make_inr loc t1 (e, t2)          = (Inr(loc, e), TEunion(t1, t2))

let make_lambda loc x t1 (e, t2)     = (Lambda(loc, (x, t1, e)), TEarrow(t1, t2))

let make_unary loc op t (e, _)       = (UnaryOp(loc, op, e), t)

let make_op loc op t (e1, _) (e2, _) = (Op(loc, e1, op, e2), t)

let make_let loc bl (e, t)           = (Let(loc, bl, e), t)

let make_letfun loc f x t1 (body, t2) (e, t) = (LetFun(loc, f, (x, t1, body), t2, e), t)

let make_letrecfun loc f x t1 (body, t2) (e, t) = (LetRecFun(loc, f, (x, t1, body), t2, e), t)

let make_eq loc (e1, t1) (e2, t2)    = 
    match t1, t2 with 
    | (TEbool, TEbool) -> (Op(loc, e1, EQB, e2), TEbool)
    | (TEint, TEint)   -> (Op(loc, e1, EQI, e2), TEbool)
    | (_, _) -> complain "(4)" 


let rec  infer env e = 
    match e with 
    | Unit _              -> (e, TEunit)
    | What _              -> (e, TEint) 
    | Integer _           -> (e, TEint) 
    | Boolean _           -> (e, TEbool)
    | Var (loc, x)        -> (e, find loc x env)
    | UnaryOp(loc, NEG, e) -> make_unary loc NEG TEint (expect env (e, TEint)) 
    | UnaryOp(loc, NOT, e) -> make_unary loc NOT TEbool (expect env (e, TEbool))
    | UnaryOp(_, READ, Unit _) -> (e, TEunit)
    | Op(loc, e1, LT, e2)  -> make_op loc LT TEbool (expect env (e1, TEint)) (expect env (e2, TEint)) 
    | Op(loc, e1, OR, e2)  -> make_op loc OR TEbool (expect env (e1, TEbool)) (expect env (e2, TEbool)) 
    | Op(loc, e1, AND, e2) -> make_op loc AND TEbool (expect env (e1, TEbool)) (expect env (e2, TEbool)) 
    | Op(loc, e1, ADD, e2) -> make_op loc ADD TEint (expect env (e1, TEint)) (expect env (e2, TEint)) 
    | Op(loc, e1, SUB, e2) -> make_op loc SUB TEint (expect env (e1, TEint)) (expect env (e2, TEint)) 
    | Op(loc, e1, MUL, e2) -> make_op loc MUL TEint (expect env (e1, TEint)) (expect env (e2, TEint)) 
    | Op(loc, e1, EQ, e2)  -> make_eq loc (infer env e1) (infer env e2)
    | If(loc, e1, e2, e3)  -> make_if loc (expect env (e1, TEbool)) (infer env e2) (infer env e3)          
    | Pair(loc, e1, e2)    -> make_pair loc (infer env e1) (infer env e2) 
    | Fst(loc, e)          -> make_fst loc (infer env e)
    | Snd (loc, e)         -> make_snd loc (infer env e)
    | Inl(loc, e)          -> make_inl loc TEwildcard (infer env e)
    | Inr(loc, e)          -> make_inr loc TEwildcard (infer env e) 

    | Case(loc, e, (x1, e1), (x2, e2)) ->  
      (match infer env e with 
      | (e', TEunion(t1, t2)) -> 
        let (e1', t3) = infer ((x1, t1) :: env) e1 in 
        let (e2', t4) = infer ((x2, t2) :: env) e2 in 
        if t3 = t4 
        then (Case(loc, e', (x1, e1'), (x2, e2')), t3)
        else complain "(5)" 
      | (_, _) -> complain "(6)")

    | Lambda (loc, (x, t, e)) -> make_lambda loc x t (infer ((x, t) :: env) e)
    | App(loc, e1, e2)      -> make_app loc (infer env e1) (infer env e2)
    | Let(loc, bl, e)       -> 
       let (env', bl') = check_bindings env loc bl in make_let loc bl' (infer env' e) 
    | LetFun(loc, f, (x, t1, body), t2, e) -> 
      let p = infer ((f, TEarrow(t1, t2)) :: env) e  in 
      let env1 = (x, t1) :: env in 
        (try 
          make_letfun loc f x t1 (expect env1 (body, t2)) p 
        with _ -> make_letrecfun loc f x t1 (expect  ((f, TEarrow(t1, t2)) :: env1) (body, t2)) p )
    | LetRecFun(_, _, _, _, _)  -> complain "Internal Error --- LetRecFun construct never passed from parser!" 
    | _ -> complain "(17)"


and expect env p = 
    match p with 
    | (Unit _,   TEunit)    -> p
    | (What _,    TEint)    -> p 
    | (Integer _, TEint)    -> p 
    | (Boolean _, TEbool)   -> p
(* example 

    let add(p : (int * int)) : int = (fst(p) + snd(p))  in add (1, add (88, 12)) 
                                          ^ 
    p is expected to have type ( _  * int)

*) 
    | (Var (loc, x), t1)     -> 
        let t2 = find loc x env in 
        if match_types(t1, t2) 
        then p 
	else complain ("variable " ^ x 
		       ^ " is expected to have type "
		       ^ (Pp_past.string_of_type t1) 
		       ^ " but has type type "
		       ^ (Pp_past.string_of_type t2) )
    | (UnaryOp(loc, NEG, e), TEint)  -> make_unary loc NEG TEint (expect env (e, TEint)) 
    | (UnaryOp(loc, NOT, e), TEbool) -> make_unary loc NOT TEbool (expect env (e, TEbool))
    | (UnaryOp(_, READ, Unit _), TEunit) -> p 
    | (Op(loc, e1, LT, e2),  TEbool) -> make_op loc LT TEbool (expect env (e1, TEint)) (expect env (e2, TEint)) 

    | (Op(loc, e1, OR, e2),  TEbool) -> make_op loc OR TEbool (expect env (e1, TEbool)) (expect env (e2, TEbool)) 
    | (Op(loc, e1, AND, e2), TEbool) -> make_op loc AND TEbool (expect env (e1, TEbool)) (expect env (e2, TEbool)) 
    | (Op(loc, e1, ADD, e2), TEint)  -> make_op loc ADD TEint (expect env (e1, TEint)) (expect env (e2, TEint)) 
    | (Op(loc, e1, SUB, e2), TEint)  -> make_op loc SUB TEint (expect env (e1, TEint)) (expect env (e2, TEint)) 
    | (Op(loc, e1, MUL, e2), TEint)  -> make_op loc MUL TEint (expect env (e1, TEint)) (expect env (e2, TEint)) 
    | (Op(loc, e1, EQ, e2), TEbool)  -> make_eq loc (infer env e1) (infer env e2)
    | (If(loc, e1, e2, e3), t)       -> 
         make_if loc (expect env (e1, TEbool)) (expect env (e2, t)) (expect env (e3, t))          
    | (Pair(loc, e1, e2), TEproduct(t1, t2)) -> make_pair loc  (expect env (e1, t1)) (expect env (e2, t2)) 
    | (Fst(loc, e), t1) -> make_fst loc (expect env (e, TEproduct(t1, TEwildcard)))
    | (Snd(loc, e), t2) -> make_snd loc (expect env (e, TEproduct(TEwildcard, t2)))
    | (Inl(loc, e), TEunion(t1, t2)) -> make_inl loc t2 (expect env (e, t1))
    | (Inr(loc, e), TEunion(t1, t2)) -> make_inr loc t1 (expect env (e, t2))

    | (Case(loc, e, (x1, e1), (x2, e2)), t) ->  
      (match infer env e with 
      | (e', TEunion(t1, t2)) -> 
        let (e1', _) = expect ((x1, t1) :: env) (e1, t) in 
        let (e2', _) = expect ((x2, t2) :: env) (e2, t) in 
            (Case(loc, e', (x1, e1'), (x2, e2')), t)
      | (_, _) -> complain "(7)")

    | (Lambda (loc, (x, t, e)), TEarrow(t1, t2)) -> 
       if t = t1 
       then make_lambda loc x t (expect ((x, t) :: env) (e, t2)) 
       else complain "(8)" 
    | (App(loc, e1, e2), t) -> let (e2', t2) = infer env e2 in make_app loc (expect env (e1, TEarrow(t2, t))) (e2', t2) 
    | (Let(loc, bl, e), t)  -> 
       let (env', bl') = check_bindings env loc bl in make_let loc bl' (expect env' (e, t)) 
    | (LetFun(loc, f, (x, t1, body), t2, e), t) -> 
      let p = (expect ((f, TEarrow(t1, t2)) :: env) (e, t))  in 
      let env1 = (x, t1) :: env in 
        (try 
          make_letfun loc f x t1 (expect env1 (body, t2)) p 
        with _ -> make_letrecfun loc f x t1 (expect  ((f, TEarrow(t1, t2)) :: env1) (body, t2)) p )
    | (LetRecFun(_, _, _, _, _), _) -> complain "Internal Error --- LetRecFun construct never passed from parser!" 
    | (e, TEwildcard) -> infer env e (* really ?? When does this work? *) 
    | (e, t) -> complain "(9)" 
       
and check_bindings env loc bl = 
    let rec aux ecarry bcarry = function 
      | [] -> (ecarry @ env, List.rev bcarry) 
      | (x, t, e) :: rest -> 
        let (e', _) = expect env (e, t) in aux ((x, t) :: ecarry) ((x, t, e') :: bcarry) rest 
    in aux [] [] bl

let env_init = [] 

let check e = 
    let (e', _) = infer env_init e 
    in e' 

