(**************************************
Compiler Construction 2015
Computer Laboratory 
University of Cambridge 
Timothy G. Griffin (tgg22@cam.ac.uk) 
*****************************************) 

(* Interpreter 4. 

   We split the single "stack" of interp_3.ml into 
   three stacks. 

   In interp_3.ml we have 

   step : state -> state 

   where 

   type state = 
      | EVAL of expr * env * continuation 
      | APPLY of continuation * value 


   Here we transform this step function to work on 
   states of the form 

   (ds, es, vs), 

   where ds is a stack of "directives" (these 
   are of a datatype derived from interp_3.continuation_item
   by stripping out values and environments),
   es is a stack of environments, and vs is a stack of values. 

   We can define a representation function r that takes each 
   Interp_3.state to an Interp_4.state as 

    r(EVAL(e, env, k) = (E e) :: ds(k), env :: es(k), vs(k)) 
    r(APPLY(k, v))    = (ds(k), es(k), v :: vs(k))

   Then, for each step rule in interp_3.ml of the form 

     | s1 -> s2 

   we simply need a step rule of the form 

     | r(s1) -> r(s2) 

   This is done, together with a few minor "optimizations". 

*) 


open Ast 
open Common 

let verbose = ref false 

type value = 
     | CON of Common.constant 
     | PAIR of value * value 
     | INL of value 
     | INR of value 
     | FUN of closure 

and closure = Past.var * expr * env 

and directive = 
  | E of expr 
  | C_UNARY of Past.unary_oper 
  | C_OPER of Past.oper 
  | C_OPER_FST of expr * Past.oper 
  | C_IF of expr * expr 
  | C_PAIR 
  | C_PAIR_FST of expr 
  | C_FST
  | C_SND
  | C_INL
  | C_INR
  | C_CASE of Past.var * expr * Past.var * expr 
  | C_APPLY of closure 
  | C_ARG of expr 

and directive_stack = directive  list

and env = Past.var -> value 

type env_stack = env list 

type value_stack = value list 

type binding = Past.var * value

type bindings = binding list

type state = directive_stack * env_stack * value_stack 

(* update : (env * binding) -> env *) 
let update(env, (x, v)) = fun y -> if x = y then v else env y

let do_unary = function 
  | (Past.NOT, CON (BOOL m)) -> CON(BOOL (not m))
  | (Past.NEG, CON (INT m))  -> CON(INT (-m))
  | (Past.READ, CON(UNIT))   -> CON(INT (readint()))
  | _ -> complain "malformed unary operator"

let do_oper = function 
  | (Past.AND, CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m && n))
  | (Past.OR,  CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m || n))
  | (Past.EQB, CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m = n))
  | (Past.LT, CON (INT m), CON (INT n)) -> CON(BOOL (m < n))
  | (Past.EQI, CON (INT m), CON (INT n)) -> CON(BOOL (m = n))
  | (Past.ADD, CON (INT m), CON (INT n)) -> CON(INT (m + n))
  | (Past.SUB, CON (INT m), CON (INT n)) -> CON(INT (m - n))
  | (Past.MUL, CON (INT m), CON (INT n)) -> CON(INT (m * n))
  | (op, _, _)  -> complain ("malformed binary operator: " ^ (Pp_past.string_of_oper op))

(* step : state -> state. 

   Evaluation of each expression consumes the env on top of the env_stack 
   and pushes a value onto the value_stack. 

  ((E e) :: ds, env :: es, vs) 
  -> ... 
  -> ... 
  -> ... 
  -> (ds, es, v :: vs) 

*) 

let step = function 
 (* EVAL -> EVAL *) 
 | (E(UnaryOp(op, e)) :: ds, es,        vs) -> ((E e)  :: (C_UNARY op) :: ds, es, vs)
 | (E(Op(e1, op, e2)) :: ds, env :: es, vs) -> ((E e1) :: (C_OPER_FST(e2, op)) :: ds,  env :: env :: es, vs)
 | (E(If(e1, e2, e3)) :: ds, env :: es, vs) -> ((E e1) :: (C_IF(e2, e3)) :: ds,        env :: env :: es, vs)
 | (E(Pair(e1, e2)) :: ds,   env :: es, vs) -> ((E e1) :: C_PAIR_FST(e2) :: ds,        env :: env :: es, vs)
 | (E(App(e1, e2)) :: ds,    env :: es, vs) -> ((E e1) :: C_ARG(e2) :: ds,             env :: env :: es, vs)
 | (E(Fst e) :: ds, es,                 vs) -> ((E e)  :: C_FST :: ds, es, vs)
 | (E(Snd e) :: ds, es,                 vs) -> ((E e)  :: C_SND :: ds, es, vs) 
 | (E(Inl e) :: ds, es,                 vs) -> ((E e)  :: C_INL :: ds, es, vs)
 | (E(Inr e) :: ds, es,                 vs) -> ((E e)  :: C_INR :: ds, es, vs) 
 | (E(LetFun(f, (x, body), e)) :: ds, env::es, vs) ->  ((E e) :: ds, (update(env, (f, FUN(x, body, env)))) :: es, vs) 
 | (E(LetRecFun(f, (x, body), e)) :: ds, env :: es, vs) -> 
       let rec new_env g = (* a recursive environment! *) 
           if g = f then FUN(x, body, new_env) else env g
       in ((E e) :: ds, new_env :: es, vs) 
 | (E(Case(e, (x1, e1), (x2, e2))) :: ds, env :: es, vs) -> ((E e) :: C_CASE(x1, e1, x1, e2) :: ds, env :: env :: es , vs)
 (* EVAL -> APPLY *) 
 | ((E Unit) :: ds,          _ :: es, vs) -> (ds, es, (CON UNIT) :: vs) 
 | (E(Integer n) :: ds,      _ :: es, vs) -> (ds, es, CON(INT n) :: vs)
 | (E(Boolean b) :: ds,      _ :: es, vs) -> (ds, es, CON (BOOL b) :: vs)
 | (E(Var x) :: ds,        env :: es, vs) -> (ds, es, (env x) :: vs)
 | (E(Lambda(x, e)) :: ds, env :: es, vs) -> (ds, es, FUN(x, e, env) ::vs)
 (* APPLY -> APPLY *) 
 | ((C_UNARY op) :: ds, es,          v::vs) -> (ds, es, (do_unary(op, v)) :: vs) 
 | ((C_OPER op) :: ds,  es, v2 :: v1 :: vs) -> (ds, es, (do_oper(op, v1, v2)) :: vs)
 | (C_PAIR :: ds, es,       v2 :: v1 :: vs) -> (ds, es, (PAIR(v1, v2)) :: vs)
 | (C_FST :: ds, es,   (PAIR (v, _)) :: vs) -> (ds, es, v :: vs)
 | (C_SND :: ds, es,   (PAIR (_, v)) :: vs) -> (ds, es, v :: vs)
 | (C_INL :: ds, es,               v :: vs) -> (ds, es, (INL v) :: vs)
 | (C_INR :: ds, es,               v :: vs) -> (ds, es, (INR v) :: vs)
 (* APPLY -> EVAL *) 
 | (C_PAIR_FST (e) :: ds,         es,                   vs) -> ((E e) :: C_PAIR :: ds, es, vs)
 | (C_OPER_FST (e, op) :: ds,     es,                   vs) -> ((E e) :: (C_OPER op) :: ds, es, vs)
 | ((C_APPLY (x, e, env)) :: ds,  es,               v ::vs) -> ((E e) :: ds, (update(env, (x, v))) :: es, vs)
 | (C_ARG (e) :: ds,              es,        (FUN cl) ::vs) -> ((E e) :: (C_APPLY cl) :: ds, es, vs)
 | (C_IF (e2, e3) :: ds, es,  (CON (BOOL true)) :: vs) -> ((E e2) :: ds, es, vs) 
 | (C_IF (e2, e3) :: ds, es, (CON (BOOL false)) :: vs) -> ((E e3) :: ds, es, vs) 
 | (C_CASE (x1, e1,  _,  _) :: ds,  env :: es, (INL v)::vs) -> ((E e1) :: ds, (update(env, (x1, v))) :: es, vs) 
 | (C_CASE ( _,  _, x2, e2) :: ds,  env :: es, (INR v)::vs) -> ((E e2) :: ds, (update(env, (x2, v))) :: es, vs) 
 | state -> complain "step : bad state" 

let rec driver = function 
  | ([], _, [v]) -> v 
  | state -> driver (step state) 


(* env_empty : env *) 
let env_empty = fun x -> complain (x ^ " is not defined!\n")

let eval(e, env) = driver([E e], [env], [])

(* interpret : expr -> simple_value *) 
let interpret e = eval(e, env_empty)

let rec string_of_value = function 
     | CON c -> string_of_constant c
     | PAIR(v1, v2) -> "(" ^ (string_of_value v1) ^ ", " ^ (string_of_value v2) ^ ")"
     | INL v -> "inl(" ^ (string_of_value v) ^ ")"
     | INR  v -> "inr(" ^ (string_of_value v) ^ ")"
     | FUN _ -> "FUNCTION( ... )" 
    

      
    
    
