open Lexing 

let verbose = ref false 

let error file action s = 
    let _ = print_string ("\nERROR in " ^ file ^ " with " ^ action ^ " : " ^ s ^ "\n") 
    in exit(-1) 

let parse_error file lexbuf = 
    let pos = lexbuf.lex_curr_p in 
    let line = string_of_int (pos.pos_lnum) in 
     let pos = string_of_int ((pos.pos_cnum - pos.pos_bol) + 1) in 
        error file "parsing" ("at line " ^ line ^ " position " ^ pos)

(* get_ast : 
   1) sets up lexer 
   2) parse input file 
      For more on lexing/parsing in Ocaml see Ch. 16 of "Real Workd Ocaml" 
   3) performes static checks
   4) translates from Past.expr to Ast.expr 
   5) alpha convert result so that all bound vars are unique 
*) 
let get_ast (file, in_chan) = 

 (* initialize lexer *) 
  let lexbuf = from_channel in_chan 
  in let _ = lexbuf.lex_curr_p <- 
           { pos_fname = file; 
  	     pos_lnum = 1;
  	     pos_bol = 0; 
  	     pos_cnum = 0; 
           }

 (* parse input file *) 
  in let e1 = try Parser.start Lexer.token lexbuf 
              with Parsing.Parse_error -> parse_error file lexbuf 

  in let _ = if !verbose 
             then (print_string "\nParsed result: \n";
                  Pp_past.print_expr e1; 
	          print_string "\n")
             else () 

 (* perform static checks *) 
  in let e2 = try Static.check e1 
              with Static.Error s -> error file "static check" s 

  in let _ = if !verbose 
             then (print_string "\nAfter static check : \n"; 
                   Pp_past.print_expr e2; 
                   print_string "\n")
             else () 

  (* translate from Past.expr to Ast.expr *) 
  in let e3 = Past_to_ast.translate_expr e2 

  in let _ = if !verbose 
             then (print_string "\nAfter Past_to_ast.translate_expr : \n";
                   Pp_ast.print_expr e3; 
                   print_string "\n")		      
             else () 

  (* alpha convert bound vars *) 
  in let e4 = Alpha.convert e3

  in let _ = if !verbose 
             then (print_string "\nAfter Alpha.convert : \n";
                   Pp_ast.print_expr e4; 
                   print_string "\n")		      
             else () 
  in e4
