open Common 

let verbose = ref false 

type oper = ADD | MUL | SUB | LT | AND | OR | EQB | EQI 

type unary_oper = NEG | NOT 

type label = string 
type code_index = int 
type stack_index = int 
type offset = int 
type static_distance = int 

let stack_max = ref 10000

type status_code = 
  | Halted 
  | Running 
  | CodeIndexOutOfBound 
  | StackIndexOutOfBound 
  | StackUnderflow 

type stack_item = 
  | CONSTANT of constant 
  | CODE_INDEX of code_index   (* return address               *) 
  | STACK_INDEX of stack_index (* frame pointer or static link *) 

type instruction = 
  | PUSH of constant 
  | GOTO of label * (code_index option) 
  | TEST of label * (code_index option) 
  | CALL of label * (code_index option) * static_distance 
  | ARG of static_distance * offset 
  | DoOp of oper 
  | DoUnary of unary_oper 
  | RETURN of int 
  | SKIP 
  | READ 
  | HALT 

type l_instruction = NotLabelled of instruction  | Labelled of label * instruction 

type listing = l_instruction list 

type vm_state = 
  {
    stack_bound : stack_index; 
    code_bound : code_index; 
    stack : stack_item array; 
    program : l_instruction array; 
    mutable sp : stack_index;  (* stack pointer *) 
    mutable fp : stack_index;  (* frame pointer *) 
    mutable cp : code_index;   (* code pointer  *) 
    mutable status : status_code; 
  } 

let stack_init()  = (Array.create !stack_max (CONSTANT(INT 0)), !stack_max)

let get_instruction = function NotLabelled inst -> inst | Labelled(_, inst) -> inst 

let map_l_instruction f = function 
  | NotLabelled inst -> NotLabelled (f inst) 
  | Labelled (lab, inst) -> Labelled (lab, f inst) 

let rec find lab = function 
  | [] -> complain ("find : " ^ lab ^ " is not found")
  | (x, v) :: rest -> if x = lab then v else find lab rest 

(* insert array index for each label *) 
let apply_label_map_to_instruction m = function 
  | GOTO (lab, _) -> GOTO(lab, Some(find lab m))
  | TEST (lab, _) -> TEST(lab, Some(find lab m))
  | CALL (lab, _, d) -> CALL (lab, Some(find lab m), d) 
  | inst -> inst 

(* find array index for each label *) 
let listing_to_label_map l = 
    let rec aux carry k = function 
      | [] -> carry 
      | (NotLabelled _) :: rest -> aux carry (k+1) rest 
      | (Labelled (lab, _)) :: rest -> aux ((lab, k) :: carry) (k+1) rest 
    in aux [] 0 l 

(* put listing into an array, associate an array index to each label *) 
let listing_to_code_array l = 
    let new_l = List.map (map_l_instruction (apply_label_map_to_instruction (listing_to_label_map l))) l in 
    (Array.of_list new_l, List.length l)  


let stack_top vm = Array.get vm.stack (vm.sp - 1) 

(* pop n items from stack *) 
let pop(n, vm) = 
    if 0 <= vm.sp - n 
    then { vm with sp =vm.sp - n  } 
    else { vm with status = StackUnderflow } 

let pop_top vm = let c = stack_top vm in (c, pop(1, vm))

let perform_op = function 
  | (ADD, CONSTANT(INT m), CONSTANT(INT n)) -> CONSTANT(INT (m + n))
  | (MUL, CONSTANT(INT m), CONSTANT(INT n)) -> CONSTANT(INT (m * n))
  | (SUB, CONSTANT(INT m), CONSTANT(INT n)) -> CONSTANT(INT (m - n))
  | (LT,  CONSTANT(INT m), CONSTANT(INT n)) -> CONSTANT(BOOL (m < n))
  | (EQI,  CONSTANT(INT m), CONSTANT(INT n)) -> CONSTANT(BOOL (m = n))
  | (EQB,  CONSTANT(BOOL m), CONSTANT(BOOL n)) -> CONSTANT(BOOL (m = n))
  | (AND,  CONSTANT(BOOL m), CONSTANT(BOOL n)) -> CONSTANT(BOOL (m && n))
  | (OR,   CONSTANT(BOOL m), CONSTANT(BOOL n)) -> CONSTANT(BOOL (m || n))
  | _ -> complain "perform_op : malformed"

let perform_unary = function 
  | (NEG, CONSTANT(INT m)) -> CONSTANT(INT (-m))
  | (NOT, CONSTANT(BOOL m)) -> CONSTANT(BOOL (not m))
  | _ -> complain "perform_unary : malformed"

(* pop c onto stack  *) 
let push(c, vm) = 
    if vm.sp < vm.stack_bound 
    then let _ = Array.set vm.stack vm.sp c in { vm with sp = vm.sp + 1 } 
    else { vm with status = StackIndexOutOfBound } 

let do_op(op, vm) = 
    let (v_right, vm1) = pop_top vm in 
    let (v_left, vm2) = pop_top vm1 in 
      push(perform_op(op, v_left, v_right), vm2) 

let do_unary(op, vm) = 
    let (v, vm1) = pop_top vm in push(perform_unary(op, v), vm1) 

(* cp := cp + 1  *)     
let advance_cp vm = 
    if vm.cp < vm.code_bound 
    then { vm with cp = vm.cp + 1 } 
    else { vm with status = CodeIndexOutOfBound } 

let goto(i, vm) = { vm with cp = i } 

let test(i, vm) = 
    pop(1, 
       if stack_top vm = CONSTANT(BOOL true)  
       then advance_cp vm 
       else { vm with cp = i })

let get_constant = function 
  | CONSTANT c -> c 
  | _ -> complain "Jargon.get_constant : expecting constant"

let return(i, vm) = 
    match Array.get vm.stack vm.fp, Array.get vm.stack (vm.fp + 1) with 
    | (STACK_INDEX saved_fp, CODE_INDEX k) -> 
       let (v, vm0) = pop_top vm in 
       let vm1 = {vm0 with sp = vm.fp} in
       let vm2 = pop(i, vm1) in 
       let vm3 = push(v, vm2) in { vm3 with cp = k; fp = saved_fp } 
    | _ -> complain "return : malformed stack frame" 

let get_static_link (fp, vm) = 
    match Array.get vm.stack (fp + 2) with 
    | STACK_INDEX i -> i 
    | _ -> complain "get_static_link : expecting stack index!"

let arg(i, d, vm) = 
    let rec aux (fp, j) = 
        if j = 0 
        then push(Array.get vm.stack (fp - i), vm) 
        else aux(get_static_link (fp, vm), j - 1) 
    in aux(vm.fp, d) 

let call(i, d, vm) = 
    let rec initialize_static_link(fp, j) = 
        if j = -1 
        then fp 
        else if j = 0
             then get_static_link (fp, vm)
             else initialize_static_link(get_static_link (fp, vm), j - 1)
    in let new_fp = vm.sp 
    in let saved_fp = STACK_INDEX vm.fp
    in let return_index = CODE_INDEX (vm.cp + 1) 
    in let static_link = STACK_INDEX (initialize_static_link (vm.fp, d)) 
    in push(static_link, push(return_index, push(saved_fp, { vm with cp = i; fp = new_fp } )))


let first_frame vm = 
    let saved_fp = STACK_INDEX 0
    in let return_index = CODE_INDEX 0 
    in let static_link = STACK_INDEX 0 
    in push(static_link, push(return_index, push(saved_fp, vm)))

let step vm = 
    match  get_instruction (Array.get vm.program vm.cp) with 
    | HALT                -> { vm with status = Halted } 
    | RETURN i            -> return(i, vm) 
    | GOTO (_, Some i)    -> goto(i, vm) 
    | TEST (_, Some i)    -> test(i, vm) 
    | CALL (_, Some i, d) -> call(i, d, vm) 
    | ARG (d, i)          -> advance_cp (arg(i, d, vm)) 
    | DoOp op             -> advance_cp (do_op(op, vm)) 
    | DoUnary op          -> advance_cp (do_unary(op, vm)) 
    | PUSH c              -> advance_cp (push(CONSTANT c, vm)) 
    | READ                -> advance_cp (push(CONSTANT (INT (readint())), vm)) 
    | SKIP                -> advance_cp vm 
    | GOTO (_, None)      -> complain "step: internal error. goto has no code index"
    | TEST (_, None)      -> complain "step: internal error. test has no code index"
    | CALL (_, None, _)   -> complain "step: internal error. call has no code index"





(* pretty printing for verbose stack trace *)


let string_of_status = function 
  | Halted -> "halted" 
  | Running -> "running" 
  | CodeIndexOutOfBound -> "code index out-of-bound" 
  | StackIndexOutOfBound -> "stack index out-of-bound" 
  | StackUnderflow -> "stack underflow" 

let string_of_stack_item = function 
  | CONSTANT c -> string_of_constant c
  | CODE_INDEX i -> "CI " ^ (string_of_int i)
  | STACK_INDEX i -> "SI " ^ (string_of_int i)

let string_of_instruction = function 
  | PUSH c -> "push " ^ ( string_of_constant c)
  | GOTO(l, _) -> "goto " ^ l 
  | TEST(l, _) -> "tst " ^ l 
  | CALL(l, _, d)   -> "call " ^ l ^ " " ^ (string_of_int d) 
  | ARG (distance, offset) -> "arg " ^ (string_of_int distance) ^ " " ^ (string_of_int offset) 
  | DoOp ADD -> "add"
  | DoOp MUL -> "mul"
  | DoOp SUB -> "sub"
  | DoOp LT -> "lt"
  | DoOp AND -> "and"
  | DoOp OR -> "or"
  | DoOp EQB -> "eqb"
  | DoOp EQI -> "eqi"
  | DoUnary NEG -> "neg"
  | DoUnary NOT -> "not"
  | RETURN n -> "return " ^ (string_of_int n) 
  | SKIP -> "skip"
  | READ -> "read"
  | HALT -> "halt"

let string_of_l_instruction = function 
  | NotLabelled inst -> string_of_instruction inst 
  | Labelled(l, inst) -> l ^ ": " ^ (string_of_instruction inst) 

let rec string_of_listing = function 
  | [] -> ""  
  | i :: rest -> (string_of_l_instruction i) ^ "\n" ^ (string_of_listing rest) 

let string_of_stack(sp, stack) = 
    let rec aux carry j = 
         if j = sp then carry 
         else aux ((string_of_int j) ^ ": " 
                   ^ (string_of_stack_item (Array.get stack j)) 
                   ^ "\n" ^ carry) (j + 1) 
    in aux "" 0

let string_of_state vm = 
    "stack = \n" ^(string_of_stack(vm.sp, vm.stack)) ^ "\n"
  ^ "cp = " ^ (string_of_int vm.cp) ^ " -> " ^ (string_of_instruction (get_instruction (Array.get vm.program vm.cp))) ^ "\n"
  ^ "fp = " ^ (string_of_int vm.fp) ^ "\n"

(**************************************) 

let rec driver n vm = 
    let _ = if !verbose then print_string ("jargon state " ^ (string_of_int n) ^ "\n" ^ (string_of_state vm) ^ "\n") else () in 
    if vm.status = Running 
    then driver (n+1) (step vm) 
    else vm 

let load l = 
  let (stack_array, s_bound) = stack_init() in 
  let (code_array, c_bound) = listing_to_code_array l in 
  { 
    stack_bound = s_bound; 
    code_bound = c_bound; 
    stack = stack_array ; 
    program = code_array; 
    sp = 0; 
    fp = 0; 
    cp = 0; 
    status = Running; 
  } 

let run l = 
    let vm = driver 1 (first_frame (load l)) in 
    match vm.status with 
    | Halted   -> get_constant (stack_top vm) 
    | status -> complain ("run : stopped wth status " ^ (string_of_status status))

