
open Ast 

(*  If e' = convert e
    then every bound variable in e' is unique. 
    This is done n a lazy way, so that if 
    e has no two bound vars with the same name, 
    then e' = e. Also, if name "f" is changed, 
    it will be changed to something of the form 
    "_f" or "__f" or "___f",  and so on. 
    This appraoch is taken in order to make it 
    easier for users to see their input programs   
    move through multiple interpreter/compiler phases. 
*) 

let seen = ref [] 

let rec inlist x = function 
  | [] ->  false
  | y :: rest -> if x = y then true else (inlist x rest)

let rec get_new_var f = 
    let g = "_" ^ f
    in  if inlist g (!seen) 
        then  get_new_var g 
        else  (seen := g :: (!seen); g) 

let rec lookup x = function 
  | [] ->  x
  | (y, z) :: rest -> if x = y then z else (lookup x rest)

let check_bound_var env f = 
    if inlist f (!seen) 
    then let g = get_new_var f in ((f, g) :: env, g) 
    else (seen := f :: (!seen); (env, f))

let rec check_bound_vars env carry = function 
  | [] -> (env, List.rev carry) 
  | x :: rest -> 
       let (env', y) = check_bound_var env x
       in  check_bound_vars env' (y :: carry) rest 

let rec aux env e = 
    match e with 
    | Unit           -> e
    | Integer _      -> e  
    | Boolean _      -> e 
    | Var x          -> Var (lookup x env)
    | If(e1, e2, e3) -> If(aux env e1, aux env e2,  aux env e3)
    | App(f, el)     -> App(lookup f env , List.map (aux env) el)
    | LetFun(f, fl, e1, e2) -> 
      let (env1, g) = check_bound_var env f 
      and (env2, fl_new) = check_bound_vars env [] fl
      in LetFun(g, fl_new,  aux env2 e1, aux env1 e2)
    | LetRecFun(f, fl, e1, e2) -> 
      let (env1, g) = check_bound_var env f 
      in let (env2, fl_new) = check_bound_vars env1 [] fl
      in LetRecFun(g, fl_new, aux env2 e1, aux env2 e2)

let convert e = 
    let _ = seen := [] (* start fresh with each new e *) 
    in aux [] e 
