// CBG TLM/ESL console uart model: in popup xterm or using stdin/out depending on the console mode
//
// (C) 2009-11 D J Greaves
// University of Cambridge, Computer Laboratory.
// ACS P35 SoC D/M Classes - $Id: uart64_cbg.cpp,v 1.3 2011/07/25 15:33:56 my294 Exp $


#define UTRC(X) 
#include "uart64_cbg.h"




void *input_caller0(void *it)
{
  return ((uart64_cbg*)it)->input_process(0);
}

void *input_caller1(void *it)
{
  return ((uart64_cbg*)it)->input_process((void *)1);
}


void *uart64_cbg::input_process(void *arg)
{
  while (1) // Copy input characters from any source into the logged variable.
    {
      if (yielding || logged) 
	{
	  usleep(10000);
	  continue;
	}

      if (arg)
	{
	  unsigned char ch;
	  // printf("Hang in xterm read\n");
	  int c = read(xterm_fd, &ch, 1); 
	  if (c) logged = ch | 256;
	}
      else
	{
	  unsigned char c = getchar();
	  //	  printf("hang in getchar >%c<\n", c);
	  // Set bit 8 as a non zero value that is not part of the character.
	  logged = c | 256;
	}
    }
}



// The input side is a bit promiscuous, reading from the xterm or stdin...
int uart64_cbg::testch()
{
  if (canned_input) return 1;
  if (!xterm_opened) open_xterm();
  if (!threads_started)
    {
      threads_started = 1;
      if (use_stdin) pthread_create(&input_thread0, 0, input_caller0, (void *)this);
      if (xterm_fd >= 0) pthread_create(&input_thread1, 0, input_caller1, (void *)this);
      if (!use_stdin && xterm_fd < 0) printf("%s: Warning: No input devices\n", name());
    }
  if (!logged) 
    {
      idle_count += 1;
      if (idle_count == 1000)
	{
	  // Yield the processor if the input is idle, to save battery
	  // or be nice to other users.
	  yielding = 1;
	  usleep(100000);
	  yielding = 0;
	  idle_count = 0;
	}
    }
  return logged != 0;
}


// Note: Immediate string or file data for uart model input.
// Note: the uart model will handle input starting with a dot or a slash as a file to read from
void uart64_cbg::register_canned_input(const char *d)
{
  if (d && (d[0] == '/' ||d[0] == '.'))
    {
      FILE *fd = fopen(d, "r");
      if (!fd)
	{
	  perror("");
	  SC_REPORT_FATAL(name(), "Cannot open canned input file");
	}
      else canned_fd = fd;
    }
  canned_input = d;
}

char uart64_cbg::rdch_nonblock()
{
  char r = 0;
  if (canned_input)
    {
      if (canned_fd)
	{
	  r = getc(canned_fd); // input from file
	  if (feof(canned_fd))
	    {
	      fclose(canned_fd);
	      canned_fd = 0;
	      canned_input = 0;
	    }
	}
      else
	{
	  r = *canned_input; // immediate string input
	  canned_input ++;
	  if (!*canned_input) canned_input = 0;
	  if (r == '\\') // handle escapes
	    {
	      r = *canned_input; 
	      canned_input ++;
	      if (!*canned_input) canned_input = 0;
	      switch (r)
		{
		case 'q':
		  fprintf(stderr, "Backdoor quit sc_stop at %s\n", name());
		  sc_stop;
		  break;

		case 'n': r = '\n'; break;
		case 'r': r = '\r'; break;
		case 'b': r = '\b'; break;
		case 'f': r = '\f'; break;
		case '\\': r = '\\'; break;
		case 'v': r = '\v'; break;
		case 't': r = '\t'; break;
		default:  r &= 0x1F;  break;
		}
	    }
	}
    }
  else if (logged)
    {
      idle_count = 0;
      r = logged & 0xFF;
      logged = 0;
    }
  return r;
}

// Read input character
char uart64_cbg::rdch()
{
  while (1) if (testch()) return rdch_nonblock();
  return 0;
}

void uart64_cbg::end_of_simulation()
{
  flush();
  if (output_f)
    {
      fclose(output_f);
      output_f = 0;
    }
}


void uart64_cbg::flush()
{
  if (output_f) fflush(output_f); else fflush(stdout);
}


void uart64_cbg::open_xterm()
{
  xterm_opened = 1; // Set this first so that no second chance is attempted on fail.

  // If no display variable, then do not try - (but some UNIX envs do not use this var?)
  const char *DISPLAY = getenv("DISPLAY");
  if (!DISPLAY) return; 


  int fd = posix_openpt(/*"/dev/ptmx",*/ O_RDWR | O_NOCTTY);
  char ptsid[132], pts[132];
  if (fd < 0 || ptsname_r(fd, ptsid, 132))
    {
      perror("pt open failed");
    }
  UTRC(printf("pts %s fd=%i\n", ptsid, fd));
  sprintf(pts, "-Svt/%i", fd);
  unlockpt(fd);
  UTRC(printf("%s: Uart server is opening xterm\n", name()));
  char * args[5];
  char line0[132], line1[132], line2[132], line3[132];
  strcpy(line0, "/usr/bin/xterm");
  strcpy(line1, pts);
  strcpy(line2, "-title");
  strcpy(line3, name());
  args[0] = line0;
  args[1] = line1;
  args[2] = line2;
  args[3] = line3;
  args[4] = 0;
  for (int i=0; args[i]; i++) printf("%s ", args[i]);
  printf("\n");
  if (!fork())
    {
      execv(args[0], args);
    }
  else
    {
      usleep(50000);
      int fd1 = open(ptsid, O_RDWR /* O_NONBLOCK*/);
      if (fd1 < 0)
	{
	  printf("%s: Failed to open %s\n", name(), ptsid);
	  perror("cannot open uart terminal");
	  exit(1);
	}
      xterm_fd = fd1;
    }
  if (threads_started) printf("%s: error: threads started too soon\n", name());
}


// Write output character
void uart64_cbg::uart_wrch(char c)
{
  extern void tmp_trace_bd(const char *);
  //if ((c & 0xFF) == 0x2A)    tmp_trace_bd("uart 2A");
  if ((c & 0xFF) == 0377)    tmp_trace_bd("uart 0370");

  if (xconsole)
    {
      int b = 0;
      if (!xterm_opened) open_xterm();
      if (xterm_fd >= 0) b = write(xterm_fd, &c, 1);
    }
  if (output_f) putc(c, output_f);
  putchar(c);
  if (c == '\n') flush();
}


// Return 0 on ok.
int uart64_cbg::open_log_file(const char *fn)
{
  if (output_f) return -1;
  output_f = fopen(fn, "w");
  if (!output_f)
    {
      printf("%s: Could not open log file\n", name());
      perror("uart64_cbg:");
      return -1;
    }
  return 0;
}


// constructor
uart64_cbg::uart64_cbg(sc_module_name name, bool x, bool u) : 
  sc_module(name), 
#ifdef TLM_POWER3
  pw_module("power_config_uart.txt"), 
#if PW_TLM_PAYLOAD > 0
  read_bus_tracker(this),
#endif
#endif
  xconsole(x), 
  use_stdin(u)
{
  canned_input = 0;
  canned_fd = 0;
  traceregions = 0;
  idle_count = 0;
  xterm_fd = -1;
  output_f = 0;
  input_f = 0;
  xterm_opened = 0;
  logged = 0;
  yielding = 0;
  threads_started = 0;

  latency = sc_time(200, SC_NS); // Assume connected to a slow I/O external bus.

  port0.register_b_transport(this, &uart64_cbg::b_access);

  txready = true; // Always ready to send in this high-level model. 

#ifdef TLM_POWER3
  // based on: just made up!
  set_excess_area(pw_length(450, PW_um), pw_length(150,  PW_um));
#endif

}



void uart64_cbg::b_access(PW_TLM_PAYTYPE &trans, sc_time &delay)
{
  tlm::tlm_command cmd = trans.get_command();

  POWER3(PW_TLM3(pw_agent_record l_agent = trans.pw_log_hop(this,  (cmd==tlm::TLM_READ_COMMAND ? PW_TGP_DATA: PW_TGP_NOFIELDS) | PW_TGP_ACCT_CKP,  &read_bus_tracker)));

  u64_t    adr = ((u64_t)trans.get_address() & 0xfc);
  u8_t*	   ptr = trans.get_data_ptr();
  // u32_t    len = trans.get_data_length();
  // u8_t*    lanes = trans.get_byte_enable_ptr();
  // u32_t    wid = trans.get_streaming_width();
  
    
  // Obliged to check address range and check for unsupported features,
  //   i.e. byte enables, streaming, and bursts
  // Can ignore DMI hint and extensions
  // Using the SystemC report handler is an acceptable way of signalling an error
  
  u8_t reg_offset = adr & 0xF8; // Extract low bits of address bus for internal register.
  // Obliged to implement read and write commands

  tlm::tlm_response_status rc = tlm::TLM_ADDRESS_ERROR_RESPONSE;
  if (cmd == tlm::TLM_READ_COMMAND)
    {
      stats.reads += 1;
      u8_t r = 0x0;
      switch (reg_offset)
	{
	  case UART2_SEND:
	    rc = tlm::TLM_GENERIC_ERROR_RESPONSE; //Should not read the send register.
	    break;

	  case UART2_RECEIVE:
	    r = rdch_nonblock();
	    rc = tlm::TLM_OK_RESPONSE;
	    break;

	  case UART2_CONTROL:
	    r = control_register;
	    rc = tlm::TLM_OK_RESPONSE;
	    break;
	  
	  case UART2_STATUS:
	    r = status_register_read();
	    rc = tlm::TLM_OK_RESPONSE;
	    break;
	    
	  }

     
      ((u64_t *)ptr)[0] = r; // IO device uses endian-ness of hosting workstation.
      if (traceregions && traceregions->check(adr, TENOS_TRACE_IO_READ))
	printf("%s:%s: read uart addr=" PFX64 " reg_offset=%02X data=%02X\n", name(), kind(),  adr, reg_offset, r);
    }
    else if (cmd == tlm::TLM_WRITE_COMMAND)
      {
	stats.writes += 1;
	u8_t d = ((u64_t *)ptr)[0];
	if (traceregions && traceregions->check(adr, TENOS_TRACE_IO_WRITE))
	  printf("%s:%s: write uart addr=" PFX64 " reg_offset=%02X data=%02X '%c'\n", name(), kind(),  adr, reg_offset, d, isprint(d) ? d: '.');
	
	switch (reg_offset)
	  {
	  case UART2_SEND:
	    UTRC(printf("%s:%s: transmit wrch trace %x '%c'\n", name(), kind(),  isprint(d) ? d: '.'));
	    uart_wrch(d);
	    stats.outs += 1;
	    rc = tlm::TLM_OK_RESPONSE;
	    break;
	    

	  case UART2_RECEIVE:
	    rc = tlm::TLM_GENERIC_ERROR_RESPONSE; //Should not write the rx register.
	    break;

	  case UART2_CONTROL:
	    control_register = d;
	    UTRC(printf("Uart2 %s transmit write control reg %x\n", name(), w));
	    update_interrupt();
	    rc = tlm::TLM_OK_RESPONSE;
	    break;

	  case UART2_STATUS:
	    rc = tlm::TLM_GENERIC_ERROR_RESPONSE; //Should not write the status register.
	    break;


	  }

	
      }
  
  
  delay += latency;
  
  
  trans.set_response_status(rc);
#if PW_TLM_PAYLOAD > 0
  POWER3(l_agent.record_energy_use(pw_energy(1.0e-3 * 1.0e-6, PW_JOULE)));
#else
  POWER3(record_energy_use(pw_energy(1.0e-3 * 1.0e-6, PW_JOULE)));
#endif

  // Not all access cycles xfer a char however (especially when polled).
  // Very rough idea of uart energy per character 10 microseconds * 1 milliamp.
}


void uart64_cbg::update_interrupt()
{
  bool rxready = testch();
  bool intt = 
    ((control_register & UART2_CONTROL_TX_INT_ENABLE) && txready) || 
    ((control_register & UART2_CONTROL_RX_INT_ENABLE) && rxready);
  interrupt_request = intt;
  UTRC(printf("Update interrupt %i\n", intt));
}

u8_t uart64_cbg::status_register_read()
{
  u8_t r = 0;
  //#define UART2_STATUS_RX_EMPTY (0x80)
  //#define UART2_STATUS_TX_EMPTY (0x40)
  r |= !testch() ? UART2_STATUS_RX_EMPTY:0;
  r |= txready   ? UART2_STATUS_TX_EMPTY:0;  // always ready to send in this model.
  return r;
}


void uart64_cbg::stat_report(const char *msg, FILE *fd, bool resetf)
{
  // no stats for now
  if (fd) fprintf(fd, "%s:%s: reads=%i, writes=%i, ins=%i, outs=%i\n", name(), kind(), stats.reads, stats.writes, stats.ins, stats.outs);
  if (resetf) stats.reset();
}

TENOS_KIND_DEFINITION(uart64_cbg)

// eof
