// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_vcd_trace.cpp,v 1.1 2011/06/30 11:10:38 my294 Exp $

/** @file pw_vcd_trace.cpp
 * @brief Tracing facilities for generating power VCD consumption reports.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 */

#include "pw_vcd_trace.h"
#include "pw_vcd_observer.h"
#include "pw_observer_ids.h"
#include "pw_tracing_ids.h"
#include "pw_debug.h"



// ===========================================================================
namespace sc_pwr
{


  // ---------------------------------------------------------------------------
  // Constructor
  pw_vcd_trace_file::pw_vcd_trace_file(const char* p_vcd_name_pt, const char* p_plot_name_pt, const plot_style_t p_plot_style): 
    pw_trace_file(),
    a_plot_style(p_plot_style),
    a_traces(NULL),
    a_vcd_file_name(p_vcd_name_pt),
    a_plot_file_name(p_plot_name_pt)
  {
    if (p_vcd_name_pt) pw_info << "Creating new VCD trace file " << p_vcd_name_pt << ".vcd\n";
    if (p_plot_name_pt) pw_info << " log (gnuplot style) file " << p_plot_name_pt; 
    pw_info << pw_endl;
    //limit = 10000000; 
    a_trocnamerr[0] = ' ';
    a_trocnamerr[1] = 0;
    local_time = 0;    
    start_vcd_files();
    decay_interval = 5 * 1000*1000;
    decay_rate = 0.6;
    n_plot_cols_in_use = 0;
  }


  // Destructor
  pw_vcd_trace_file::~pw_vcd_trace_file(void)
  {
    end_vcd_files();    
    pw_info << "Closed VCD trace file(s)\n";
  }


  // ---------------------------------------------------------------------------
  void pw_vcd_trace_file::trace(sc_core::sc_object& p_obj,
				const std::string& p_str,
				trace_t p_do_children,
				plot_control_t p_plot_control
				)
  {
    pw_info << "Trace " << p_obj.name() << " (" << p_str
            << ") in " << a_vcd_file_name << ".vcd..." << pw_endl;
    
    
    sc_core::sc_attr_base* l_attr_pt = NULL;
    if((l_attr_pt = p_obj.get_attribute(PW_VCD_OBSERVER_ATTRIBUTE_ID))
       == NULL)
      {
        pw_vcd_observer* l_observer_pt =
	  new pw_vcd_observer(p_obj, PW_VCD_OBSERVER_ATTRIBUTE_ID);
	
        // [NOTE] Recursively attach this observer to the power module.
        attach_observer(*l_observer_pt,
                        &p_obj,
			p_do_children==sum_children
			);

        // [NOTE] Add the created observer to the attribute list of the traced SystemC object.
        l_attr_pt = l_observer_pt;
        if (not p_obj.add_attribute(*l_attr_pt))
        {
            SC_REPORT_FATAL(PW_OBSERVER_REDEFINITION_TYPE_,
                            PW_OBSERVER_REDEFINITION_MSG_);
        }
    }
    pw_assert(l_attr_pt != NULL  && "Incoherent NULL pointer.");

    try
    {
        pw_vcd_observer& l_observer =
            dynamic_cast< pw_vcd_observer& >(*l_attr_pt);
        l_observer.trace(this, p_str, p_plot_control);
    }
    catch(std::bad_cast)
    {
        SC_REPORT_ERROR(PW_VCD_OBSERVER_UNAVAILABLE_TYPE_,
                        PW_VCD_OBSERVER_UNAVAILABLE_MSG_);
    }
    if (p_do_children == also_trace_children) this->trace_all_below(&p_obj, p_do_children, p_plot_control);
}


  // ---------------------------------------------------------------------------
  void pw_vcd_trace_file::set_time_unit(double p_value, sc_core::sc_time_unit p_unit)
  { 
    // TODO
  }
  

  // ---------------------------------------------------------------------------
  void pw_vcd_trace_file::write_comment(const std::string& p_str)
  {
    if (a_plot_fd) fprintf(a_plot_fd, "# %s\n", p_str.c_str());
    if (a_vcd_fd) fprintf(a_plot_fd, "$comment\n%s\n$end comment\n", p_str.c_str());
  }



  // ---------------------------------------------------------------------------
  pw_trace_file* pw_create_vcd_trace_file(const char* p_vcd_name_pt, const char* p_plot_name_pt, plot_style_t p_plot_style)
  {
    pw_trace_file *r= new pw_vcd_trace_file(p_vcd_name_pt, p_plot_name_pt, p_plot_style);
    return r;
  }


  // ---------------------------------------------------------------------------
  void pw_close_vcd_trace_file(pw_trace_file *pt)
  {
    pw_vcd_trace_file *pt1 = dynamic_cast<pw_vcd_trace_file *>(pt);
    if (pt1) pt1->end_vcd_files();
  }

  // ---------------------------------------------------------------------------

  pw_vcd_trace_file::traced_variable::traced_variable(pw_vcd_trace_file *parent, const std::string name, int p_plot_col): // constructor
    next(0),
    a_plot_col(p_plot_col),
    name(name),
    parent(parent)
  {
    vale = 0.0;
    lasttime = 0;
    vcd_string = parent->inc_trocnamer();
  }

  int pw_vcd_trace_file::alloc_next_plot_column(std::string name)
  {
    int ans = -1;
    if (n_plot_cols_in_use >= PW_MAX_PLOT_COLS) return ans;
    ans = n_plot_cols_in_use ++;
    Plot_cols[ans].name = name;
    return ans;
  }


  pw_vcd_trace_file::traced_variable *pw_vcd_trace_file::add_trace(int acct_, pw_energy startingvale_, const std::string name, int p_plot_col)
  {
    if (warmed)
      {
	std::cout << "Error : added trace after start\n"; // TODO syscerror
      }
    traced_variable *x = new traced_variable(this, name, p_plot_col);
    // Append to end of list because we have to print out in ascending column number.
    traced_variable **xp = &a_traces;
    while (*xp) xp = &((*xp)->next);
    *xp = x;
    return x;
  }

  // ---------------------------------------------------------------------------
  // Routine to generate the next vcd string in sequence.
  char *pw_vcd_trace_file::inc_trocnamer()
    {
      char *a = a_trocnamerr;
      *a += 1;
      while (*a == (char) 127)
	{
	  *a = (char) 33;
	  a++;
	  if (*a == (char) 0) *a = ' ';
	  *a += 1;
      }
      a++;
      if (*a == ' ') *a = (char) 0;
      return strdup(a_trocnamerr);
    }



  // TODO yupdate flush

  void pw_vcd_trace_file::rdump(traced_variable *trace, float vale)
  {
    fprintf(a_vcd_fd, "r%f %s\n", vale, trace->vcd_string);
#if 0
    if (limit -- < 0) 
      {
	std::cout << "power vcd " << a_vcd_file_name << " truncated at " << sc_time_stamp() << "\n";
	exit(1);
      }
#endif
  }


  void pw_vcd_trace_file::start_vcd_files()
  {
    /* It is best to have a single, top-level scope in the input signals.
    ** In that case, the 'toplevel' manufactured here never appears. */
    
    if (a_plot_file_name.size()) 
      { 
	a_plot_fd = fopen(a_plot_file_name.c_str(), "w");       /* Value Change Dump (vcd) file */
	if (!a_plot_fd) fprintf(stderr, "Failed to open plot (gnuplot style) file: %s\n", a_plot_file_name.c_str());
      }
    else a_plot_fd = 0;

    /* Reset the VCD name */
    strcpy(a_trocnamerr, " ");
    
    if (a_vcd_file_name.size())
      {
	a_vcd_fd = fopen(a_vcd_file_name.c_str(), "w");       /* Value Change Dump (vcd) file */
	if (!a_vcd_fd)
	  {
	    fprintf(stderr, "Failed to open vcd tracing file: %s\n", a_vcd_file_name.c_str());
	    return;
	  }
	osci_code_for_timescale();
      }
    else a_vcd_fd = 0;
  }

  int pw_vcd_trace_file::findscope(char *scope, const char *name)
  // Find the name's prefix, up to the last '.'.  return the length of the prefix, +1 for the dot.
  {
    const char *a = name + strlen(name);
    int scopelen;
    while (a > name && a[-1] != '.') a--;
    if (a == name) /* no scope found */
      {
	scope[0] = 0;
	return 0;
      }
    else
      {
	scopelen = a - name; /* includes the . */
	memcpy(scope, name, scopelen);
	scope[scopelen-1] = 0; /* remove the . */
	return scopelen;
      }
  }


  void pw_vcd_trace_file::warmup_vcd_files()
  {
    warmed = true;
    if (a_plot_fd)
      {
	fprintf(a_plot_fd, "# TLM_POWER plot file: colums mapping now follows:\n");
	for (int i=0; i<n_plot_cols_in_use; i++)
	  fprintf(a_plot_fd, "# %i   %s\n", i, Plot_cols[i].name.c_str());
      }
    if (!a_vcd_fd) return;
    char existing_scope[2048];
    char scope[2048];
    existing_scope[0] = 0;
    const char *default_scope = "toplevel";
    for (traced_variable *t = a_traces; t; t= t->next)
      {
	int scopelen = findscope(scope, t->name.c_str());
	assert(strlen(scope) < sizeof(scope));
	if (scope[0] == 0) strcpy(scope, default_scope);
	if (!strcmp(scope, existing_scope))
	  {
	    fprintf(a_vcd_fd, "$scope module %s $end\n", scope);
	    strcpy(existing_scope, scope);
	  }
	const char *x = t->name.c_str();
	pw_debug << "logging scope=" << scope << "x=" << x << "; len=" << scopelen << std::endl;
	char line[1024];
	snprintf(line, 1024, "%s", x + scopelen);
	char *p = line;
	while (*p) { if (isspace(*p)) *p = '_'; p++; }
	fprintf(a_vcd_fd, "$var real 32 %s %s $end\n", t->vcd_string, line);
      }
    fprintf(a_vcd_fd, "$enddefinitions $end\n");
    
    fprintf(a_vcd_fd, "$dumpvars\n");
    for (traced_variable *t = a_traces; t; t= t->next) // initial values
      {
	rdump(t, 0.0);
	//t->last_vale = t->vale;
      }
    fprintf(a_vcd_fd, "$end\n"); // end $dumpvars
  }


  void pw_vcd_trace_file::traced_variable::log(sc_time nt, pw_energy e)
  {
    sc_dt::uint64 n = nt.value();
    if (!dirty()) lasttime = n;
    vale += e.to_double();
    if (dirty() && n > lasttime) parent->yupdate(n);
  };

  // We can plot average power over the last interval, or exponentially decaying plots of various granularities.
  float pw_vcd_trace_file::traced_variable::compute(sc_dt::uint64 nt)
  {
    float ov;
    if (parent->a_plot_style == PlotStyleDecaying)
      {
	ov = vale;
	vale = vale * parent->decay_rate;
      }
    else if (parent->a_plot_style == PlotStyleAveraging)
      {
	// Power is energy logged last time divided by time since last time.
	sc_dt::uint64 deltat = nt - lasttime;
	if (deltat >= 0)
	  {
	    ov = vale / float(deltat);
	    vale = 0;
	  }
      }
    else assert(0);
    lasttime = nt;
    return ov;	
  }

  // Next time request: report when this monitor would prefer to update its display.
  sc_dt::uint64 pw_vcd_trace_file::traced_variable::next_time_request(sc_dt::uint64 nt)
  {
    return lasttime + parent->decay_interval;
  }


  // Give creation data and timescale
  void pw_vcd_trace_file::osci_code_for_timescale()
  {
    // The code in this method is lifted from the OSCI SystemC vcd tracer.
    //date:
    time_t long_time;
    time(&long_time);
    struct tm* p_tm;
    p_tm = localtime(&long_time);
    char buf[199];
    strftime(buf, 199, "%b %d, %Y       %H:%M:%S", p_tm);
    std::fprintf(a_vcd_fd, "$date\n     %s\n$end\n\n", buf);

    //timescale:
    static struct SC_TIMESCALE_TO_TEXT {
        double       unit;
        const char*  text;
    } timescale_to_text [] = {
        { sc_time(1, SC_FS).to_seconds(), "1 fs" },
        { sc_time(10, SC_FS).to_seconds(), "10 fs" },
        { sc_time(100, SC_FS).to_seconds(),"100 fs" },
        { sc_time(1, SC_PS).to_seconds(),  "1 ps" },
        { sc_time(10, SC_PS).to_seconds(), "10 ps" },
        { sc_time(100, SC_PS).to_seconds(),"100 ps" },
        { sc_time(1, SC_NS).to_seconds(),  "1 ns" },
        { sc_time(10, SC_NS).to_seconds(), "10 ns" },
        { sc_time(100, SC_NS).to_seconds(),"100 ns" },
        { sc_time(1, SC_US).to_seconds(),  "1 us" },
        { sc_time(10, SC_US).to_seconds(), "10 us" },
        { sc_time(100, SC_US).to_seconds(),"100 us" },
        { sc_time(1, SC_MS).to_seconds(),  "1 ms" },
        { sc_time(10, SC_MS).to_seconds(), "10 ms" },
        { sc_time(100, SC_MS).to_seconds(),"100 ms" },
        { sc_time(1, SC_SEC).to_seconds(),  "1 sec" },
        { sc_time(10, SC_SEC).to_seconds(), "10 sec" },
        { sc_time(100, SC_SEC).to_seconds(),"100 sec" }
    };
    static int timescale_to_text_n =
        sizeof(timescale_to_text)/sizeof(SC_TIMESCALE_TO_TEXT);

    double timescale_unit = sc_get_time_resolution().to_seconds();
    for ( int time_i = 0; time_i < timescale_to_text_n; time_i++ )
    {
        if (timescale_unit == timescale_to_text[time_i].unit)
        {
            std::fprintf(a_vcd_fd, "$timescale\n     %s\n$end\n\n",
                timescale_to_text[time_i].text);
            break;
        }
    }
  }

  void pw_vcd_trace_file::yupdate(sc_dt::uint64 n)
  {
    if (!warmed) warmup_vcd_files();
    while (local_time < n)
      {
	sc_dt::uint64 min = 0;
	bool first = true;
	for (traced_variable *t = a_traces; t; t= t->next)
	  {
	    sc_dt::uint64 pref = t->next_time_request(local_time);
	    if (!t->dirty()) continue;
	    if (first || pref < min) { first = false; min = pref; }
          }
	if (first) break; // Nothing to print.

	assert(local_time < min);//need to make progress and cannot rewrite history
	local_time = min;
	if (a_vcd_fd) fprintf(a_vcd_fd, "#%llu\n", local_time); // in current time units

	bool pf = false; int pfv = -1; 	    float pfs = 0.0;
        for (traced_variable *t = a_traces; t; t= t->next)
	  {
	    // compute() is a side-effecting call, so do once, but only want to enter in VCD trace if dirty (ie non-trivial).
	    // TODO - sum all accounts
	    bool was_dirty = t->dirty();
	    if (t->a_plot_col >= 0 && a_plot_fd)
	      {
		float vale =  t->compute(min);
		if (!pf)
		  { fprintf(a_plot_fd, "%llu", local_time); // Print plot time stamp if first column.
		    pf = true;
		  }
		if (t->a_plot_col >0 && t->a_plot_col < pfv) assert(0); // Need to always be in ascending col order.
		if (t->a_plot_col == pfv) pfs += vale;//If same as current totaliser then accumulate.
		else if (t->a_plot_col > pfv)
		  {
		    fprintf(a_plot_fd, "\t%f", pfs);//Move on to next column.
		    pfs = vale;
		    pfv = t->a_plot_col;
		  }
		if (was_dirty && a_vcd_fd) rdump(t,vale);
	      }
	    else if (was_dirty) rdump(t, t->compute(min));
	  }
	if (pfv >= 0) fprintf(a_plot_fd, "\t%f", pfs);
	if (pf) fprintf(a_plot_fd, "\n");
      }
  }

  bool pw_vcd_trace_file::traced_variable::dirty() 
  { return vale > 0.01; // TODO make tidier.
  }

  void pw_vcd_trace_file::end_vcd_files()
  {
    if (a_plot_fd != stdout && a_plot_fd) 
      {
	pw_info <<  "Plot data written to file " << a_plot_file_name << "\n";
	fprintf(a_plot_fd, "#eof\n");
	fclose(a_plot_fd); 
	a_plot_fd = 0;
      }
    if (a_vcd_fd != stdout && a_vcd_fd) 
      {
	pw_info <<  "VCD written to file " << a_vcd_file_name << "\n";
	fprintf(a_vcd_fd, "$comment eof\n$end comment\n");
	fclose(a_vcd_fd); 
	a_vcd_fd = 0;
      }
  }



} // namespace sc_pwr
