// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_txt_trace.cpp,v 1.2 2011/07/25 15:32:45 my294 Exp $

/** @file pw_txt_trace.cpp
 * @brief Tracing facilities for generating statistics on power consumption.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 */

#include <ios>
#include <ctime>
#include <cerrno>
#include <string>
#include <iomanip>
#include <utility>
#include <cstring>
#include <fstream>
#include <sstream>
#include <algorithm>
#include <iostream>

#include "pw_txt_trace.h"
#include "pw_observer_ids.h"
#include "pw_tracing_ids.h"
#include "pw_power.h"
#include "pw_energy.h"
#include "pw_physical_operators.h"
#include "pw_debug.h"
#include "pw_common.h"



// ===========================================================================
namespace
{


// ---------------------------------------------------------------------------
// GLOBAL CONSTANT
static const unsigned LINE_SIZE = 80;
static const unsigned DATE_SIZE = 61;
static const unsigned NAME_SIZE = 24;
static const unsigned ENERGY_PRECISION = 9;
static const unsigned POWER_PRECISION = 9;
static const unsigned RATIO_PRECISION = 2;
static const unsigned GLOBAL_RATIO_PRECISION = 3;
static const unsigned ENERGY_SIZE = 18;
static const unsigned POWER_SIZE = 18;
static const unsigned RATIO_SIZE = 6;
static const unsigned GLOBAL_RATIO_SIZE = 26;


// ---------------------------------------------------------------------------
//! Return the formatted local time.
std::string formatted_date(void)
{
    const char l_date_format[] = "%H:%M:%S -- %d/%m/%Y ";
    char l_date[100];
    std::time_t l_time = std::time(NULL);
    struct tm* l_localtime = std::localtime(&l_time);

    if(l_localtime == NULL)
    {
        std::ostringstream l_msg;
        l_msg << sc_pwr::PW_TXT_TIME_ERROR_MSG_
              << std::strerror(errno) << std::endl;
        SC_REPORT_ERROR(sc_pwr::PW_TXT_TIME_ERROR_TYPE_,
                        l_msg.str().c_str());
        return "undef";
    }

    if(strftime(l_date, sizeof(l_date), l_date_format, l_localtime) == 0)
    {
        std::ostringstream l_msg;
        l_msg << sc_pwr::PW_TXT_TIME_ERROR_MSG_
              << std::strerror(errno) << std::endl;
        SC_REPORT_ERROR(sc_pwr::PW_TXT_TIME_ERROR_TYPE_,
                        l_msg.str().c_str());
        return "undef";
    }

    return std::string(l_date) ;

}


// ---------------------------------------------------------------------------
//! Return the formatted name.
std::string formatted_name(const std::string& p_name)
{
    std::string l_name = p_name;
    if(l_name.length() > NAME_SIZE)
    {
      // If too long, put some dots, dots at the start may be more useful?
      const std::string l_pading("...");
      l_name.erase(0, l_name.length() - (NAME_SIZE-l_pading.length()));
      l_name = l_pading + l_name;
    }
    
    std::ostringstream l_result;
    l_result << std::left << std::setw(NAME_SIZE) << std::setfill(' ')
             << l_name;

    return l_result.str();
}


// ---------------------------------------------------------------------------
std::string formatted_energy_line(const std::string& p_name,
				  bool l_summed,
				  int p_energies,
				  const sc_pwr::pw_energy p_Energies[],
				  sc_pwr::pw_energy e_gtotal
				  )
{
    // Create name column.
    std::ostringstream l_line;
    l_line << "| " << formatted_name(p_name + (l_summed ? "++":"")) << " | ";

    sc_pwr::pw_energy l_total_energy = e_gtotal;

    for (int l_c = 0; l_c < p_energies; l_c++)
      {
	sc_pwr::pw_energy l_e = p_Energies[l_c];
	//std::cout << p_name << " handling energy " <<  l_e << "\n";
	if(false && l_e == sc_pwr::PW_ZERO_ENERGY)
	  {
	    l_line <<
	      "                          0J |"
	      "                          0J |";
	  }
	else
	  {
	    std::ios_base::fmtflags l_flags = l_line.flags();
	    l_line << std::setprecision(ENERGY_PRECISION)
		   << std::right << std::setw(ENERGY_SIZE) << std::setfill(' ')
		   << l_e.to_joules() << "J "
		   << std::fixed << std::setprecision(RATIO_PRECISION)
		   << std::right << std::setw(RATIO_SIZE) << std::setfill(' ')
		   << 100 * l_e.to_double() / l_total_energy.to_double() 
		   << "% | ";
	    l_line.flags(l_flags);
	  }
      }
    l_line << std::endl;
    return l_line.str();
}


// ---------------------------------------------------------------------------
std::string formatted_power_line(const std::string& p_name,
				 bool l_summed,
				 int p_energies,
				 const sc_pwr::pw_energy p_Energies[],
				 sc_pwr::pw_power p_Powers[],
				 const sc_core::sc_time& p_last_update,
				 sc_pwr::pw_power p_gtotal)
{
    // Create name column.
    std::ostringstream l_line;
    l_line << "| " << formatted_name(p_name + (l_summed ? "++":"")) << " | ";


    //sc_pwr::pw_energy l_total_energy = p_gtotal;
    //sc_pwr::pw_energy l_total_energy = sc_pwr::PW_ZERO_ENERGY;
    //for (int l_c = 0; l_c < p_energies; l_c++) l_total_energy += p_Energies[l_c];

    for (int l_c = 0; l_c < p_energies; l_c++)
      {
	sc_pwr::pw_energy l_e = p_Energies[l_c];
	const sc_pwr::pw_power l_power =
	  (p_last_update == sc_core::SC_ZERO_TIME)?	\
	  (sc_pwr::PW_ZERO_POWER):			\
	  (l_e / p_last_update);
	p_Powers[l_c] = l_power;
	//std::cout << p_name << " divided " <<  l_e << " " << l_e.to_double() << " by " << p_last_update << " to get " << l_power << " " << l_power.to_double() << "\n";
      }

    sc_pwr::pw_power l_total_power = sc_pwr::PW_ZERO_POWER;
    for (int l_c = 0; l_c < p_energies; l_c++) l_total_power += p_Powers[l_c];

    for (int l_c = 0; l_c < p_energies; l_c++)
      {
	if(false && l_total_power == sc_pwr::PW_ZERO_POWER)
	  {
	    l_line <<
	      "                        0 W |"
	      "                         0 W |";
	  }
	else
	  {
	    sc_pwr::pw_power l_power = p_Powers[l_c];
	    std::ios_base::fmtflags l_flags = l_line.flags();
	    l_line << std::setprecision(POWER_PRECISION)
		   << std::right << std::setw(POWER_SIZE) << std::setfill(' ')
		   << l_power.to_watts() << "W "
		   << std::fixed << std::setprecision(RATIO_PRECISION)
		   << std::right << std::setw(RATIO_SIZE) << std::setfill(' ')
		   << 100 * l_power.to_double()
              / l_total_power.to_double() << "% | ";
	    l_line.flags(l_flags);
	  }
      }
    l_line << std::endl;    
    return l_line.str();
}


// ---------------------------------------------------------------------------
std::string formatted_ratio_line(const std::string& p_name,
                                const double p_dynamic_ratio,
                                const double p_static_ratio)
{
    // Create name column.
    std::ostringstream l_line;
    l_line << "| " << formatted_name(p_name) << " | ";

    // Dump Dynamic ratio column
    std::ios_base::fmtflags l_flags = l_line.flags();
    l_line << std::fixed << std::setprecision(GLOBAL_RATIO_PRECISION)
           << std::right << std::setw(GLOBAL_RATIO_SIZE)
           << std::setfill(' ') << p_dynamic_ratio << "% | ";
    l_line.flags(l_flags);

    // Dump Static ratio column
    l_line << std::fixed << std::setprecision(GLOBAL_RATIO_PRECISION)
           << std::right << std::setw(GLOBAL_RATIO_SIZE)
           << std::setfill(' ') << p_static_ratio << "% |" << std::endl;

    return l_line.str();
}



} // ANONYMOUS NAMESPACE (replace static declaration)


// ===========================================================================
namespace sc_pwr
{


  // ---------------------------------------------------------------------------
  // Constructor
  pw_txt_trace_file::pw_txt_trace_file(const char* p_name_pt):
    pw_observer_if(),
    pw_trace_file(),
    a_file((std::string(p_name_pt)+".txt").c_str(),
           std::ios_base::out | std::ios_base::trunc),
    a_file_name(p_name_pt),
    a_observers_pt(),
    //a_observers_name(),
    //a_observers_energy(),
    a_energy_stat(""),
    a_power_stat(""),
    a_observer_stat(),
    a_comment("")
{
    pw_info << "Creating new textual trace file " << p_name_pt
            << ".txt..." << pw_endl;

    if(not a_file)
    {
        std::ostringstream l_msg;
        l_msg << PW_TXT_TRACE_FILE_NAME_MSG_
              << " (" << p_name_pt << ") : "
              << std::strerror(errno) << std::endl;
        SC_REPORT_ERROR(PW_TXT_TRACE_FILE_NAME_TYPE_,
                        l_msg.str().c_str());
        return;
    }

    a_file <<
"################################################################################\n"
"#                     TLM POWER3 (Univ Cambridge, UK)                          #\n"
"#                                                                              #\n"
"#                Statistics file: energy/power consumption.                    #\n"
"# ---------------------------------------------------------------------------- #\n"
"# For more information see the TLM POWER3 manual pdf.                   p      #\n"
"# ---------------------------------------------------------------------------- #\n"
"# Creation Date: "
          << std::left << std::setw(DATE_SIZE) << std::setfill(' ')
          << formatted_date() << " #\n"
"################################################################################\n\n\n";

    a_file.flush();
}


  // ---------------------------------------------------------------------------
  pw_txt_trace_file::~pw_txt_trace_file(void)
  {
    pw_debug  << "Destructor invoked\n";
    pw_info << "Close textual trace file " << a_file_name
            << ".txt" << pw_endl;

    a_file << "Title: " << simulation_title() << std::endl;
    // Print comment - specific to this trace.
    if(a_comment != "")
    {
    a_file <<
"###                                COMMENT                                   ###\n"
           << a_comment << std::endl <<
"###                                COMMENT                                   ###\n\n\n";

    }

    a_hline_sep = "+--------------------------+";

    a_energy_stat << "| " + formatted_name("MODULE  NAME") + " |";
    a_power_stat  << "| " + formatted_name("MODULE  NAME") + " |";

    for (int a=0; a<PW_ACCT_MAX_NO_ACCOUNTS; a++) 
      if (pw_stat_observer_base::global_inuse(a)) 
	{
	  std::string n =  pw_stat_observer_base::get_global_name(a);
	  a_energy_stat << std::right << std::setw(4+ENERGY_SIZE+RATIO_SIZE) << std::setfill(' ') << (n + " ENERGY") << " |";
	  a_power_stat << std::right << std::setw(4+POWER_SIZE+RATIO_SIZE) << std::setfill(' ') << (n + " POWER") << " |";
	  a_hline_sep += "-----------------------------+";
	}

    a_energy_stat << '\n';
    a_power_stat << '\n';
    a_hline_sep += '\n';
    
    a_energy_stat << a_hline_sep;
    a_power_stat << a_hline_sep;


    // Some useful constants.
    const int l_n_accounts = pw_stat_observer_base::global_n_accounts_in_use();
    const sc_pwr::pw_energy l_global_energy = pw_stat_observer_base::get_global_energy();
    const sc_core::sc_time l_last_update = pw_stat_observer_base::get_global_last_update();

    a_do_powers = l_last_update != SC_ZERO_TIME;
    // This is ok provided power was set to zero before end: l_last_update != sc_pwr::sc_time_max();

    if (a_do_powers) pw_debug << "do_powers: Dividing " << l_global_energy << " by " << l_last_update << "\n";
    const sc_pwr::pw_power l_global_power = (a_do_powers) ? l_global_energy / l_last_update: PW_ZERO_POWER;

    pw_info << "Number of global accounts: " << l_n_accounts << pw_endl;

    sc_pwr::pw_energy Energies [PW_ACCT_MAX_NO_ACCOUNTS];
    sc_pwr::pw_power Powers [PW_ACCT_MAX_NO_ACCOUNTS];


    for (int mode=0; mode <3; mode++) // Three types of adding up (two in practice): list in separate parts of the table.
      {
	trace_t m = mode==0? no_children: mode==1 ? sum_children: also_trace_children;
	const char *subtitle = mode==0 ? "Standalone modules:": mode==1 ? "With child components summed:": "SHOULD NOT BE SEEN:";
	// Main table: for each observer - not every observer will have every account.
	for(unsigned l_cpt = 0;
	    l_cpt < a_observers_pt.size();
	    ++l_cpt)
	  {
	    pw_stat_observer_record* l_r = a_observers_pt.at(l_cpt);
	    if(l_r && l_r->m_do_children == m)
	      {
		pw_debug << l_cpt << " " << l_r->m_name << " fixout." << pw_endl;
		pw_debug << "tmp debug main table add " 
			 <<  l_r->m_observer.get_sc_object().name() << pw_endl;
		if (subtitle)
		  {
		    a_energy_stat << subtitle << "\n";
		    a_power_stat  << subtitle << "\n";
		    subtitle = 0;
		  }
		// Dump power information into the a_observer_stat attribute
		dump_observer(l_last_update, *l_r);
#if 0
		a_observers_energy.at(l_cpt) =
		  std::pair< pw_energy,
			     pw_energy>(l_observer_pt->get_energy(0),
					l_observer_pt->get_energy(1));
#endif
	      }
	  }
      }

    //-----------------------------------
    // Global: Dump total energy and close the array energy stat.
    for (int a=0, b=0; a<PW_ACCT_MAX_NO_ACCOUNTS; a++) 
      if (pw_stat_observer_base::global_inuse(a)) 
	{
	  Energies[b++] = pw_stat_observer_base::get_global_energy(a);
	  // const char *n =  + pw_stat_observer_base::get_global_name(a);

	}

    std::string l_msg = "Each line is for a separately-traced subsystem. These lines may be neither disjoint or complete.\n"
      "The TOP LEVEL figure is simply another line in the table that relates to the highest module found.\n";


    a_energy_stat << a_hline_sep;
    a_energy_stat << formatted_energy_line("TOP LEVEL",
					   true,
					   l_n_accounts,
					   Energies,
					   l_global_energy);
    a_energy_stat << a_hline_sep;
    a_energy_stat << l_msg;

    a_energy_stat << "Total energy used: " << l_global_energy.round3sf() << "  (" << l_global_energy << ")\n";

    if (a_do_powers)
      {
	// Dump total power and close the array power stat.
	a_power_stat << a_hline_sep;
	a_power_stat << formatted_power_line("TOP LEVEL",
					     true,
					     l_n_accounts,
					     Energies,
					     Powers,
					     l_last_update,
					     l_global_power);
	a_power_stat << a_hline_sep;
	a_power_stat << l_msg;
	a_power_stat << "Average power used: " << l_global_power.round3sf() << "  (" << l_global_power << ")\n";
      }

    bool l_infinite = sc_time_stamp() == sc_pwr::sc_time_max();
    // Main components now added to the file:
    a_file 
      << "# Simulation duration 1: " << pw_stat_observer_base::get_global_last_update() << (l_infinite ? " INFINITE": "") << std::endl 
      << "# Simulation duration 2: " << sc_time_stamp() << std::endl 
      << std::endl
      << a_hline_sep << a_energy_stat.str() << std::endl << std::endl
      << a_hline_sep << a_power_stat.str() << std::endl << std::endl
      << dump_ratio() << std::endl << std::endl;

    typedef std::list< std::string >::const_iterator list_const_iterator;

#if 0
    // Disable this old code: it copes only with two accounts.
    // Now write each observer's details to the file.

    for(list_const_iterator l_it = a_observer_stat.begin();
        l_it != a_observer_stat.end();
        ++l_it)
        a_file << *l_it;
#endif

    // Now some comma-separated information for quick grep and paste to spreadsheet:
    a_file 
      << "CSV, " 
      << simulation_title() << ", "
      << pw_stat_observer_base::get_global_last_update().to_double() << ", "
      << l_global_energy.to_double() 
      << std::endl;


    a_file << "#EOF\n";
    pw_debug  << "Destructor finished. File written.\n";
}


  // ---------------------------------------------------------------------------
  void pw_txt_trace_file::update(pw_subject& p_subject)
  {
#if 0
    // XXX Store information before destruction of the parameter subject.
    try
      {
        pw_stat_observer_base& l_observer = dynamic_cast< pw_stat_observer_base& >(p_subject);

#if 0
	// Why do all this searching on every update?
        std::vector< pw_stat_observer_record* >::iterator l_it =
	  std::find(a_observers_pt.begin(),
		    a_observers_pt.end(),
		    &l_observer);

        pw_assert(l_it != a_observers_pt.end()
                  and "Oups, observers not recorded in this trace file.");
        const size_t l_pos = std::distance(a_observers_pt.begin(), l_it);
#endif
	
        dump_observer(l_observer);
	
#if 0
        a_observers_energy.at(l_pos) =
	  std::pair< pw_energy, pw_energy>(l_observer.get_energy(0),
					   l_observer.get_energy(1));
#endif
        // XXX Remove the l_observer from the list of the observer to dump.
        *l_it = NULL;
      }
    catch(std::bad_cast)
      {
        pw_assert(false
                  and "Casting problem with the \"Observer\" design pattern");
      }
#endif
  }


  // ---------------------------------------------------------------------------
  void pw_txt_trace_file::trace(sc_core::sc_object& p_obj,
				const std::string& p_str,
				trace_t p_do_children,
				plot_control_t p_plot_control
				)
  {
    pw_info << "Trace txt " << p_obj.name() << " (" << p_str
            << ") with output to " << a_file_name << ".txt.  children mode=" << p_do_children << pw_endl;
    
    trace_t l_mode = p_do_children==also_trace_children?no_children:p_do_children;
    //sc_module *l_mod = dynamic_cast<sc_module *>(&p_obj);
    //pw_assert(l_mod and "trace should be applied to an sc_module");
    pw_stat_observer_base *l_observer_pt = pw_get_observer(&p_obj, l_mode);
    
    // An appropriate observer is now added to this module.
    pw_assert(l_observer_pt != NULL and "Incoherent NULL pointer.");
    
    // An new trace line is now added to this file.
    pw_debug << "Pushing observer " << p_str << "\n";
    a_observers_pt.push_back(new pw_stat_observer_record(l_mode, *l_observer_pt, p_str));
    
    if (p_do_children == also_trace_children) this->trace_all_below(&p_obj, p_do_children);
  }


  // ---------------------------------------------------------------------------
  void pw_txt_trace_file::set_time_unit(double p_value,
					sc_core::sc_time_unit p_unit)
  {
    // Nothing todo
    return;
  }


  // ---------------------------------------------------------------------------
  void pw_txt_trace_file::write_comment(const std::string& p_str)
  {
    a_comment = p_str;
  }


// ---------------------------------------------------------------------------
  void pw_txt_trace_file::dump_observer(sc_time p_eos, pw_stat_observer_record &p_record)
  {
    // Do not print information line for unpowered module.
    if(p_record.m_observer.get_energy() == PW_ZERO_ENERGY) return;
    
    const pw_energy l_global_energy = pw_stat_observer_base::get_global_energy();
    //const pw_energy l_global_power = pw_stat_observer_base::get_global_power();
    const sc_core::sc_time l_last_update = pw_stat_observer_base::get_global_last_update();
    const sc_pwr::pw_power l_global_power = (a_do_powers) ? l_global_energy / l_last_update: PW_ZERO_POWER;
    const int l_n_accounts = p_record.m_observer.global_n_accounts_in_use();
    
    
    bool l_summed = p_record.m_observer.get_att_name() == PW_STAT_OBSERVER_ATTRIBUTE_ID_CHILD_SUMMED;
    
    sc_pwr::pw_energy Energies [PW_ACCT_MAX_NO_ACCOUNTS];
    sc_pwr::pw_power Powers [PW_ACCT_MAX_NO_ACCOUNTS];
    
    for (int a=0, b=0; a<PW_ACCT_MAX_NO_ACCOUNTS; a++) 
      Energies[b++] = (pw_stat_observer_base::global_inuse(a)) ? p_record.m_observer.get_energy(a): PW_ZERO_ENERGY;
    
    if (1) 
    {
      /*sc_time l_last_update = */p_record.m_observer.get_last_updatet(); // Note: must call this before reading energies since some standing power may be folded in.
      //pw_energy e = p_observer.get_energy();
      // Dump energy info
      a_energy_stat <<
	formatted_energy_line(p_record.m_name,
			      l_summed,
			      l_n_accounts,
			      Energies,
			      l_global_energy);
      
      if (a_do_powers)
	{
	  // Dump power info
	  a_power_stat << formatted_power_line(p_record.m_name,
					       l_summed,
					       l_n_accounts,
					       Energies,
					       Powers,
					       p_eos,
					       l_global_power);
	}
    }
    
#if 0 // TODO ?
    try
      {
        //pw_stat_observer& l_observer = dynamic_cast< pw_stat_observer& >(p_record.m_observer);
        dump_observer(l_observer);
      }
    catch(std::bad_cast)
      {
        // Nothing todo
      }
#endif
  }


// ---------------------------------------------------------------------------

// This ratio table gives the ratio out of the global total
// wheras the percentages in the first-printed table are out of ...

// It gives energy ratios.

  std::string pw_txt_trace_file::dump_ratio(void) const
  {
    std::string l_result = std::string("");
# if 0
    const pw_energy l_global_energy =
      pw_stat_observer_base::get_global_energy();
    
    if(l_global_energy == PW_ZERO_ENERGY) return "";
    
    l_result +=
      "+------------------+-----------------------------+-----------------------------+\n"
      "|   MODULE NAME    |        DYNAMIC RATIO        |        STATIC RATIO         |\n" + a_hline_sep;
    
    typedef std::vector< std::pair< pw_energy, pw_energy > >::const_iterator
      const_iterator;
    for(const_iterator l_it = a_observers_energy.begin();
        l_it != a_observers_energy.end();
        ++l_it)
      {
        const size_t l_pos = std::distance(a_observers_energy.begin(),
                                           l_it);
        const std::string& l_name = a_observers_name.at(l_pos);
        const double l_ratio =
	  (l_global_energy == PW_ZERO_ENERGY)?	\
            (0.):\
	  (100 * l_it->second.to_double()
	   / l_global_energy.to_double());

	//        if(l_ratio == 0. ) continue;
	
        l_result +=
	  formatted_ratio_line(l_name,
			       0.0,
			       l_ratio);
	
      }
    
    l_result += a_hline_sep;
#endif
    return l_result;
}



// ---------------------------------------------------------------------------

  pw_trace_file* pw_create_txt_trace_file(const char* p_name_pt)
  {
    pw_trace_file *r = new pw_txt_trace_file(p_name_pt);
    // Doing all_modules must be requested after they have been constructed.
    //if (p_do_children != pw_trace_file::no_children) r->trace_all_below(0);
    return r;
  }


// ---------------------------------------------------------------------------
void pw_close_txt_trace_file(pw_trace_file* p_file_pt)
{
    delete p_file_pt;
}


} // namespace sc_pwr
