// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_power.cpp,v 1.1 2011/06/30 11:10:38 my294 Exp $

/** @file pw_power.cpp
 * @brief Define power data type.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 */

#include <ios>
#include <cmath>
#include <sstream>
#include <iomanip>
#include <iostream>
#include <algorithm>

#include "pw_power.h"
#include "pw_kernel_ids.h"
#include "pw_debug.h"
#include "pw_common.h"



// ===========================================================================
namespace
{


// ---------------------------------------------------------------------------
using sc_dt::uint64;


// ---------------------------------------------------------------------------
//! Convertion table of power unit into string.
const char* g_tab_power_units[] =
{
    "fW",
    "pW",
    "nW",
    "uW",
    "mW",
    "W"
};


// ---------------------------------------------------------------------------
const double g_tab_power_scales[] =
{
    1,       // fw
    1e3,     // pw
    1e6,     // nw
    1e9,     // uw
    1e12,    // mw
    1e15     // w
};


// ---------------------------------------------------------------------------
/**
 * @brief Test if an uint is a power of 10.
 * @param[in] p_val The uint to test.
 * @return true only if p_val is a power of 10.
 */
static bool is_pow10(double p_val)
{
    double l_tmp;
    return std::modf(std::log10(p_val), &l_tmp) == 0.0;
}


// ---------------------------------------------------------------------------
//! Simple rinting function
inline double pw_rint(double p_val)                                                     
{
    return std::floor( p_val + 0.5 );
}

} // ANONYMOUS NAMESPACE (replace static declaration)


// ===========================================================================
namespace sc_pwr
{


// ---------------------------------------------------------------------------
using sc_dt::uint64;


// ---------------------------------------------------------------------------
//! Power resolution in femto watt.
double pw_power::a_resolution_value = g_tab_power_scales[PW_pW];
//! True if a pw_power different of 0 have been constructed.
bool pw_power::a_is_resolution_fixed = false;


// ---------------------------------------------------------------------------
//! ZERO power constant
const pw_power PW_ZERO_POWER;

#define PW_POWER_UNIT_UNDERFLOW_TYPE_ PW_POWER_UNIT_OVERFLOW_TYPE_
#define PW_POWER_UNIT_UNDERFLOW_MSG_  PW_POWER_UNIT_OVERFLOW_MSG_

// ---------------------------------------------------------------------------
pw_power_unit& operator++(pw_power_unit& p_unit) // We do not really need ++ etc on these units!
{
    if(p_unit < PW_fW)
    {
        p_unit = PW_fW;
        SC_REPORT_WARNING(PW_POWER_UNIT_UNDERFLOW_TYPE_,
                          PW_POWER_UNIT_UNDERFLOW_MSG_);
    }
    if(p_unit >= PW_WATT)
    {
        p_unit = PW_WATT;
        SC_REPORT_WARNING(PW_POWER_UNIT_OVERFLOW_TYPE_,
                          PW_POWER_UNIT_OVERFLOW_MSG_);
        return p_unit;
    }

    unsigned l_uint = static_cast< unsigned >(p_unit);
    p_unit = static_cast< pw_power_unit >(++l_uint);

    pw_assert(p_unit > PW_fW and p_unit <= PW_WATT \
              and "Unexpected overflow!!!");

    return p_unit;
}


// ---------------------------------------------------------------------------
pw_power_unit operator++(pw_power_unit& p_unit,
                         int)
{
    if(p_unit < PW_fW)
    {
        p_unit = PW_fW;
        SC_REPORT_WARNING(PW_POWER_UNIT_UNDERFLOW_TYPE_,
                          PW_POWER_UNIT_UNDERFLOW_MSG_);
    }
    if(p_unit >= PW_WATT)
    {
        p_unit = PW_WATT;
        SC_REPORT_WARNING(PW_POWER_UNIT_OVERFLOW_TYPE_,
                          PW_POWER_UNIT_OVERFLOW_MSG_);
        return p_unit;
    }

    unsigned l_uint = static_cast< unsigned >(p_unit);
    pw_power_unit l_unit = static_cast< pw_power_unit >(++l_uint);

    pw_assert(l_unit > PW_fW and l_unit <= PW_WATT \
              and "Unexpected overflow!!!");

    return l_unit;
}


// ---------------------------------------------------------------------------
pw_power_unit& operator--(pw_power_unit& p_unit)
{
    if(p_unit <= PW_fW)
    {
        p_unit = PW_fW;
        SC_REPORT_WARNING(PW_POWER_UNIT_OVERFLOW_TYPE_,
                          PW_POWER_UNIT_OVERFLOW_MSG_);
        return p_unit;
    }
    if(p_unit > PW_WATT)
    {
        p_unit = PW_WATT;
        SC_REPORT_WARNING(PW_POWER_UNIT_OVERFLOW_TYPE_,
                          PW_POWER_UNIT_OVERFLOW_MSG_);
    }

    unsigned l_uint = static_cast< unsigned >(p_unit);
    p_unit = static_cast< pw_power_unit >(--l_uint);

    pw_assert(p_unit >= PW_fW and p_unit < PW_WATT \
              and "Unexpected overflow!!!");

    return p_unit;
}


// ---------------------------------------------------------------------------
pw_power_unit operator--(pw_power_unit& p_unit,
                         int)
{
    if(p_unit <= PW_fW)
    {
        p_unit = PW_fW;
        SC_REPORT_WARNING(PW_POWER_UNIT_UNDERFLOW_TYPE_,
                          PW_POWER_UNIT_UNDERFLOW_MSG_);
        return p_unit;
    }
    if(p_unit > PW_WATT)
    {
        p_unit = PW_WATT;
        SC_REPORT_WARNING(PW_POWER_UNIT_OVERFLOW_TYPE_,
                          PW_POWER_UNIT_OVERFLOW_MSG_);
    }

    unsigned l_uint = static_cast< unsigned >(p_unit);
    pw_power_unit l_unit = static_cast< pw_power_unit >(--l_uint);

    pw_assert(l_unit >= PW_fW and l_unit < PW_WATT \
              and "Unexpected overflow!!!");

    return l_unit;
}


// ---------------------------------------------------------------------------
std::ostream& operator<<(std::ostream& p_os,
                         const pw_power_unit& p_unit)
{
    std::ostream::sentry l_init(p_os);
    if(!l_init)
    {
        SC_REPORT_ERROR(PW_POWER_UNIT_OSTREAM_ERROR_TYPE_,
                        PW_POWER_UNIT_OSTREAM_ERROR_MSG_);
        return p_os;
    }

    if(p_unit < PW_fW or p_unit > PW_WATT)
    {
        SC_REPORT_ERROR(PW_POWER_UNIT_OVERFLOW_TYPE_,
                        PW_POWER_UNIT_OVERFLOW_MSG_);
        return (p_os << "undef");
    }

    return (p_os << g_tab_power_units[p_unit]);
}


// ---------------------------------------------------------------------------
std::istream& operator>>(std::istream& p_is,
                         pw_power_unit& p_unit)
{
    std::istream::sentry l_init(p_is,
                                false);
    if(!l_init)
    {
        SC_REPORT_ERROR(PW_POWER_UNIT_ISTREAM_ERROR_TYPE_,
                        PW_POWER_UNIT_ISTREAM_ERROR_MSG_);
        return p_is;
    }

    std::string l_unit_idf;
    p_is >> l_unit_idf;
    if(not p_is)
    {
        SC_REPORT_ERROR(PW_POWER_UNIT_ISTREAM_ERROR_TYPE_,
                        PW_POWER_UNIT_ISTREAM_ERROR_MSG_);
        return p_is;
    }

    unsigned l_pos = 0;
    const unsigned l_size =
        sizeof(g_tab_power_units) / sizeof(const char*);
    while(l_pos < l_size
          and l_unit_idf != g_tab_power_units[l_pos])
        ++l_pos;

    if(l_pos == l_size)
    {
        p_is.setstate(std::ios_base::failbit);
        SC_REPORT_ERROR(PW_POWER_UNIT_ISTREAM_ERROR_TYPE_,
                        PW_POWER_UNIT_ISTREAM_ERROR_MSG_);
        return p_is;
    }

    p_unit = static_cast< pw_power_unit >(l_pos);
    return p_is;
}


// ---------------------------------------------------------------------------
pw_power::pw_power(void):
    a_value(0)
{ }


// ---------------------------------------------------------------------------

  void breakpoint()
  {
    static volatile int a=0;
    a++;
  }

pw_power::pw_power(double p_value,
                   pw_power_unit p_unit):
    a_value(0)
{
    if(p_unit < PW_fW or p_unit > PW_WATT)
    {
        SC_REPORT_ERROR(PW_POWER_UNIT_OVERFLOW_TYPE_,
                        PW_POWER_UNIT_OVERFLOW_MSG_);
        return;
    }

    if(p_value < 0.0)
    {
        a_value = 0;
        SC_REPORT_ERROR(PW_POWER_NEGATIVE_TYPE_,
                        PW_POWER_NEGATIVE_MSG_);
        return;
    }

    if (p_value == 0) return;

    double l_scaling_factor =
        g_tab_power_scales[p_unit] / a_resolution_value;
    double l_value = p_value * l_scaling_factor;

    /* XXX conversion problem when pw_rint(l_value) == PW_UINT64_MAX
     * We have to manage when these two values are equal.
     */
    if (pw_rint(l_value) >= (static_cast<double>(PW_UINT64_MAX)))
    {
      if (pw_rint(l_value) > (static_cast<double>(PW_UINT64_MAX)))
        {
	  pw_debug << "constructor overflow " << p_value << " " << p_unit << "\n";
	  breakpoint();
	  SC_REPORT_WARNING(PW_POWER_OVERFLOW_TYPE_,
			    PW_POWER_OVERFLOW_MSG_);
        }
      a_value = PW_UINT64_MAX;
      return;
    }

    /* XXX DEPRECATED XXX
     * Arithmetic operation with double are not enough accurate for checking 
     * the possible loss of information.
    double l_tmp;
    if(std::modf(l_value, &l_tmp) != 0.0)
    {
        SC_REPORT_WARNING(PW_POWER_OVERFLOW_TYPE_,
                          PW_POWER_OVERFLOW_MSG_);
    }
    XXX DEPRECATED XXX*/

    a_value = static_cast<uint64>(pw_rint(l_value));
    if(a_value != 0)
    {
        if(not pw_power::a_is_resolution_fixed)
            pw_info << "Resolution Fixed to: " << pw_power::a_resolution_value
                    << " Femto Watt" << pw_endl;
        a_is_resolution_fixed = true;
    }
}


// ---------------------------------------------------------------------------
pw_power::pw_power(const pw_power& p_power):
    a_value(p_power.a_value)
{ }


// ---------------------------------------------------------------------------
pw_power& pw_power::operator=(const pw_power& p_power)
{
    a_value = p_power.a_value;
    return *this;
}


// ---------------------------------------------------------------------------
pw_power& pw_power::operator+=(const pw_power& p_power)
{
    if(PW_UINT64_MAX - p_power.a_value < a_value)
    {
        a_value = PW_UINT64_MAX;
	pw_debug << "+= overflow1";
        SC_REPORT_WARNING(PW_POWER_OVERFLOW_TYPE_,
                          PW_POWER_OVERFLOW_MSG_);
    }
    else
        a_value += p_power.a_value;

    return *this;
}


// ---------------------------------------------------------------------------
pw_power& pw_power::operator-=(const pw_power& p_power)
{
    if(p_power.a_value > a_value)
    {
        a_value = 0;
        SC_REPORT_WARNING(PW_POWER_UNDERFLOW_TYPE_,
                          PW_POWER_UNDERFLOW_MSG_);
    }
    else
        a_value -= p_power.a_value;

    return *this;
}


// ---------------------------------------------------------------------------
pw_power& pw_power::operator*=(double p_value)
{
    if(p_value == 0 or a_value == 0)
        a_value = 0;
    else
    {
        double l_tmp = to_double() * p_value;
        a_value = static_cast<uint64>(pw_rint(l_tmp));
        if(std::fabs(l_tmp - static_cast< double >(a_value)) > 1.)
        {
            a_value = PW_UINT64_MAX;
	    pw_debug << "*= overflow";
            SC_REPORT_WARNING(PW_POWER_OVERFLOW_TYPE_,
                              PW_POWER_OVERFLOW_MSG_);
        }
    }

    return *this;
}


// ---------------------------------------------------------------------------
pw_power& pw_power::operator/=(double p_value)
{
    if(p_value == 0.)
    {
        a_value = PW_UINT64_MAX;
        SC_REPORT_ERROR(PW_POWER_DIVISION_BY_ZERO_TYPE_,
                        PW_POWER_DIVISION_BY_ZERO_MSG_);
        return *this;
    }

    double l_tmp = to_double() / static_cast< double >(p_value);
    a_value = static_cast< uint64 >(pw_rint(l_tmp));

    return *this;
}


// ---------------------------------------------------------------------------
pw_power& pw_power::operator++(void)
{
    if(PW_UINT64_MAX == a_value)
    {
	pw_debug << "++2 overflow";
        SC_REPORT_WARNING(PW_POWER_OVERFLOW_TYPE_,
                          PW_POWER_OVERFLOW_MSG_);
    }
    else
        ++a_value;

    return *this;
}


// ---------------------------------------------------------------------------
pw_power pw_power::operator++(int)
{
    if(PW_UINT64_MAX == a_value)
    {
	pw_debug << "++ overflow2";
        SC_REPORT_WARNING(PW_POWER_OVERFLOW_TYPE_,
                          PW_POWER_OVERFLOW_MSG_);
        return *this;
    }

    pw_power l_res;
    l_res.a_value = a_value;
    ++a_value;
    return l_res;
}


// ---------------------------------------------------------------------------
pw_power& pw_power::operator--(void)
{
    if(0 == a_value)
    {
        SC_REPORT_WARNING(PW_POWER_UNDERFLOW_TYPE_,
                          PW_POWER_UNDERFLOW_MSG_);
    }
    else
        --a_value;

    return *this;
}


// ---------------------------------------------------------------------------
pw_power pw_power::operator--(int)
{
    if(0 == a_value)
    {
        SC_REPORT_WARNING(PW_POWER_UNDERFLOW_TYPE_,
                          PW_POWER_UNDERFLOW_MSG_);
        return *this;
    }

    pw_power l_tmp;
    l_tmp.a_value = a_value;
    --a_value;
    return l_tmp;
}


// ---------------------------------------------------------------------------
const std::string pw_power::to_string(void) const
{
    if(a_value == 0)
    {
        std::ostringstream l_os;
        l_os << "0" << " " << PW_WATT;
        return l_os.str();
    }

    pw_assert(pw_power::a_resolution_value <=
           (static_cast<double>(PW_UINT64_MAX))
           and "Incoherent power resolution value.");

    uint64 l_resolution_value =
        static_cast<uint64>(pw_power::a_resolution_value);
    unsigned l_zero_cpt = 0;
    while((l_resolution_value % 10) == 0)
    {
        ++l_zero_cpt;
        l_resolution_value /= 10;
    }
    uint64 l_value = a_value;
    while((l_value % 10) == 0)
    {
        ++l_zero_cpt;
        l_value /= 10;
    }
    std::ostringstream l_os;
    l_os << l_value;

    pw_power_unit l_unit = PW_fW;
    while(l_zero_cpt >= 3 and l_unit < PW_WATT)
    {
        ++l_unit;
        l_zero_cpt -= 3;
    }
    for(unsigned l_cpt = 0;
        l_cpt < l_zero_cpt;
        ++l_cpt)
        l_os << "0";
    l_os << " " << g_tab_power_units[l_unit];

    return l_os.str();
}


// ---------------------------------------------------------------------------
uint64 pw_power::value(void) const
{
    return this->a_value;
}


// ---------------------------------------------------------------------------
double pw_power::to_double(void) const
{
    return static_cast< double >(this->a_value);
}


// ---------------------------------------------------------------------------
double pw_power::to_watts(void) const
{
    return static_cast< double >(this->a_value) \
        * (pw_power::a_resolution_value / g_tab_power_scales[PW_WATT]);
}


// ---------------------------------------------------------------------------
pw_power pw_power::max(void)
{
    pw_power l_power(pw_get_power_resolution());
    l_power.a_value = PW_UINT64_MAX;
    return l_power;
}


// ---------------------------------------------------------------------------
bool operator==(const pw_power& p_power1,
                const pw_power& p_power2)
{
    return p_power1.a_value == p_power2.a_value;
}


// ---------------------------------------------------------------------------
bool operator!=(const pw_power& p_power1,
                const pw_power& p_power2)
{
    return p_power1.a_value != p_power2.a_value;
}


// ---------------------------------------------------------------------------
bool operator<(const pw_power& p_power1,
               const pw_power& p_power2)
{
    return p_power1.a_value < p_power2.a_value;
}


// ---------------------------------------------------------------------------
bool operator>(const pw_power& p_power1,
               const pw_power& p_power2)
{
    return p_power1.a_value > p_power2.a_value;
}


// ---------------------------------------------------------------------------
bool operator<=(const pw_power& p_power1,
                const pw_power& p_power2)
{
    return p_power1.a_value <= p_power2.a_value;
}


// ---------------------------------------------------------------------------
bool operator>=(const pw_power& p_power1,
                const pw_power& p_power2)
{
    return p_power1.a_value >= p_power2.a_value;
}


// ---------------------------------------------------------------------------
pw_power operator+(const pw_power& p_power1,
                   const pw_power& p_power2)
{
    pw_power l_power(p_power1);
    return l_power += p_power2;
}


// ---------------------------------------------------------------------------
pw_power operator-(const pw_power& p_power1,
                   const pw_power& p_power2)
{
    pw_power l_power(p_power1);
    return l_power -= p_power2;
}


// ---------------------------------------------------------------------------
pw_power operator*(const pw_power& p_power,
                   double p_value)
{
    pw_power l_power(p_power);
    return l_power *= p_value;
}


// ---------------------------------------------------------------------------
pw_power operator*(double p_value,
                   const pw_power& p_power)
{
    pw_power l_power(p_power);
    return l_power *= p_value;
}


// ---------------------------------------------------------------------------
double operator/(const pw_power& p_power1,
                 const pw_power& p_power2)
{
    return p_power1.to_double() / p_power2.to_double();
}


// ---------------------------------------------------------------------------
pw_power operator/(const pw_power& p_power,
                   double p_value)
{
    pw_power l_power(p_power);
    return l_power /= p_value;
}


// ---------------------------------------------------------------------------
std::ostream& operator<<(std::ostream& p_os,
                         const pw_power& p_power)
{
    std::ostream::sentry l_init(p_os);
    if(!l_init)
    {
        SC_REPORT_ERROR(PW_POWER_OSTREAM_ERROR_TYPE_,
                        PW_POWER_OSTREAM_ERROR_MSG_);
        return p_os;
    }

    if(p_os.flags()
       & (std::ios_base::scientific | std::ios_base::fixed))
        return p_os << p_power.to_watts() << " "
                    << g_tab_power_units[PW_WATT];
    else
        return p_os << p_power.to_string();
}


// ---------------------------------------------------------------------------
std::istream& operator>>(std::istream& p_is,
                         pw_power& p_power)
{
    std::istream::sentry l_init(p_is,
                                false);
    if(!l_init)
    {
        SC_REPORT_ERROR(PW_POWER_ISTREAM_ERROR_TYPE_,
                        PW_POWER_ISTREAM_ERROR_MSG_);
        return p_is;
    }

    double l_value = 0;
    p_is >> l_value;
    if(not p_is)
    {
        SC_REPORT_ERROR(PW_POWER_ISTREAM_ERROR_TYPE_,
                        PW_POWER_ISTREAM_ERROR_MSG_);
        return p_is;
    }

    pw_power_unit l_unit;
    p_is >> l_unit;
    if(not p_is)
    {
        SC_REPORT_ERROR(PW_POWER_ISTREAM_ERROR_TYPE_,
                        PW_POWER_ISTREAM_ERROR_MSG_);
        return p_is;
    }

    p_power = pw_power(l_value,
                       l_unit);
    return p_is;

}


// ---------------------------------------------------------------------------
void pw_set_power_resolution(double p_value,
                             pw_power_unit p_unit)
{
    // Sanity checks
    if(sc_core::sc_is_running())
    {
        SC_REPORT_ERROR(PW_POWER_RESOLUTION_ELABORATION_TYPE_,
                        PW_POWER_RESOLUTION_ELABORATION_MSG_);
        return;
    }
    if(pw_power::a_is_resolution_fixed)
    {
        SC_REPORT_ERROR(PW_POWER_RESOLUTION_FIXED_TYPE_,
                        PW_POWER_RESOLUTION_FIXED_MSG_);
        return;
    }

    if(p_value == 0)
    {
        SC_REPORT_ERROR(PW_POWER_RESOLUTION_ZERO_TYPE_,
                        PW_POWER_RESOLUTION_ZERO_MSG_);
        return;
    }

    if(p_value < 0)
    {
        SC_REPORT_ERROR(PW_POWER_RESOLUTION_NEGATIVE_TYPE_,
                        PW_POWER_RESOLUTION_NEGATIVE_MSG_);
        return;
    }

    if(p_unit < PW_fW or p_unit > PW_WATT)
    {
        SC_REPORT_ERROR(PW_POWER_UNIT_OVERFLOW_TYPE_,
                        PW_POWER_UNIT_OVERFLOW_MSG_);
        return;
    }

    if(not is_pow10(p_value))
    {
        SC_REPORT_ERROR(PW_POWER_RESOLUTION_POW10_TYPE_,
                        PW_POWER_RESOLUTION_POW10_MSG_);
        return;
    }

    double l_resolution_value = pw_rint(p_value * g_tab_power_scales[p_unit]);
    if(l_resolution_value < 1.0
       or l_resolution_value > (static_cast<double>(PW_UINT64_MAX)))
    {
        SC_REPORT_ERROR(PW_POWER_RESOLUTION_OVERFLOW_TYPE_,
                        PW_POWER_RESOLUTION_OVERFLOW_MSG_);
        return;
    }

    pw_power::a_resolution_value = l_resolution_value;
    pw_power::a_is_resolution_fixed = true;

    pw_info << "Resolution Fixed to: " << pw_power::a_resolution_value
            << " Femto Watt" << pw_endl;
}

  // ---------------------------------------------------------------------------

  // This simply rounds downwards.
  pw_power pw_power::round3sf(void) const
  {
    if (a_value == 0) return *this;
    sc_dt::uint64 v0 = a_value;
    sc_dt::uint64 v1 = a_value;
    sc_dt::uint64 v2 = a_value;
    int count = 0;
    while (true)
      {
	sc_dt::uint64 n = v0/10U;
	if (n == 0) break;
	count ++;
	v2 = v1; v1 = v0; v0 = n;
      }
    pw_power result(*this);
    result.a_value  = v2;
    for (int i = 0; i<count-2; i++) result.a_value *= 10;
    // std::cout << "Rounded " << *this << " to " << result << "\n";
    return result;
  }

// ---------------------------------------------------------------------------
pw_power pw_get_power_resolution(void)
{
    if(not pw_power::a_is_resolution_fixed)
        pw_info << "Resolution Fixed to: " << pw_power::a_resolution_value
                << " Femto Watt" << pw_endl;

    pw_power l_power;
    l_power.a_value = 1;
    pw_power::a_is_resolution_fixed = true;
    return l_power;
}


} // namespace sc_pwr
