// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_module_base.cpp,v 1.2 2011/07/25 15:32:45 my294 Exp $

/** @file pw_module_base.cpp
 * @brief Base class used by the SystemC module for recording their power 
 * consumption.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 *
 * This class has to be used as a base class of the SystemC components for 
 * recording thier power consumption.
 */

#include <sstream>
#include "pw_module_base.h"
#include "pw_kernel_ids.h"
#include "pw_debug.h"
#include "math.h"
#include "pw_physical_operators.h"
#include "pw_common.h"
#include "pw_tracing_ids.h"
#include "pw_mode.h"
#include "pw_phase.h"


using namespace std;
// ===========================================================================
namespace
{
//! Convertion table of power unit into string.
const char* g_tab_attribute_ids[] =
{
    "INIT",
    "STATIC_POWER",
    "TRANSACTION",
    "QUANTUM_UP",
    "LOG_EVENT",
    "NONE"
};

} // ANONYMOUS NAMESPACE (replace static declaration)

// ===========================================================================
namespace sc_pwr
{
 
  // ---------------------------------------------------------------------------
  // Constructor
  pw_module_base::pw_module_base(void):
    a_last_modified_attribute(NONE),
    m_excess_area(PW_ZERO_AREA),
    m_area(PW_ZERO_AREA),
    m_area_set(false),
    m_placement_set(false),
    a_transaction_energy(PW_ZERO_ENERGY),
    a_transaction_delay(SC_ZERO_TIME),
    a_vcc_set(false),
    a_power_mode(PW_MODE_UNDEF),
    a_power_phase(PW_PHASE_UNDEF),
    m_sc_object(0)

  {
  }

  // ---------------------------------------------------------------------------
  void pw_module_base::attach(pw_observer_if& p_observer, const char *p_aid)
  {
    pw_subject::attach(p_observer, p_aid);
    a_transaction_acct = 0;
    a_last_modified_attribute = INIT;
    p_observer.update(*this);
  }

// --------------------------------------------------------------------------
void pw_module_base::quantum_up()
{
    pw_debug << " quantum kick to module <" << get_name()
	     << "> @ " << sc_time_stamp() << pw_endl;
    a_last_modified_attribute = QUANTUM_UP;
    this->notify();
}

// --------------------------------------------------------------------------
void pw_module_base::log_event()
{
    pw_debug << " log event to module <" << get_name()
	     << "> @ " << sc_time_stamp() << pw_endl;
    a_last_modified_attribute = LOG_EVENT;
    this->notify();

    // Recursively log the children of this object.
    const std::vector< sc_core::sc_object* > l_tab_objs = get_sc_object().get_child_objects();

    for(std::vector< sc_core::sc_object* >::const_iterator l_it =
        l_tab_objs.begin();
        l_it != l_tab_objs.end();
        ++l_it)
        { 
          pw_module_base* l_module = dynamic_cast< pw_module_base* > (*l_it); 
          if (l_module == 0x0) 
	    { pw_debug << "Skipping null module!" << pw_endl;
	    }
          else
	    {
	      l_module->set_modified_attribute(LOG_EVENT);  // XXX Recursive CALL
	      l_module->notify();
	    }
        }

}

  // ---------------------------------------------------------------------------
  void pw_module_base::record_utilisation(const sc_time busytime, const sc_time *startref) 
  {
    assert(0); // ... unfinished
    a_transaction_delay = busytime;
    // we need a start ref for LT mode (it does not start at sc_time_stamp() but at some point in the future.
    this->notify();
  }


  // ---------------------------------------------------------------------------
  void pw_module_base::record_energy_use(const pw_energy& p_energy,
					 int account_no,
					 // Using noscale is advisable since these multiplies by vsquared are very slow... : largest contributor in the gprof output!  Instead, components should do their own squaring in recompute_pvt_params.
					 bool noscale
					 )
  {
    assert (account_no >= 0 && account_no <= PW_ACCT_MAX_NO_ACCOUNTS);
    if (!noscale)
      {
	// Dynamic energy is proportional to square of Vcc. 
	double frac = get_vcc().to_volts();
	a_transaction_energy = p_energy * frac * frac;
      }
    else a_transaction_energy = p_energy;
    pw_debug << "Energy act " << account_no << " of module <" << get_name() << "> to be incremented by " << a_transaction_energy.round3sf() << pw_endl;
    //std::cout << ((void *)this) << " Energy act " << account_no << " of module <" << get_name() << "> to be incremented by " << a_transaction_energy.round3sf() << pw_endl;
    a_transaction_acct = account_no;
    a_last_modified_attribute = TRANSACTION;
    this->notify();
    //std::cout << "notify done " << ((void *)this) << "\n";
  }

  // ---------------------------------------------------------------------------
  void pw_module_base::set_power_mode(const std::string& p_mode)
  {
    pw_debug << "Power Mode of module <" << get_name() << "> set to " << p_mode << pw_endl;

    pw_assert(false and "needs re-implementing");
    a_power_mode = p_mode;
    //a_last_modified_attribute = POWER_MODE;
    this->notify();
  }
  
  
  // ---------------------------------------------------------------------------
  void pw_module_base::set_power_phase(const std::string& p_phase)
  {
    pw_debug << "Power Phase of module <" << get_name() << "> set to " << p_phase << pw_endl;
    a_power_phase = p_phase;
    //a_last_modified_attribute = POWER_PHASE;

    pw_assert(false and "needs re-implementing");
    this->notify();
  }
  


// ---------------------------------------------------------------------------
  pw_energy pw_module_base::pacct::set_local_phasemode_power(const pw_power& p_power, sc_time p_time)
  {
    // Integrate power consumption in interval that just ended.
    sc_time l_interval = p_time - m_last_baseline_update;
    pw_energy l_energy = m_phasemode_power * l_interval;
    m_last_baseline_update = p_time;
    m_phasemode_power = p_power;
    pw_debug << "local_fold:x standing power  acct=" << m_parent->get_name() << " t=" << l_interval << " P=" << m_phasemode_power.round3sf() << " e=" << l_energy << ".  New power=" << p_power.round3sf() << "\n";
    return l_energy;
  }

  void pw_module_base::set_static_power(const pw_power& p_power, int l_acct)
  {
    pw_debug << "Static Power of module <" << get_name() << "> set to " << p_power << pw_endl;
    sc_time when = sc_time_stamp();
    if (!a_paccts[l_acct].inuse()) a_paccts[l_acct].start(this);
    pw_energy l_energy = a_paccts[l_acct].set_local_phasemode_power(p_power, when);
    a_transaction_acct = l_acct;
    a_transaction_energy = l_energy;
    a_last_modified_attribute = PHASEMODE_POWER;
    this->notify();
  }


  // ---------------------------------------------------------------------------
  const std::string& pw_module_base::get_power_mode(void) const
  {
    return a_power_mode;
  }


  // ---------------------------------------------------------------------------
  const std::string& pw_module_base::get_power_phase(void) const
  {
      return a_power_phase;
  }

  // ---------------------------------------------------------------------------
  void pw_module_base::set_static_power(double p_value,
					pw_power_unit p_unit, 
					int l_acct)
  {
      set_static_power(pw_power(p_value, p_unit), l_acct);
  }

// ---------------------------------------------------------------------------
void pw_module_base::set_scaled_static_power(const pw_power& p_power)
{
  pw_debug << "Static Power of module scaled to Vcc = 1.0V " << get_name() << "> set to " << p_power << pw_endl;
  //    a_scaled_static_power = p_power;
}

  // ---------------------------------------------------------------------------
void pw_module_base::set_scaled_static_power(double p_value, pw_power_unit p_unit)
{
  set_scaled_static_power(pw_power(p_value, p_unit));
}

  // ---------------------------------------------------------------------------
  // Every component is nominally on a given chip which has a textual name.
  // If no name has been set then the name of the instantiating parent is used. 
  // At the top level, if no name is give, then the SystemC name() is used.

  void pw_module_base::set_chip_region(const char *p_name, bool p_update_children)
  {
    std::string l = p_name;
    set_chip_region(l, p_update_children);
  }

  //! Set chip or region name and also, optionally, that of all children not yet set.
  void pw_module_base::set_chip_region(std::string &p_name, bool p_update_children)
  {
    m_chip_region = p_name;
    // Recursively set child region
    if (p_update_children)
      {
	const std::vector< sc_core::sc_object* > l_tab_objs = get_sc_object().get_child_objects();
	for(std::vector< sc_core::sc_object* >::const_iterator l_it =
	      l_tab_objs.begin();
	    l_it != l_tab_objs.end();
	    ++l_it)
	  { 
	    pw_module_base* l_module = dynamic_cast< pw_module_base* > (*l_it); 
	    if (!l_module)
	      {
		pw_debug << "set_chip_region: Skipping non-module object " 
			 << (*l_it)->kind() << "\n";
	      }
	    else
	      {
		if (l_module->get_chip_region() == get_chip_region())
		  l_module->set_chip_region(p_name, p_update_children);
	      }
	  }
      }

  }

  const std::string& pw_module_base::get_chip_region(void)
  {
    if (m_chip_region.empty()) 
      {
	sc_object *l_p = get_sc_object().get_parent_object();
	if (!l_p)
	  {
	    m_chip_region = get_sc_object().name();
	  }
	else 
	  {
	    pw_module_base* l_module = dynamic_cast< pw_module_base* > (l_p); 
	    if (!l_module)
	      {
		pw_debug << "get_chip_region: Skipping non-module object " << l_p->kind() << "\n";
		m_chip_region = get_sc_object().name();
	      }
	    else  m_chip_region = l_module->get_chip_region();
	  }
      }
    return m_chip_region;
  }

  // ---------------------------------------------------------------------------
  void pw_module_base::set_vcc(const pw_voltage& p_vcc, bool p_update_children) 
  {
    pw_debug << "Working Vcc Power of module <" << get_name() 
	     << "> set to " << p_vcc << pw_endl;
    pw_debug << "set_vcc: set Vcc for module <" << get_name() 
	     << "> set to " << p_vcc << pw_endl;
    //double frac = (p_vcc.to_volts());
    //frac = pow(frac, 1.8);
    //pw_power l_power = a_scaled_static_power * frac;
    //cout << "[[.. scaled = " << a_scaled_static_power << "..]]" << endl;
    //set_static_power(l_power);
    a_vcc = p_vcc;

    // Recursively set child VCC when has the same chip/region name.
    if (p_update_children)
      {
	const std::vector< sc_core::sc_object* > l_tab_objs = get_sc_object().get_child_objects();
	for(std::vector< sc_core::sc_object* >::const_iterator l_it =
	      l_tab_objs.begin();
	    l_it != l_tab_objs.end();
	    ++l_it)
	  { 
	    pw_module_base* l_module = dynamic_cast< pw_module_base* > (*l_it); 
	    if (!l_module)
	      {
		//cout << "set_vcc: Skipping non-module object " << (*l_it)->kind() << "\n";
	      }
	    else
	      {
		if (!l_module->m_sc_object) l_module->m_sc_object = *l_it; // Be helpful
		if (l_module->get_chip_region() == get_chip_region())
		  l_module->set_vcc(p_vcc, p_update_children);
	      }
	  }
      }

    recompute_pvt_parameters(); // Invoke the callback on the component.
  }


  // ---------------------------------------------------------------------------
  void pw_module_base::set_vcc(double p_value, pw_voltage_unit p_unit, bool p_update_children)   
  {
    set_vcc(pw_voltage(p_value, p_unit), p_update_children);
  }
  
  // ---------------------------------------------------------------------------
  void pw_module_base::set_modified_attribute(attribute_id p_attr)
  {
    a_last_modified_attribute = p_attr;
  }

// ---------------------------------------------------------------------------
pw_module_base::attribute_id
pw_module_base::get_modified_attribute(void) const
{
    return a_last_modified_attribute;
}

// ---------------------------------------------------------------------------
const pw_energy& pw_module_base::get_transaction_energy(void) const
{
    return a_transaction_energy;
}

  // ---------------------------------------------------------------------------
  // Get current supply voltage.  If not set yet get from parent.
  const pw_voltage& pw_module_base::get_vcc(void)
  {
    if (!a_vcc_set) //Needs setting?
      {
	sc_object *l_p = get_sc_object().get_parent_object();
	if (!l_p)
	  {
	    a_vcc = pw_voltage(1.0, PW_VOLT); // Default voltage.
	  }
	else 
	  {
	    pw_module_base* l_module = dynamic_cast< pw_module_base* > (l_p); 
	    if (!l_module)
	      {
		pw_debug << "get_vcc: Parent was not a power module: " << l_p->kind() << "\n";
		a_vcc = pw_voltage(1.0, PW_VOLT);
	      }
	    else  a_vcc = l_module->get_vcc();
	  }
	a_vcc_set = true;
	recompute_pvt_parameters(); // Invoke the callback on the component.
      }

    return a_vcc;
  }

// ---------------------------------------------------------------------------
const sc_time& pw_module_base::get_transaction_delay(void) const
{
    return a_transaction_delay;
}

// ---------------------------------------------------------------------------

const int pw_module_base::get_transaction_acct(void) const
{
    return a_transaction_acct;
}

// ---------------------------------------------------------------------------
#if 0
const pw_power& pw_module_base::get_phasemode_power(int acct) const
{
  
    return a_phasemode_power;
}
#endif

// ---------------------------------------------------------------------------

  
  sc_core::sc_module_name pw_module_base::get_name(void)
  {
    // This can fail depending if called locally, but is ok if called from
    // within a class that has an sc_object as a base.
    try
      {
        return dynamic_cast< sc_core::sc_object& >(*this).name();
      }
    catch(std::bad_cast)
      {
	return "<pw_module_base.noname>";
      }
    
    // Dead Code for removing compilation warning.
    return "<pw_module_base.noname>";
  }

  // Fill in the module_base to object shortcuts.
  // TODO check attributes.
  void pw_module_base::scan_all_module_maps(sc_object *pos)
  {
    typedef std::vector< sc_core::sc_object* > vect_const;
    typedef std::vector< sc_core::sc_object* >::const_iterator vect_const_iterator;

    if (!pos)
      { // Seed top of tree walk.
	const vect_const & l_objs = sc_core::sc_get_top_level_objects();
	for(vect_const_iterator l_it = l_objs.begin();
	    l_it != l_objs.end();
	    ++l_it)
	  scan_all_module_maps(*l_it);
      }
    else
      { // Recursive step.
	sc_module *mod = dynamic_cast<sc_module *>(pos);
	if (mod)
	  {
	    pw_debug << "scan_all_module_maps:: " << mod->name() << pw_endl;
	    pw_module_base *pmod = dynamic_cast<pw_module_base *>(mod);
	    if (pmod)
	      {
		pmod->m_sc_object = pos;
	      }
	    //printf("This Q %p %p %p\n", pos, mod, pmod);
	  }
	const vect_const l_objs = pos->get_child_objects();
	for(vect_const_iterator l_it = l_objs.begin();
	    l_it != l_objs.end();
	    ++l_it)
	  scan_all_module_maps(*l_it);
      }
  }


  // This can fail when called locally (trying to find peer sub-class under multiple inheritance is not possible), but it is ok if called from
  // within a class that inherits from  sc_object.
  // Or it can fail if called before all constructors for children have been run.
  sc_core::sc_object& pw_module_base::get_sc_object(void)
  {
    if (m_sc_object) return *m_sc_object;
    try
      {
	// GCC can make this work by magic sometimes?
        return dynamic_cast< sc_core::sc_object& >(*this);
      }
    catch(std::bad_cast)
      {
	scan_all_module_maps(0);
	if (m_sc_object) return *m_sc_object; // May be completed now!
	//printf("This S %p\n", this);
	pw_debug << "Note: probably a constructor race." << pw_endl;
        SC_REPORT_ERROR(PW_MODULE_DYNAMIC_CAST_FAILURE_TYPE_,
                        PW_MODULE_DYNAMIC_CAST_FAILURE_MSG_);
      }
    
    // Dead Code for removing compilation warning.
    return dynamic_cast< sc_core::sc_object& >(*this);
  }

// ---------------------------------------------------------------------------
// We may not have inherited the sc_module: it might just be an attribute.

pw_module_base *get_pw_module_base(sc_object &it)
{
  pw_module_base *r = dynamic_cast< pw_module_base *>(&it);
  if (!r)
    {
      // todo it.get_attribute( ... ) 
      pw_error << "get_pw_module_base " << &it << " " << r << pw_endl;
      pw_error << "Error in " << it.name() << ":" << it.kind() 
	       << " (must be pw_module_base annotated component: subsequent error message wrong?) :" << pw_endl;
#if 0
  TODO
      // unannotated module
      SC_REPORT_ERROR(PW_MODULE_DYNAMIC_CAST_FAILURE_TYPE_,
		      PW_MODULE_DYNAMIC_CAST_FAILURE_MSG_);
#endif
    }
  return r;
}


// ---------------------------------------------------------------------------
sc_core::sc_module& pw_module_base::get_sc_module(void)
{
    try
    {
        return dynamic_cast< sc_core::sc_module& >(*this);
    }
    catch(std::bad_cast)
    {
        SC_REPORT_ERROR(PW_MODULE_DYNAMIC_CAST_FAILURE_TYPE_,
                        PW_MODULE_DYNAMIC_CAST_FAILURE_MSG_);
    }
    
    // Dead Code for removing compilation warning.
    return dynamic_cast< sc_core::sc_module& >(*this);
  }


  // ---------------------------------------------------------------------------  

  //! Lowest Common Parent 
  // This is at first quadratic slow search but the answer is cached here.
  sc_object *pw_module_base::lowest_common_parent_scan(sc_module *other)
  {
    
    for (sc_object *here = &get_sc_object(); here; here=here->get_parent())
      {
	for (sc_object *there = (sc_object *)other; there; there=there->get_parent())
	  {
	    if (here == there) return here;
	  }
      }
    pw_info  << "No common parent between " << get_sc_object().name() 
	      << " and " << other->name() << pw_endl;
    return 0;
  }

  pw_module_base::pw_parent_cache_entry *pw_module_base::lowest_common_parent(sc_module *other)
  {
    pw_parent_cache_entry *ans = m_lowest_common_parent[other];
    if (!ans)
      {
	sc_module * l_lc_parent = dynamic_cast<sc_module *>(lowest_common_parent_scan(other));
	pw_length l_distance = rentian_distance(l_lc_parent);
	ans = new pw_parent_cache_entry(other, l_lc_parent, l_distance);
	m_lowest_common_parent[other] = ans;
      }
    return ans;
  }

  // ---------------------------------------------------------------------------
  // Mean distance given sensible place-and-route.
  pw_length rentian_distance(sc_module *parent)
  {
    pw_module_base *par = get_pw_module_base(*parent);
    pw_area area = par->get_area();
    
    double len = sqrt(area.to_sqmeters());
#if 0	      
    pw_length q = pw_length(10, PW_um);
    for (int i=0;i<100; i++) 
      {
	q = (q + area/q)/2.0;
	//cout << "sqrt iteration " << q << "\n";
      }
    q = q/2;
#endif
    pw_length q1 = pw_length(len/2, PW_METER).round3sf();
    pw_info << "parent area is " << area.round3sf() << " (" << area.to_sqmeters() 
	    << ") hop distance is " << q1 << pw_endl;
    return q1;
  }

  // ---------------------------------------------------------------------------
  //! Area in excess of internal components
  void pw_module_base::set_excess_area(pw_length x, pw_length y, float max_aspect_ratio, const char *l_chipname)
  {
    assert(!m_area_set); // must not be set already
    pw_area l_a = x * y;
    pw_debug << x << " * " << y << " Log area " << l_a << pw_endl;
    set_excess_area(l_a); 
    if (l_chipname) set_chip_region(l_chipname);
  }


  //! Area in excess of internal components
  void pw_module_base::set_excess_area(pw_area l_a, float max_aspect_ratio, const char *l_chipname)
  {
    assert(!m_area_set); // must not be set already
    pw_debug << " Log area " << l_a << pw_endl;
    m_excess_area = l_a; 
    if (l_chipname) set_chip_region(l_chipname);
  }

  void pw_module_base::set_fixed_area(pw_area l_a, const char *l_chipname)
  {
    assert(!m_area_set); // must not be set already
    pw_debug << " Log area " << l_a << pw_endl;
    m_area_set = true;
    m_area = l_a; 
    m_excess_area = PW_ZERO_AREA; 
    if (l_chipname) set_chip_region(l_chipname);
  }
  
// ---------------------------------------------------------------------------
const pw_area &pw_module_base::get_area(void) 
{
  if (!m_area_set)
    {
      // Recursively sum the children of this object.
      const std::vector< sc_core::sc_object* > l_tab_objs = get_sc_object().get_child_objects();
      pw_area total = PW_ZERO_AREA;
      for(std::vector< sc_core::sc_object* >::const_iterator l_it =
	    l_tab_objs.begin();
	  l_it != l_tab_objs.end();
	  ++l_it)
	{ 
	  pw_module_base* l_module = dynamic_cast< pw_module_base* > (*l_it); 
	  if (!l_module)
	    {
	      //cout << "get_area: Skipping non-module object " << (*l_it)->kind() << "\n";
	    }
	  else
	    {
	      pw_area child_area = l_module->get_area();
	      total += child_area;
	      //cout /*pw_debug*/ << "Summing area of child " << get_name() << " child " << l_module->get_sc_object().name() << " area=" << child_area.round3sf() << "\n";
	    }
	}
      m_area = total + m_excess_area;
      pw_debug << "Computed area " << get_name() << " area =" 
	       << m_area.round3sf() << pw_endl;
      m_area_set = true;
    }
  return m_area;
}
// ---------------------------------------------------------------------------
const std::string pw_module_base::to_string(void) const
{
    std::ostringstream l_os;

    l_os << "{ ";
    try
    {
        const sc_core::sc_object& l_object =
            dynamic_cast< const sc_core::sc_object& >(*this);
        l_os << l_object.name();
    }
    catch(std::bad_cast)
    {
        SC_REPORT_ERROR(PW_MODULE_DYNAMIC_CAST_FAILURE_TYPE_,
                        PW_MODULE_DYNAMIC_CAST_FAILURE_MSG_);
    }

    l_os // << ", ( " << get_old_static_power()
         //      << " -> " << get_phasemode_power()
         << ", " << get_modified_attribute() << " }";

    return l_os.str();
}

// ---------------------------------------------------------------------------
// Output stream operator of power module object.
std::ostream& operator<<(std::ostream& p_os,
                         const pw_module_base::attribute_id p_id)
{
    std::ostream::sentry l_init(p_os);
    if(!l_init)
    {
        SC_REPORT_ERROR(PW_KERNEL_OSTREAM_ERROR_TYPE_,
                        PW_KERNEL_OSTREAM_ERROR_MSG_);
        return p_os;
    }

    if(p_id < pw_module_base::INIT
       or p_id > pw_module_base::NONE)
    {
        SC_REPORT_ERROR(PW_MODULE_ATTRIBUTE_ID_OVERFLOW_TYPE_,
                        PW_MODULE_ATTRIBUTE_ID_OVERFLOW_MSG_);
        return (p_os << "undef");
    }

    return (p_os << g_tab_attribute_ids[p_id]);

}


  // Write physical layout information to report file.
  void pw_module_base::physical_report(std::fstream *osp, sc_module &module)
  {
    if (osp)
      {
	std::fstream l_os, *fd = (osp) ? osp:&l_os;
	*fd << "\n"; 
	*fd << "Physical report " << module.name() << ":" << module.kind() << "\n"; 
	*fd << "      Chip/region name: " << m_chip_region << "\n";
	if (m_area_set)
	  {
	    *fd << "      Area " << m_area.round3sf() << "\n";
	  }
	*fd << "      Vcc= " << get_vcc() << "\n";
	//	if (!osp) cout << fd->str();
      }
  }


  void pw_module_base::flush_all_modules(sc_object *pos, std::fstream *osp)
  {
    typedef std::vector< sc_core::sc_object* > vect_const;
    typedef std::vector< sc_core::sc_object* >::const_iterator vect_const_iterator;

    if (!pos)
      { // Seed top of tree walk.
	const vect_const & l_objs = sc_core::sc_get_top_level_objects();
	for(vect_const_iterator l_it = l_objs.begin();
	    l_it != l_objs.end();
	    ++l_it)
	  flush_all_modules(*l_it, osp);
      }
    else
      { // Recursive step.
	sc_module *mod = dynamic_cast<sc_module *>(pos);
	if (mod)
	  {
	    pw_module_base *pmod = dynamic_cast<pw_module_base *>(mod);
	    if (pmod) pmod->physical_report(osp, *mod);
	    //printf("Siz %s %s %p\n", mod->name(), mod->kind(), pmod);
	    if (pmod) pmod->destructor_prologue(); // aka flush
	    const vect_const l_objs = pos->get_child_objects();
	    for(vect_const_iterator l_it = l_objs.begin();
		l_it != l_objs.end();
		++l_it)
	      flush_all_modules(*l_it, osp);
	  }
      }
  }


  void pw_module_base::destructor_prologue()
  {
    sc_time eot = sc_pwr::sc_time_max();
    sc_time when = sc_time_stamp();

    for (int l_acct=0; l_acct<PW_ACCT_MAX_NO_ACCOUNTS; l_acct++) 
      if (a_paccts[l_acct].inuse())
	{
	  pw_debug << "Close account " __FILE__  << " acct=" << l_acct << pw_endl;
	  pw_energy l_energy = a_paccts[l_acct].set_local_phasemode_power(PW_ZERO_POWER, when);
	  if (l_energy != PW_ZERO_ENERGY)
	    {
	      if (eot == when) 
		{
		  const string l_warning_msg =
		    string(PW_POWERED_OBSERVER_STOPPED_ACTIVE_MSG_)
		    + " (module \"" + this->to_string() + "\").";
		  SC_REPORT_WARNING(PW_POWERED_OBSERVER_STOPPED_ACTIVE_TYPE_,
				    l_warning_msg.c_str());
		}
	      a_transaction_acct = l_acct;
	      a_transaction_energy = l_energy;
	      a_last_modified_attribute = PHASEMODE_POWER;
	      this->notify();
	    }
	}
  }


  pw_module_base::~pw_module_base(void)
  {
    pw_debug  << "Destructor invoked\n";
    destructor_prologue();
  }


// ---------------------------------------------------------------------------
// Output stream operator of power object.
std::ostream& operator<<(std::ostream& p_os,
                         const pw_module_base& p_module)
{
    std::ostream::sentry l_init(p_os);
    if(!l_init)
    {
        SC_REPORT_ERROR(PW_KERNEL_OSTREAM_ERROR_TYPE_,
                        PW_KERNEL_OSTREAM_ERROR_MSG_);
        return p_os;
    }
    return (p_os << p_module.to_string());
}

} // namespace sc_pwr
