// TLM_POWER3: Area-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_area.cpp,v 1.1 2011/09/08 10:38:43 my294 Exp $

/** @file pw_area.cpp
 * @brief Define area data type.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 */

#include <ios>
#include <cmath>
#include <sstream>
#include <iomanip>
#include <iostream>
#include <algorithm>

#include "pw_area.h"
#include "pw_kernel_ids.h"
#include "pw_debug.h"
#include "pw_common.h"

// ===========================================================================
namespace
{


// ---------------------------------------------------------------------------
using sc_dt::uint64;


// ---------------------------------------------------------------------------
//! Convertion table of energy unit into string.
const char* g_tab_area_units[] =
{
    "sqnm",
    "squm",
    "sqmm",
    "sqcm",
    "sqm",
};

const double g_tab_area_scales[] =
{
    1e+0,    // square nanometres
    1e+6,    // square microns
    1e+12,     // square millimetre
    1e+14,     // square centimeters
    1e+18       // square metres
};

const unsigned int g_tab_area_logs[] =
{
    0,    // square nanometres
    6,    // square microns
    12,     // square millimetre
    14,     // square centimeters
    18       // square metres
};


// ---------------------------------------------------------------------------
/**
 * @brief Test if an uint is a power of 10.
 * @param[in] p_val The uint to test.
 * @return true only if p_val is an area of 10.
 */
static bool is_pow10(double p_val)
{
    double l_tmp;
    return std::modf(std::log10(p_val), &l_tmp) == 0.0;
}


// ---------------------------------------------------------------------------
//! Simple rinting function
inline double pw_rint(double p_val)                                                     
{
    return std::floor( p_val + 0.5 );
}

} // ANONYMOUS NAMESPACE (replace static declaration)


// ===========================================================================
namespace sc_pwr
{


// ---------------------------------------------------------------------------
using sc_dt::uint64;


// ---------------------------------------------------------------------------
//! Area resolution
double pw_area::a_resolution_value = g_tab_area_scales[PW_sqnm];
//! True if a pw_area different of 0 have been constructed.
bool pw_area::a_is_resolution_fixed = false;


// ---------------------------------------------------------------------------
//! ZERO energy constant
const pw_area PW_ZERO_AREA;


// ---------------------------------------------------------------------------
pw_area_unit& operator++(pw_area_unit& p_unit)
{
    if(p_unit < PW_sqnm)
    {
        p_unit = PW_sqnm;
        SC_REPORT_WARNING(PW_AREA_UNIT_OVERFLOW_TYPE_,
                          PW_AREA_UNIT_OVERFLOW_MSG_);
    }
    if(p_unit >= PW_sqm)
    {
        p_unit = PW_sqm;
        SC_REPORT_WARNING(PW_AREA_UNIT_OVERFLOW_TYPE_,
                          PW_AREA_UNIT_OVERFLOW_MSG_);
        return p_unit;
    }

    unsigned l_uint = static_cast< unsigned >(p_unit);
    p_unit = static_cast< pw_area_unit >(++l_uint);

    pw_assert(p_unit > PW_sqnm and p_unit <= PW_sqm \
              and "Unexpected overflow!!!");

    return p_unit;
}


// ---------------------------------------------------------------------------
pw_area_unit operator++(pw_area_unit& p_unit,
                          int)
{
    if(p_unit < PW_sqnm)
    {
        p_unit = PW_sqnm;
        SC_REPORT_WARNING(PW_AREA_UNIT_OVERFLOW_TYPE_,
                          PW_AREA_UNIT_OVERFLOW_MSG_);
    }
    if(p_unit >= PW_sqm)
    {
        p_unit = PW_sqm;
        SC_REPORT_WARNING(PW_AREA_UNIT_OVERFLOW_TYPE_,
                          PW_AREA_UNIT_OVERFLOW_MSG_);
        return p_unit;
    }

    unsigned l_uint = static_cast< unsigned >(p_unit);
    pw_area_unit l_unit = static_cast< pw_area_unit >(++l_uint);

    pw_assert(l_unit > PW_sqnm and l_unit <= PW_sqm \
              and "Unexpected overflow!!!");

    return l_unit;
}


// ---------------------------------------------------------------------------
pw_area_unit& operator--(pw_area_unit& p_unit)
{
    if(p_unit <= PW_sqnm)
    {
        p_unit = PW_sqnm;
        SC_REPORT_WARNING(PW_AREA_UNIT_OVERFLOW_TYPE_,
                          PW_AREA_UNIT_OVERFLOW_MSG_);
        return p_unit;
    }
    if(p_unit > PW_sqm)
    {
        p_unit = PW_sqm;
        SC_REPORT_WARNING(PW_AREA_UNIT_OVERFLOW_TYPE_,
                          PW_AREA_UNIT_OVERFLOW_MSG_);
    }

    unsigned l_uint = static_cast< unsigned >(p_unit);
    p_unit = static_cast< pw_area_unit >(--l_uint);

    pw_assert(p_unit >= PW_sqnm and p_unit < PW_sqm \
              and "Unexpected overflow!!!");

    return p_unit;
}


// ---------------------------------------------------------------------------
pw_area_unit operator--(pw_area_unit& p_unit,
                          int)
{
    if(p_unit <= PW_sqnm)
    {
        p_unit = PW_sqnm;
        SC_REPORT_WARNING(PW_AREA_UNIT_OVERFLOW_TYPE_,
                          PW_AREA_UNIT_OVERFLOW_MSG_);
        return p_unit;
    }
    if(p_unit > PW_sqm)
    {
        p_unit = PW_sqm;
        SC_REPORT_WARNING(PW_AREA_UNIT_OVERFLOW_TYPE_,
                          PW_AREA_UNIT_OVERFLOW_MSG_);
    }

    unsigned l_uint = static_cast< unsigned >(p_unit);
    pw_area_unit l_unit = static_cast< pw_area_unit >(--l_uint);

    pw_assert(l_unit >= PW_sqnm and l_unit < PW_sqm \
              and "Unexpected overflow!!!");

    return l_unit;
}


// ---------------------------------------------------------------------------
std::ostream& operator<<(std::ostream& p_os,
                         const pw_area_unit& p_unit)
{
    std::ostream::sentry l_init(p_os);
    if(!l_init)
    {
        SC_REPORT_ERROR(PW_AREA_UNIT_OSTREAM_ERROR_TYPE_,
                        PW_AREA_UNIT_OSTREAM_ERROR_MSG_);
        return p_os;
    }

    if(p_unit < PW_sqnm or p_unit > PW_sqm)
    {
        SC_REPORT_ERROR(PW_AREA_UNIT_OVERFLOW_TYPE_,
                        PW_AREA_UNIT_OVERFLOW_MSG_);
        return (p_os << "undef");
    }

    return (p_os << g_tab_area_units[p_unit]);
}


// ---------------------------------------------------------------------------
std::istream& operator>>(std::istream& p_is,
                         pw_area_unit& p_unit)
{
    std::istream::sentry l_init(p_is,
                                false);
    if(!l_init)
    {
        SC_REPORT_ERROR(PW_AREA_UNIT_ISTREAM_ERROR_TYPE_,
                        PW_AREA_UNIT_ISTREAM_ERROR_MSG_);
        return p_is;
    }

    std::string l_unit_idf;
    p_is >> l_unit_idf;
    if(not p_is)
    {
        SC_REPORT_ERROR(PW_AREA_UNIT_ISTREAM_ERROR_TYPE_,
                        PW_AREA_UNIT_ISTREAM_ERROR_MSG_);
        return p_is;
    }

    unsigned l_pos = 0;
    const unsigned l_size =
        sizeof(g_tab_area_units) / sizeof(const char*);
    while(l_pos < l_size
          and l_unit_idf != g_tab_area_units[l_pos])
        ++l_pos;

    if(l_pos == l_size)
    {
        p_is.setstate(std::ios_base::failbit);
        SC_REPORT_ERROR(PW_AREA_UNIT_ISTREAM_ERROR_TYPE_,
                        PW_AREA_UNIT_ISTREAM_ERROR_MSG_);
        return p_is;
    }

    p_unit = static_cast< pw_area_unit >(l_pos);
    return p_is;
}


// ---------------------------------------------------------------------------
pw_area::pw_area(void):
    a_value(0)
{ }


// ---------------------------------------------------------------------------
pw_area::pw_area(double p_value,
                     pw_area_unit p_unit):
    a_value(0)
{
    if(p_unit < PW_sqnm or p_unit > PW_sqm)
    {
        SC_REPORT_ERROR(PW_AREA_UNIT_OVERFLOW_TYPE_,
                        PW_AREA_UNIT_OVERFLOW_MSG_);
        return;
    }

    if(p_value < 0.0)
    {
        a_value = 0;
        SC_REPORT_ERROR(PW_AREA_NEGATIVE_TYPE_,
                        PW_AREA_NEGATIVE_MSG_);
        return;
    }

    if(p_value == 0) return;

    //printf("area: %e %e \n", g_tab_area_scales[p_unit], a_resolution_value);
    double l_scaling_factor =
      g_tab_area_scales[p_unit] / a_resolution_value;
    double l_value = p_value * l_scaling_factor;

    //printf("area p_unit=%i p_value %e  l_scaling  %e -> lvalue %e \n", p_unit, p_value, l_scaling_factor, l_value);

    /* XXX conversion problem when pw_rint(l_value) == PW_UINT64_MAX
     * We have to manage when these two values are equal.
     */
    if(pw_rint(l_value) >= (static_cast<double>(PW_UINT64_MAX)))
    {
        if(pw_rint(l_value) > (static_cast<double>(PW_UINT64_MAX)))
        {
            SC_REPORT_WARNING(PW_AREA_OVERFLOW_TYPE_,
                              PW_AREA_OVERFLOW_MSG_);
        }
        a_value = PW_UINT64_MAX;
        return;
    }

    /* XXX DEPRECATED XXX
     * Arithmetic operation with double are not enough accurate for checking 
     * the possible loss of information.
    double l_tmp;
    if(std::modf(l_value, &l_tmp) != 0.0)
    {
        SC_REPORT_WARNING(PW_AREA_OVERFLOW_TYPE_,
                          PW_AREA_OVERFLOW_MSG_);
    }
    XXX DEPRECATED XXX*/

    a_value = static_cast<uint64>(pw_rint(l_value));
    if(a_value != 0)
    {
        if(not a_is_resolution_fixed)
            pw_info << "Resolution Fixed to: " << a_resolution_value
                    << " Femto Meter" << pw_endl;
        a_is_resolution_fixed = true;
    }
}


// ---------------------------------------------------------------------------
pw_area::pw_area(const pw_area& p_area):
    a_value(p_area.a_value)
{ }


// ---------------------------------------------------------------------------
pw_area& pw_area::operator=(const pw_area& p_area)
{
    a_value = p_area.a_value;
    return *this;
}


// ---------------------------------------------------------------------------
pw_area& pw_area::operator+=(const pw_area& p_area)
{
//    printf("In += p_area %llx  a_value %llx \n", p_area.a_value, a_value);
    if(PW_UINT64_MAX - p_area.a_value < a_value)
    {
        a_value = PW_UINT64_MAX;
        SC_REPORT_WARNING(PW_AREA_OVERFLOW_TYPE_,
                          PW_AREA_OVERFLOW_MSG_);
    }
    else
        a_value += p_area.a_value;

    return *this;
}


// ---------------------------------------------------------------------------
pw_area& pw_area::operator-=(const pw_area& p_area)
{
    if(p_area.a_value > a_value)
    {
        a_value = 0;
        SC_REPORT_WARNING(PW_AREA_UNDERFLOW_TYPE_,
                          PW_AREA_UNDERFLOW_MSG_);
    }
    else
        a_value -= p_area.a_value;

    return *this;
}


// ---------------------------------------------------------------------------
pw_area& pw_area::operator*=(double p_value)
{
    if(p_value == 0 or a_value == 0)
        a_value = 0;
    else
    {
        double l_tmp = to_double() * p_value;
        a_value = static_cast<uint64>(pw_rint(l_tmp));
	// more likely to underflow!
        if(std::fabs(l_tmp - static_cast< double >(a_value)) > 1.)
        {
            a_value = PW_UINT64_MAX;
            SC_REPORT_WARNING(PW_AREA_OVERFLOW_TYPE_,
                              PW_AREA_OVERFLOW_MSG_);
        }
    }

    return *this;
}


// ---------------------------------------------------------------------------
pw_length operator/(const pw_area &n, const pw_length &d)
{
  double f = n.to_sqmeters() / d.to_meters();
  return pw_length(f, PW_METER);
}


pw_area& pw_area::operator/=(double p_value)
{
    if(p_value == 0.)
    {
        a_value = PW_UINT64_MAX;
        SC_REPORT_ERROR(PW_AREA_DIVISION_BY_ZERO_TYPE_,
                        PW_AREA_DIVISION_BY_ZERO_MSG_);
        return *this;
    }

    double l_tmp = to_double() / static_cast< double >(p_value);
    a_value = static_cast< uint64 >(pw_rint(l_tmp));

    return *this;
}


// ---------------------------------------------------------------------------
pw_area& pw_area::operator++(void)
{
    if(PW_UINT64_MAX == a_value)
    {
        SC_REPORT_WARNING(PW_AREA_OVERFLOW_TYPE_,
                          PW_AREA_OVERFLOW_MSG_);
    }
    else
        ++a_value;

    return *this;
}


// ---------------------------------------------------------------------------
pw_area pw_area::operator++(int)
{
    if(PW_UINT64_MAX == a_value)
    {
        SC_REPORT_WARNING(PW_AREA_OVERFLOW_TYPE_,
                          PW_AREA_OVERFLOW_MSG_);
        return *this;
    }

    pw_area l_res;
    l_res.a_value = a_value;
    ++a_value;
    return l_res;
}


// ---------------------------------------------------------------------------
pw_area& pw_area::operator--(void)
{
    if(0 == a_value)
    {
        SC_REPORT_WARNING(PW_AREA_UNDERFLOW_TYPE_,
                          PW_AREA_UNDERFLOW_MSG_);
    }
    else
        --a_value;

    return *this;
}


// ---------------------------------------------------------------------------
pw_area pw_area::operator--(int)
{
    if(0 == a_value)
    {
        SC_REPORT_WARNING(PW_AREA_UNDERFLOW_TYPE_,
                          PW_AREA_UNDERFLOW_MSG_);
        return *this;
    }

    pw_area l_tmp;
    l_tmp.a_value = a_value;
    --a_value;
    return l_tmp;
}


// ---------------------------------------------------------------------------
const std::string pw_area::to_string(void) const
{
    if(a_value == 0)
    {
        std::ostringstream l_os;
        l_os << "zero " << PW_sqm;
        return l_os.str();
    }

    pw_assert(pw_area::a_resolution_value <=
              (static_cast<double>(PW_UINT64_MAX))
              and "Incoherent energy resolution value.");

    uint64 l_resolution_value =
        static_cast<uint64>(pw_area::a_resolution_value);
    assert(l_resolution_value != 0);
    unsigned l_zero_cpt = 0;
    while((l_resolution_value % 10) == 0)
    {
        ++l_zero_cpt;
        l_resolution_value /= 10;
    }
    uint64 l_value = a_value;
    while((l_value % 10) == 0)
    {
        ++l_zero_cpt;
        l_value /= 10;
    }
    std::ostringstream l_os;
    l_os << l_value;

    pw_area_unit l_unit = PW_sqnm;

    while(l_zero_cpt >= g_tab_area_logs[l_unit+1] and l_unit < PW_sqm)
    {
        ++l_unit;
    }

    l_zero_cpt -= g_tab_area_logs[l_unit];

    for(unsigned l_cpt = 0;
        l_cpt < l_zero_cpt;
        ++l_cpt)
        l_os << "0";
    l_os << " " << g_tab_area_units[l_unit];

    return l_os.str();
}


// ---------------------------------------------------------------------------
uint64 pw_area::value(void) const
{
    return this->a_value;
}


// ---------------------------------------------------------------------------
double pw_area::to_double(void) const
{
    return static_cast< double >(this->a_value);
}


// ---------------------------------------------------------------------------
double pw_area::to_sqmeters(void) const
{
    return static_cast< double >(this->a_value) \
        * (pw_area::a_resolution_value / g_tab_area_scales[PW_sqm]);
}


// ---------------------------------------------------------------------------
pw_area pw_area::max(void)
{
    pw_area l_area(pw_get_area_resolution());
    l_area.a_value = PW_UINT64_MAX;
    return l_area;
}


// ---------------------------------------------------------------------------
bool operator==(const pw_area& p_area1,
                const pw_area& p_area2)
{
    return p_area1.a_value == p_area2.a_value;
}


// ---------------------------------------------------------------------------
bool operator!=(const pw_area& p_area1,
                const pw_area& p_area2)
{
    return p_area1.a_value != p_area2.a_value;
}


// ---------------------------------------------------------------------------
bool operator<(const pw_area& p_area1,
               const pw_area& p_area2)
{
    return p_area1.a_value < p_area2.a_value;
}


// ---------------------------------------------------------------------------
bool operator>(const pw_area& p_area1,
               const pw_area& p_area2)
{
    return p_area1.a_value > p_area2.a_value;
}


// ---------------------------------------------------------------------------
bool operator<=(const pw_area& p_area1,
                const pw_area& p_area2)
{
    return p_area1.a_value <= p_area2.a_value;
}


// ---------------------------------------------------------------------------
bool operator>=(const pw_area& p_area1,
                const pw_area& p_area2)
{
    return p_area1.a_value >= p_area2.a_value;
}


// ---------------------------------------------------------------------------
pw_area operator+(const pw_area& p_area1,
                    const pw_area& p_area2)
{
    pw_area l_area(p_area1);
    return l_area += p_area2;
}


// ---------------------------------------------------------------------------
pw_area operator-(const pw_area& p_area1,
                    const pw_area& p_area2)
{
    pw_area l_area(p_area1);
    return l_area -= p_area2;
}


// ---------------------------------------------------------------------------
pw_area operator*(const pw_area& p_area,
                    double p_value)
{
    pw_area l_area(p_area);
    return l_area *= p_value;
}


// ---------------------------------------------------------------------------
pw_area operator*(double p_value,
                    const pw_area& p_area)
{
    pw_area l_area(p_area);
    return l_area *= p_value;
}


// ---------------------------------------------------------------------------
double operator/(const pw_area& p_area1,
                 const pw_area& p_area2)
{
    return p_area1.to_double() / p_area2.to_double();
}


// ---------------------------------------------------------------------------
pw_area operator/(const pw_area& p_area,
                    double p_value)
{
    pw_area l_area(p_area);
    return l_area /= p_value;
}


// ---------------------------------------------------------------------------
std::ostream& operator<<(std::ostream& p_os,
                         const pw_area& p_area)
{
    std::ostream::sentry l_init(p_os);
    if(!l_init)
    {
        SC_REPORT_ERROR(PW_AREA_OSTREAM_ERROR_TYPE_,
                        PW_AREA_OSTREAM_ERROR_MSG_);
        return p_os;
    }

    if(p_os.flags()
       & (std::ios_base::scientific | std::ios_base::fixed))
        return p_os << p_area.to_sqmeters() << " "
                    << g_tab_area_units[PW_sqm];
    else
        return p_os << p_area.to_string();
}


// ---------------------------------------------------------------------------
std::istream& operator>>(std::istream& p_is,
                         pw_area& p_area)
{
    std::istream::sentry l_init(p_is,
                                false);
    if(!l_init)
    {
        SC_REPORT_ERROR(PW_AREA_ISTREAM_ERROR_TYPE_,
                        PW_AREA_ISTREAM_ERROR_MSG_);
        return p_is;
    }

    double l_value = 0;
    p_is >> l_value;
    if(not p_is)
    {
        SC_REPORT_ERROR(PW_AREA_ISTREAM_ERROR_TYPE_,
                        PW_AREA_ISTREAM_ERROR_MSG_);
        return p_is;
    }

    pw_area_unit l_unit;
    p_is >> l_unit;
    if(not p_is)
    {
        SC_REPORT_ERROR(PW_AREA_ISTREAM_ERROR_TYPE_,
                        PW_AREA_ISTREAM_ERROR_MSG_);
        return p_is;
    }

    p_area = pw_area(l_value,
                       l_unit);
    return p_is;

}


// ---------------------------------------------------------------------------
void pw_set_energy_resolution(double p_value,
                              pw_area_unit p_unit)
{
    // Sanity checks
    if(sc_core::sc_is_running())
    {
        SC_REPORT_ERROR(PW_AREA_RESOLUTION_ELABORATION_TYPE_,
                        PW_AREA_RESOLUTION_ELABORATION_MSG_);
        return;
    }
    if(pw_area::a_is_resolution_fixed)
    {
        SC_REPORT_ERROR(PW_AREA_RESOLUTION_FIXED_TYPE_,
                        PW_AREA_RESOLUTION_FIXED_MSG_);
        return;
    }

    if(p_value == 0)
    {
        SC_REPORT_ERROR(PW_AREA_RESOLUTION_ZERO_TYPE_,
                        PW_AREA_RESOLUTION_ZERO_MSG_);
        return;
    }

    if(p_value < 0)
    {
        SC_REPORT_ERROR(PW_AREA_RESOLUTION_NEGATIVE_TYPE_,
                        PW_AREA_RESOLUTION_NEGATIVE_MSG_);
        return;
    }

    if(p_unit < PW_sqnm or p_unit > PW_sqm)
    {
        SC_REPORT_ERROR(PW_AREA_UNIT_OVERFLOW_TYPE_,
                        PW_AREA_UNIT_OVERFLOW_MSG_);
        return;
    }

    if(not is_pow10(p_value))
    {
        SC_REPORT_ERROR(PW_AREA_RESOLUTION_POW10_TYPE_,
                        PW_AREA_RESOLUTION_POW10_MSG_);
        return;
    }

    double l_resolution_value = pw_rint(p_value * g_tab_area_scales[p_unit]);
    if(l_resolution_value < 1.0
       or l_resolution_value > (static_cast<double>(PW_UINT64_MAX)))
    {
        SC_REPORT_ERROR(PW_AREA_RESOLUTION_OVERFLOW_TYPE_,
                        PW_AREA_RESOLUTION_OVERFLOW_MSG_);
        return;
    }

    pw_area::a_resolution_value = l_resolution_value;
    pw_area::a_is_resolution_fixed = true;

    pw_info << "Resolution Fixed to: " << pw_area::a_resolution_value
            << " Femto Meter" << pw_endl;
}


// ---------------------------------------------------------------------------
pw_area pw_get_area_resolution(void)
{
    if(not pw_area::a_is_resolution_fixed)
        pw_info << "Resolution Fixed to: " << pw_area::a_resolution_value
                << " Femto Meter" << pw_endl;

    pw_area l_area;
    l_area.a_value = 1;
    pw_area::a_is_resolution_fixed = true;
    return l_area;
}

// ---------------------------------------------------------------------------
  // This simply rounds downwards.
  pw_area pw_area::round3sf(void) const
  {
    if (a_value == 0) return *this;
    sc_dt::uint64 v0 = a_value;
    sc_dt::uint64 v1 = a_value;
    sc_dt::uint64 v2 = a_value;
    int count = 0;
    while (true)
      {
	sc_dt::uint64 n = v0/10U;
	if (n == 0) break;
	count ++;
	v2 = v1; v1 = v0; v0 = n;
      }
    pw_area result(*this);
    result.a_value  = v2;
    for (int i = 0; i<count-2; i++) result.a_value *= 10;
    //std::cout << "Rounded " << *this << " to " << result << "\n";
    return result;
  }

// ---------------------------------------------------------------------------
pw_area operator*(const pw_length &a, const pw_length &b)
{
  double l_r = a.to_meters() * b.to_meters();
  //printf("l_tmp is %e\n", l_r);

  pw_area result(l_r, PW_sqm);
  return result;
}

} // namespace sc_pwr
