// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_power_map.h,v 1.2 2011/07/25 15:32:39 my294 Exp $

/** @file pw_power_map.h
 * @brief Associate power value to mode and phase.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 */

#ifndef _PW_POWER_MAP_
#define _PW_POWER_MAP_

#include <list>
#include <string>
#include <utility>
#include <map>
#include <systemc>
#include "pw_phase.h"
#include "pw_power.h"
#include "pw_energy.h"
#include "pw_voltage.h"
#include "pw_length.h"

// ===========================================================================
namespace sc_pwr
{

// ---------------------------------------------------------------------------
//! Class associating a power value to a power mode and phase.
class pw_power_map
{
    public: // CONSTRUCTOR
        /**
         * @brief Default constructor.
         * param[in] p_file_name input file name containing power information.
         */
        pw_power_map(const char* p_file_name);

        //! Default destructor.
        virtual ~pw_power_map(void);

    public: // ACCESS FUNCTION

        //! Informative string of this power map
        virtual std::string to_string(void) const;

 private: // PRIVATE CLASSES
        //! Key entry of a power map.
        struct map_key
        {
	  // Default constructor
	  map_key(const std::string& p_mode,
		  const std::string& p_phase);
	  //! Power mode value.
	  std::string a_mode;
	  //! Power phase value.
	  std::string a_phase;
        };

        //! Relational LESS THAN operator.
        friend bool operator<(const map_key&,
                              const map_key&);

        //! Relational EQUAL operator.
        friend bool operator==(const map_key&,
                               const map_key&);


    private: // PRIVATE ATTRIBUTES
        //! Configuration file name.
        std::string a_file_name;


	//! Map associating a power to a couple (power, mode).
	std::map< map_key,
	  std::pair< pw_power,
	  pw_power > > a_power_map;



    private: // PARSING METHODS
        //! Parse a line of a configuration file
        void parse_line(const std::string&,
                        unsigned);
        //! Parse the token string of the line of a configuration file.
        void parse_tokens(std::list< std::string >&,
                          unsigned, char);

        //! Convert the parameter string into power value.
        pw_power extract_power(const std::string&,
                               unsigned) const;

        //! Convert the parameter string into energy value.
        pw_energy extract_energy(const std::string&,
                               unsigned) const;

        //! Convert the parameter string into energy value.
        pw_voltage extract_voltage(const std::string&,
                               unsigned) const;

        //! Convert the parameter string into energy value.
        pw_length extract_length(const std::string&,
                               unsigned) const;
        public:
        const pw_voltage& get_nominal_vcc(void) const;
        const pw_power& get_scaled_static_power(void) const;

        /**
         * @brief Return the length of the module.
         * @return length value.
         * @note A report of severity SC_ERROR is thrown when their is no 
         * length value associated to the module.
         */
        const pw_length& get_length( void ) const;

        /**
         * @brief Return the width of the module.
         * @return width value.
         * @note A report of severity SC_ERROR is thrown when their is no 
         * width value associated to the module.
         */
        const pw_length& get_width( void ) const;

        /**
         * @brief Return the x-coordinate associated to the module.
         * @return x-coordinate value.
         * @note A report of severity SC_ERROR is thrown when their is no 
         * location value associated to the module.
         */
        const pw_length& get_loc_x( void ) const;

        /**
         * @brief Return the y-ccordinate associated to the module.
         * @return y-coordinate value.
         * @note A report of severity SC_ERROR is thrown when their is no 
         * location value associated to the module.
         */
        const pw_length& get_loc_y( void ) const;

        /**
         * @brief Predicate indicating if a power mode or phase is defined.
         * @param[in] p_mode the power mode.
         * @param[in] p_phase the power phase.
         */
        bool is_defined(const std::string& p_mode,
                        const std::string& p_phase = PW_PHASE_UNDEF) const;



    private: // DISABLED FUNCTIONS
        //! Disabled default constructor.
        pw_power_map(void);

        //! Disabled default copy constructor.
        pw_power_map(const pw_power_map&);

        //! Disabled assignment operator.
        pw_power_map& operator=(const pw_power_map&);

    protected:
        //! Size of module <length, width>
  //      std::pair< pw_length&, pw_length& >a_size;
        pw_length a_length;
        pw_length a_width;
        //! Location of module <x, y>
  //      std::pair< pw_length&, pw_length& >a_location;
        pw_length a_loc_x;
        pw_length a_loc_y;
        //! Nominal Vcc of module
        pw_voltage a_nominal_vcc;
        //! static power scaled to Vcc=1.0V one-bit module
        pw_power a_scaled_static_power;


}; // pw_power_map

// ---------------------------------------------------------------------------
// Output stream operator of a power map record.
std::ostream& operator<<(std::ostream&,
                         const pw_power_map&);
} // namespace sc_pwr


#endif // _PW_POWER_MAP_
