// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_power.h,v 1.1 2011/06/30 08:53:15 my294 Exp $

/** @file pw_power.h
 * @brief Define power data type.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 */

#ifndef _PW_POWER_
#define _PW_POWER_

#include <iostream>
#include <string>

#include <systemc>



// ===========================================================================
namespace sc_pwr
{


// ---------------------------------------------------------------------------
//! Standard physical power unit.
enum pw_power_unit
{
    //! Femto Watt = 10e-15
    PW_fW   = 0,
    //! Pico Watt = 10e-12
    PW_pW   = 1,
    //! Nano Watt  = 10e-9
    PW_nW   = 2,
    //! Micro Watt = 10e-6
    PW_uW   = 3,
    //! Milli Watt = 10e-3
    PW_mW   = 4,
    //! Watt = 10e0
    PW_WATT = 5,
};


// ---------------------------------------------------------------------------
// INCREMENT and DECREMENT OPERATORS

//! Prefix INCREMENT of power unit.
pw_power_unit& operator++(pw_power_unit&);
//! Postfix INCREMENT of power unit.
pw_power_unit operator++(pw_power_unit&,
                         int);
//! Prefix DECREMENT of power unit.
pw_power_unit& operator--(pw_power_unit&);
//! Postfix DECREMENT of power unit.
pw_power_unit operator--(pw_power_unit&,
                         int);


// ---------------------------------------------------------------------------
//! Output stream operator of power unit object.
std::ostream& operator<<(std::ostream&,
                         const pw_power_unit&);


// ---------------------------------------------------------------------------
//! Intput stream operator of power unit object.
std::istream& operator>>(std::istream&,
                         pw_power_unit&);


// ---------------------------------------------------------------------------
/**
 * @brief Represent a power value using the standard unit (WATT).
 * @note Arithmetic operations generate a report of severity PW_WARNING or 
 * PW_ERROR (division by zero) when an overflow is detected.  In this case it 
 * is initialized to 0 or to the max power value (according to the performed 
 * arithmetic operation).
 */
class pw_power
{
    public: // CONSTRUCTORS
        /** @brief Default constructor.
         * @note
         * - Initialized to a value of PW_ZERO_WATT.
         */
        pw_power(void);

        /**
         * Constructor of a power value.
         * @param[in] p_value Raw power value.
         * @param[in] p_unit Power unit multiple of the raw value.
         * @pre The parameter power value should be equal to 0, or >= to the 
         * power resoultion and <= to the max power value, otherwise a report 
         * of severity PW_WARNING or PW_ERROR is thrown.  In this last case, 
         * the constructed power value will be equal to 0 or to the maximal 
         * power value (according to the parameters value).
         */
        pw_power(double p_value,
                 pw_power_unit p_unit);

        //! Copy constructor.
        pw_power(const pw_power&);

        //! Assignment operator.
        pw_power& operator=(const pw_power&);

    public:  // ARITHMETIC ASSIGNMENT OPERATORS
        //! Assignment by SUM.
        pw_power& operator+=(const pw_power&);
        //! Assignment by DIFFERENCE.
        pw_power& operator-=(const pw_power&);
        //! Assignment by MULTIPLICATION.
        pw_power& operator*=(double);
        //! Assignment by DIVISION.
        pw_power& operator/=(double);

        //! Prefix INCREMENT.
        pw_power& operator++(void);
        //! Postfix INCREMENT.
        pw_power operator++(int);
        //! Prefix DECREMENT.
        pw_power& operator--(void);
        //! Postfix DECREMENT
        pw_power operator--(int);

    private: // RELATIONAL OPERATORS
        //! Relational EQUAL operator.
        friend bool operator==(const pw_power&,
                               const pw_power&);
        //! Relational NOT EQUAL operator.
        friend bool operator!=(const pw_power&,
                               const pw_power&);
        //! Relational LESS THAN operator.
        friend bool operator<(const pw_power&,
                               const pw_power&);
        //! Relational GREATER THAN operator.
        friend bool operator>(const pw_power&,
                               const pw_power&);
        //! Relational LESS THAN OR EQUAL operator.
        friend bool operator<=(const pw_power&,
                               const pw_power&);
        //! Relational GREATER THAN OR EQUAL operator.
        friend bool operator>=(const pw_power&,
                               const pw_power&);
    
    public: // MISCELLANOUS METHODS
	//! Round to two three sig figures
	pw_power round3sf() const;

        //! Return a string representing its pysical value.
        const std::string to_string(void) const;

        //! Return it raw value (multiple of the power resolution).
        sc_dt::uint64 value(void) const;

        //! Return it raw value (multiple of the power resolution).
        double to_double(void) const;

        //! Return its value in watts
        double to_watts(void) const;

        //! Return the maximal power value.
        static pw_power max(void);

    private: // MSICELLANOUS OPERATORS AND FUNCTIONS
        //! Set the power resolution
        friend void pw_set_power_resolution(double,
                                            const pw_power_unit);

        //! Return the current power resolution.
        friend pw_power pw_get_power_resolution(void);

    private: // PRIVATE MENBERS
        //! Current power value.
        sc_dt::uint64 a_value;

        //! Power resolution in femto watt.
        static double a_resolution_value;
        //! True if a pw_power different of 0 have been construted.
        static bool a_is_resolution_fixed;
}; // class pw_power


// ---------------------------------------------------------------------------
//! Relational EQUAL operator.
bool operator==(const pw_power&,
                const pw_power&);

// ---------------------------------------------------------------------------
//! Relational NOT EQUAL operator.
bool operator!=(const pw_power&,
                const pw_power&);

// ---------------------------------------------------------------------------
//! Relational LESS THAN operator.
bool operator<(const pw_power&,
               const pw_power&);

// ---------------------------------------------------------------------------
//! Relational GREATER THAN operator.
bool operator>(const pw_power&,
               const pw_power&);

// ---------------------------------------------------------------------------
//! Relational LESS THAN OR EQUAL operator.
bool operator<=(const pw_power&,
                const pw_power&);

// ---------------------------------------------------------------------------
//! Relational GREATER THAN OR EQUAL operator.
bool operator>=(const pw_power&,
                const pw_power&);

// ---------------------------------------------------------------------------
//! ADDITION operator.
pw_power operator+(const pw_power&,
                   const pw_power&);

// ---------------------------------------------------------------------------
//! SUBSTRACTION operator.
pw_power operator-(const pw_power&,
                   const pw_power&);

// ---------------------------------------------------------------------------
//! MULTIPLICATION with constant operator.
pw_power operator*(double,
                   const pw_power&);

// ---------------------------------------------------------------------------
//! MULTIPLICATION with constant operator.
pw_power operator*(const pw_power&,
                   double);

// ---------------------------------------------------------------------------
//! DIVISION operator.
double operator/(const pw_power&,
                 const pw_power&);

// ---------------------------------------------------------------------------
//! DIVISION by constant operator.
pw_power operator/(const pw_power&,
                   double);


// ---------------------------------------------------------------------------
// Output stream operator of power object.
std::ostream& operator<<(std::ostream&,
                         const pw_power&);

// ---------------------------------------------------------------------------
// Input stream operator of power object.
std::istream& operator>>(std::istream&,
                         pw_power&);


// ---------------------------------------------------------------------------
/**
 * @brief Set the power resoultion.
 * @param[in] p_value Raw value of the power resoultion.
 * @param[in] p_unit Power unit of the power resolution.
 * @pre This function have to be called during the elaboration phase, 
 * otherwise a report of severity PW_ERROR is thrown.
 * @pre This function can only be called one time, otherwise a report of 
 * severity PW_ERROR is thrown.
 * @pre This function can only be called before any pw_power different from 0 
 * are constructed, otherwise a report of severity PW_ERROR is thrown.
 * @pre The p_value value should be a power of 10, otherwise a report of 
 * severity PW_WARNING is thrown.
 * @note The default power resolution is equal to 1 pico watt. 
 */
void pw_set_power_resolution(double p_value,
                             pw_power_unit p_unit);


// ---------------------------------------------------------------------------
//! Return the current power resolution.
pw_power pw_get_power_resolution(void);


// ---------------------------------------------------------------------------
//! Constant for representing a power of 0 watt.
extern const pw_power PW_ZERO_POWER;


} // namespace sc_pwr


#endif // _PW_POWER_
