// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_accounting_base.h,v 1.2 2011/07/25 15:32:39 my294 Exp $

/** @file pw_accounting_base.h
 * @brief Implements the main account type that holds power and energy.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 *
 * This class is used by the TXT and CSV trace file for generating stastics on 
 * the power variation.
 */

#ifndef _PW_ACCOUNTING_BASE_
#define _PW_ACCOUNTING_BASE_

#include <list>
#include <string>
#include <utility>
#if __GNUC__ == 4
#   include <tr1/unordered_map>
#else
#   include <map>
#endif

#include <systemc>
#include "pw_power.h"
#include "pw_energy.h"
#include "pw_subject.h"
#include "pw_module_base.h"
#include "pw_observer_base.h"

// ===========================================================================
namespace sc_pwr
{



// ---------------------------------------------------------------------------
/**
 * @brief Record power and energy consumption using accounts
 */
  class pw_accounting_base:
  public pw_observer_base
{

    public: // CONSTRUCTOR AND DESTRUCTOR
        /**
         * @brief Constructor of a statistic observer.
         * @param[in] p_obj SystemC object monitored by this observer.
         * @param[in] p_name SystemC attribute name.
         */
        pw_accounting_base(sc_core::sc_object& p_obj,
                              const std::string& p_name);

        // virtual destructor
        virtual ~pw_accounting_base(void);


	// Destructor prologue
	void flush_and_close(sc_time);

    public: // UPDATE FUNCTIONS
        //! Update the account's energy based on elapsed time and baseline power.
        void update_acct_energy(int acct, const sc_core::sc_time&);


    public: // ACCESS FUNCTIONS
        //! Return true if all accounts have no energy logged.
	bool all_accounts_empty();

        //! Return whether account is in use.
	bool inuse(int) const;
	

        //! Return the last update time for an account.
        const sc_core::sc_time get_last_updatet(int p_acct) const;

        //! Return the last update time (and fold energy of all accounts to this point)
        const sc_core::sc_time get_last_updatet();

	const std::string get_name(int acct);

        //! Return the current mode/phase power - all accounts. Meaningless? 
        const pw_power get_power() const;

	//! Power - specific account
        const pw_power get_power(int acct) const;

        //! Return the energy consumed until this date.
        pw_energy get_energy(int acct);

        //! Return the energy consumed until this date - all accounts.
        pw_energy get_energy();


        //! Check is account is in use, globally.
        static bool global_inuse(int);

	//! Return number of accounts in use.
	static int global_n_accounts_in_use();

        //! Return account name
	static std::string get_global_name(int acct);

	//! Install account names for the first n standard accounts (if none already started).
	void start_default_accounts(int n=3);


 protected: // PRIVATE ATTRIBUTES

	friend class global_power;
        //! Energy accounts.
	class acct
	{
	public: 
	  void fold_power_to_energy(sc_time);

	  void update_power(const sc_time when, const pw_power& inc, const pw_power& dec);

	  void record_energy(pw_energy &new_energy);

	  void end_of_simulation(sc_time when);

	  pw_accounting_base *m_parent;
	  //! The name of this account
	  std::string m_name;
	  //! The energy in the account - this is the primary data of the account.
	  pw_energy m_energy; 
	  //! Steady-state (phase-based) power consumption being charged to this account.
	  pw_power m_phasemode_power;
	  //! Time of the last energy bill 
	  sc_time m_last_baseline_update;

	  // Return updated energy.
	  pw_energy get_energy() const;

	  sc_time get_last_update_at() const { assert(inuse()); return m_last_baseline_update; }

	  void start(pw_accounting_base *parent, const char *name);
	  acct() // constructor
	    {
	      m_parent = 0; // Not yet in use
	      m_name = "not-in-use";
	    }

	  bool inuse() const { return m_parent != 0; }
	  void flush_and_close(sc_time);
	  void update_acct_energy(const sc_core::sc_time& p_time);
	};

	acct a_accts[PW_ACCT_MAX_NO_ACCOUNTS];

	int start_acct(const char *name);

	int lookup_acct(const char *name);

 protected:
	void auto_start_acct(int act); 
	void create_global(); 
	
	
 private: // DISABLED FUNCTIONS
        //! DISABLE default constructor
        pw_accounting_base(void);
        //! DISABLE copy constructor.
        pw_accounting_base(const pw_accounting_base&);
        //! DISABLE assignment operator.
        pw_accounting_base& operator=(const pw_accounting_base&);


	
 }; // pw_accounting_base



} // namespace sc_pwr


#endif // _PW_ACCOUNTING_BASE_
