// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $


/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_stat_observer_base.cpp,v 1.3 2011/07/25 15:32:45 my294 Exp $

/** @file pw_stat_observer_base.cpp
 * @brief Makes statistic on power variation.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 *
 * This class is used by the TXT and CSV trace file for generating statistics re power/energy use.
 */

#include <sstream>
#include <systemc>
#include "pw_stat_observer_base.h"
#include "pw_observer_ids.h"
#include "pw_tracing_ids.h"
#include "pw_physical_operators.h"
#include "pw_debug.h"
#include "pw_common.h"
#include "pw_trace.h"
// ===========================================================================
using std::string;
using sc_core::sc_time;
using sc_core::SC_ZERO_TIME;
using sc_core::sc_time_stamp;
using namespace std;

// ===========================================================================
namespace
{

// ---------------------------------------------------------------------------
using namespace sc_pwr;

// ---------------------------------------------------------------------------
//! Macros definition of the maximal time value.
#define SC_TIME_MAX sc_time_max()

// ---------------------------------------------------------------------------
//! Indicate that at least one statistic observer was created.
static bool g_is_stat_initialized = false;

  
// ---------------------------------------------------------------------------
//! Store the global power of the system.
//! All leaf power measurements are accumulated here. We have our own instance of this if the user does not create one.
  class global_power: public pw_stat_observer_base
  {
  public: // CONSTRUCTOR
        /**
         * @brief Constructor of the global power management object. 
	 * @brief The object pointer is a dummy SC_MODULE used to receive flush callbacks.
         * @param[in] p_obj SystemC object monitored by this observer. 
         * @param[in] p_name SystemC attribute name.
         */
        global_power(sc_core::sc_object& p_obj,
                     const string& p_name);


  public: // ACCESS FUNCTION
    
    // Return the last updating time
    //const sc_time get_last_update(); use that of base class

        //! Attach this observer to all the pw_module_bases
        void attach_to_all(void);
        //! Indicate if this observer was attached.
        bool is_attached(void) const;


    private: // ATTRIBUTES
        //! Indicate if this observer was attached.
        bool a_is_attached;
}; // global_power


//! Reference to singleton object which stores the global power.
static global_power* g_global_power_pt = NULL;

  // Constructor - is attached to the SC_GLOBAL_POWER_CALLBACK dummy module.
  global_power::global_power(sc_core::sc_object& p_obj,
			     const string& p_name):
    pw_stat_observer_base(p_obj, PW_STAT_OBSERVER_ATTRIBUTE_ID_CHILD_SUMMED),
    a_is_attached(false)
  {
    // XXX NEEDED for proper management of simulation end.
    if(not p_obj.add_attribute(*this))
      {
	pw_error << "Error with observer " << p_name 
		 << " observing module " << p_obj.name() << pw_endl;
	SC_REPORT_FATAL(PW_OBSERVER_REDEFINITION_TYPE_,
			PW_OBSERVER_REDEFINITION_MSG_);
      }
  }

  // ---------------------------------------------------------------------------
  // This global monitor an observer that observes all power records (i.e. all relevant pw_subjects - which is all of them at the moment?).
  void global_power::attach_to_all(void)
  {
    std::list<pw_subject *>::iterator l_it;
    for (l_it = pw_subject::g_all_pw_subjects.begin(); l_it != pw_subject::g_all_pw_subjects.end(); ++l_it) 
      {
	const char *l_aid = PW_STAT_OBSERVER_ATTRIBUTE_ID_SINGLE_MODULE;
	(*l_it)->attach(*this, l_aid);
      }

    // OLD WAY (missed those that would not dynamic_cast correctly) attach_observer(*this, 0, true);
    a_is_attached = true;
  }



// ---------------------------------------------------------------------------
bool global_power::is_attached(void) const
{
    return a_is_attached;
}

// ---------------------------------------------------------------------------
//! SystemC callback for creating the global power object at start of simulation.
class sc_global_power_callback:
    public sc_core::sc_module
{
    public: // CONSTRUCTOR
  /**
         * @brief Call back constructor.
         * @param[in] p_name name of the call back.
         */
        sc_global_power_callback(const sc_core::sc_module_name& p_name);

    public: // SYSTEMC CALLBACK
        //! SystemC callback for checking coherency of power values.
        virtual void start_of_simulation(void);
}; // sc_global_power_callback

// ---------------------------------------------------------------------------
sc_global_power_callback::sc_global_power_callback(
    const sc_core::sc_module_name& p_name):
    sc_core::sc_module(p_name)
{ }

// ---------------------------------------------------------------------------

void sc_global_power_callback::start_of_simulation(void)
{
    pw_assert(g_global_power_pt == NULL
              and "Undefined reference to the global power observer.");

    if(g_is_stat_initialized)
    {
      sc_core::sc_attr_base *l_already = get_attribute(PW_STAT_OBSERVER_ATTRIBUTE_ID_CHILD_SUMMED);
      g_global_power_pt = l_already ? (global_power *)l_already : new global_power(*this, "GLOBAL_POWER");
      g_global_power_pt->attach_to_all();
    }
    else
    {
        g_is_stat_initialized = true;  // For skiping the initialization
        g_global_power_pt = new global_power(*this, "GLOBAL_POWER");
        g_is_stat_initialized = false;
    }
}

// ---------------------------------------------------------------------------
// XXX NOTE: Aggresive G++ optimization request to use dynamic instantiation 
// of the global power SystemC call back.
//
//! SystemC call back for computing the global energy
static sc_core::sc_object& g_sc_global_power_callback =
    *(new sc_global_power_callback(
           sc_core::sc_gen_unique_name("SC_GLOBAL_POWER_CALLBACK")));

} // ANONYMOUS NAMESPACE (replace static declaration)

// ===========================================================================
namespace sc_pwr
{

  // User can change the name of this file by assigning his own string to this global.
  const char *phy_report_name = "physical.txt";


// ---------------------------------------------------------------------------
/**
 * @brief SystemC callback for flushing the power value at the end of 
 * the simulation.
 */
class sc_power_flush_callback:
    public sc_core::sc_module
{
    public: // CONSTRUCTOR
        /**
         * @brief Call back constructor.
         * @param[in] p_name name of the call back.
         */
        sc_power_flush_callback(const sc_core::sc_module_name& p_name);

    public: // SYSTEMC CALLBACK
        //! SystemC callback for checking coherency of power values.
        virtual void end_of_simulation(void);

        //! recursive flushing functions
        void end_of_simulation_rec(sc_core::sc_object&);
}; // sc_power_flush_callback


// ---------------------------------------------------------------------------
// XXX NOTE: Aggresive G++ optimization request to use dynamic instantiation 
// of the global power SystemC call back.
//
//! SystemC call back for computing the global energy
static sc_power_flush_callback& g_sc_power_flush_callback =
    *(new sc_power_flush_callback(
           sc_core::sc_gen_unique_name("SC_POWER_FLUSH_CALLBACK")));

// ---------------------------------------------------------------------------
  bool pw_stat_observer_base::g_lock_warning = false;

// ---------------------------------------------------------------------------


  // Constructor.
  pw_stat_observer_base::pw_stat_observer_base(sc_core::sc_object& p_obj,
                                             const std::string& p_attr_name):
    pw_accounting_base(p_obj, p_attr_name),
    a_is_dumped(false),
    a_is_simulation_terminated(false),
    a_utilization(SC_ZERO_TIME)
  {
    if(g_is_stat_initialized) return;
    g_is_stat_initialized = true;
 

    start_default_accounts();

    if(g_global_power_pt != NULL)
    {
        pw_assert(not g_global_power_pt->is_attached()
                  and "Incohernt status of simulation status variable.");
        g_global_power_pt->attach_to_all();
    }
  }

// ---------------------------------------------------------------------------
  pw_stat_observer_base::~pw_stat_observer_base(void)
  {
    pw_stat_observer_base::destructor_prologue();
    a_is_simulation_terminated = true;
    
    // XXX Informs the trace file using this observers of its destruction.
    if (not a_is_dumped)
      { assert(0); // using notify for this is the wrong direction this->notify();  
      }
    a_is_dumped = true;
  }
  

  // ---------------------------------------------------------------------------
  void pw_stat_observer_base::update(pw_subject& p_subject)
  {
    //std::cout << "update observer " << ((void *)this) << "\n";
    try
    {
        const pw_module_base& l_module = dynamic_cast< const pw_module_base& >(p_subject);

        switch(l_module.get_modified_attribute())
        {
            case pw_module_base::INIT:
                pw_debug << "Initialization of statistic observer <"
                         << get_obj_name() << ", "
                         << get_att_name() << ">" << pw_endl;
                update_energy_infos(l_module);
                break;


            case pw_module_base::PHASEMODE_POWER:
                pw_debug << "Update Phasemode Energy of statistic observer <"
                         << get_obj_name() << ", "
                         << get_att_name() << ">" << pw_endl;
                pw_debug << "tmp debug Update Phasemode Energy of statistic observer <"
                         << get_obj_name() << ", "
                         << get_att_name() << ">" << pw_endl;
                update_energy_infos(l_module);
                break;

            case pw_module_base::TRANSACTION:
                pw_debug << "Update transaction energies of statistic observer <"
                         << get_obj_name() << ", "
                         << get_att_name() << ">" << pw_endl;
                update_energy_infos(l_module);
                update_transaction_infos(l_module);
                break;

            case pw_module_base::QUANTUM_UP:
                pw_debug << "Quantum Up of statistic observer <"
                         << get_obj_name() << ", "
                         << get_att_name() << ">" << pw_endl;
 //               update_quantum_infos(l_module);
                break;
            case pw_module_base::LOG_EVENT:
                pw_debug << "Log dynamic and static energy statistic observer <"
                         << get_obj_name() << ", "
                         << get_att_name() << ">" << pw_endl;
//                update_log_infos(l_module);
                break;
            default:
                pw_assert(false
                          and "Incoherent call of update.");
                break;
        }
    }
    catch(std::bad_cast)
    {
        pw_assert(false
                  and "Casting problem with the \"Observer\" design pattern");
    }
}


// ---------------------------------------------------------------------------

#if 0
  const sc_core::sc_time pw_stat_observer_base::get_last_update(int p_acct)
  {
    if(not a_is_simulation_terminated
       and sc_time_stamp() == SC_TIME_MAX)
      {
        check_end_of_simulation();
        pw_stat_observer_base::end_of_simulation();
        a_is_simulation_terminated = true;
      }
    
    return a_accts[p_acct].get_last_update();
  }
#endif

// ---------------------------------------------------------------------------
bool pw_stat_observer_base::is_simulation_terminated(void) const
{
    return a_is_simulation_terminated;
}

  



  // ---------------------------------------------------------------------------
  pw_energy pw_stat_observer_base::get_global_energy()
  {
    pw_energy l_sum = PW_ZERO_ENERGY;
    if (!g_global_power_pt) return l_sum;
    
    for (int a=0;a<PW_ACCT_MAX_NO_ACCOUNTS;a++)
      if (g_global_power_pt->a_accts[a].inuse()) l_sum += g_global_power_pt->a_accts[a].get_energy();
    
    return l_sum;
  }


pw_energy pw_stat_observer_base::get_global_energy(int act)
{
    if (!g_global_power_pt) return PW_ZERO_ENERGY;

    return g_global_power_pt->get_energy(act);
}

// ---------------------------------------------------------------------------

const sc_time pw_stat_observer_base::get_global_last_update()
{
    if(g_global_power_pt == NULL)
        return SC_ZERO_TIME;

    return g_global_power_pt->get_last_updatet();
}




// ---------------------------------------------------------------------------
string pw_stat_observer_base::to_string(void) const
{
    std::ostringstream l_os;
    assert(0);
    l_os << "{ " << a_obj.name()
#if 0
      TODO
         << ", ( DYN, " << a_dynamic_energy
         << ", " << a_last_dynamic_update
         << " ), ( STAT, " << get_power()
         << ", " << a_static_energy
         << ", " << a_last_static_update
#endif
         << " ) }";
    return l_os.str();
}


// ---------------------------------------------------------------------------


//! Update transaction busytime/utilisation.
  void pw_stat_observer_base::update_transaction_infos(const pw_module_base& p_module)
  {
    sc_time l_busytime = p_module.get_transaction_delay();
    a_utilization += l_busytime;
  }




  //! Update phase/mode power - read transaction event from pw_module base.
#if 0
  void pw_stat_observer_base::update_power_infos(const pw_module_base& p_module)
  {
    sc_time when = sc_time_stamp();
    int l_acct = p_module.get_transaction_acct();
    pw_power l_old_power = p_module.get_old_phasemode_power();
    pw_power l_power = p_module.get_phasemode_power(l_acct);
    pw_power inc = PW_ZERO_POWER, dec = PW_ZERO_POWER;
    if(l_power > l_old_power) inc = l_power - l_old_power;
    else if(l_power < l_old_power) dec = l_old_power - l_power;
    pw_debug << get_obj_name() << " tmp debug update_power_infos power=" << l_power  <<  " acct=" << l_acct << pw_endl;
    auto_start_acct(l_acct);
    a_accts[l_acct].update_power(when, inc, dec);
  }
#endif

  //! Update transaction energy/power - read transaction event from pw_module base.
  void pw_stat_observer_base::update_energy_infos(const pw_module_base& p_module)
  {
    pw_energy l_energy = p_module.get_transaction_energy();
    if (l_energy == PW_ZERO_ENERGY) return;
    int l_acct = p_module.get_transaction_acct();
    auto_start_acct(l_acct);
    pw_debug << "Record energy acct=" << l_acct << ", e=" << l_energy << "  "  << p_module << ">" << pw_endl;
    a_accts[l_acct].record_energy(l_energy);
  }


  // ---------------------------------------------------------------------------


void pw_stat_observer_base::end_of_simulation(void)
{
    sc_time when = sc_time_stamp();
    pw_info << when << ": End of Simulation of the statistic observer <"
	    << get_obj_name() << ", " << get_att_name() << ">" << pw_endl;
    
    sc_time l_when = sc_time_stamp();
    flush_and_close(l_when);  // TODO repeated - by the next bit of code


    //     if (this != g_global_power_pt) -- global perhaps needs to be done last?
      {
	for (int a=0;a<PW_ACCT_MAX_NO_ACCOUNTS;a++)
	  if (a_accts[a].inuse()) a_accts[a].end_of_simulation(when);
      }
    
}



// ---------------------------------------------------------------------------
void pw_stat_observer_base::check_end_of_simulation(void)
{
  if(not a_is_simulation_terminated
     and not pw_stat_observer_base::g_lock_warning)
    {
      // "Reached end of infinite simulation without calling sc_stop.";
      SC_REPORT_WARNING(PW_POWERED_OBSERVER_END_ACTIVE_TYPE_,
			PW_POWERED_OBSERVER_END_ACTIVE_MSG_);
    }
  pw_stat_observer_base::g_lock_warning = true;
}



void pw_stat_observer_base::destructor_prologue(void)
{
  check_end_of_simulation();
  //  pw_module_base::flush_all_modules();
  if(not a_is_simulation_terminated)
    {
      pw_debug << "End of Simulation of the statistic observer <"
	       << get_obj_name() << ", " << get_att_name()
                 << ">" << pw_endl;
      
      sc_time l_o = get_global_last_update();
      sc_time l_eos = (sc_time_stamp() != SC_TIME_MAX && sc_time_stamp() > l_o) ? sc_time_stamp(): l_o;
      pw_debug << "l_eos = " << l_eos << "\n";
      for (int a=0; a<PW_ACCT_MAX_NO_ACCOUNTS; a++) if (a_accts[a].inuse()) a_accts[a].fold_power_to_energy(l_eos);
    }
}



// ---------------------------------------------------------------------------
sc_power_flush_callback::sc_power_flush_callback(
        const sc_core::sc_module_name& p_name):
  sc_core::sc_module(p_name)
{ }


// ---------------------------------------------------------------------------

// Walk over all callbacks at end of simulation.

  void sc_power_flush_callback::end_of_simulation(void)
  {
    // TODO : provide name for this report.
    pw_debug << "Creating physical summary file " << sc_pwr::phy_report_name << "\n";
    std::fstream *osp = new std::fstream(sc_pwr::phy_report_name, std::ios_base::out | std::ios_base::trunc);

    pw_module_base::flush_all_modules(0, osp);
    osp->flush();
    // 
    // TODO: This is a second full walk - using a different iteration technique!
    const std::vector< sc_core::sc_object* >& l_objects =
        sc_core::sc_get_top_level_objects();
    typedef
        std::vector< sc_core::sc_object* >::const_iterator
        vect_const_iterator;
    for(vect_const_iterator l_it = l_objects.begin();
        l_it != l_objects.end();
        ++l_it)
        end_of_simulation_rec(**l_it);
}



void sc_power_flush_callback::end_of_simulation_rec(sc_core::sc_object& p_obj)
{
    sc_core::sc_attr_base* l_attr_pt = NULL;
    if((l_attr_pt = p_obj.get_attribute(PW_STAT_OBSERVER_ATTRIBUTE_ID_CHILD_SUMMED))
       != NULL)
    {
        try
        {
            pw_stat_observer_base* l_observer_pt =
                dynamic_cast< pw_stat_observer_base* >(l_attr_pt);
            l_observer_pt->end_of_simulation();
            l_observer_pt->a_is_simulation_terminated = true;
        }
        catch(std::bad_cast)
        {
            pw_assert(false
                      and "FATAL casting of a STAT observer");
        }
    }

    // Recursively notify the children of this object.
    const std::vector< sc_core::sc_object* >& l_tab_objs =
        p_obj.get_child_objects();
    for(std::vector< sc_core::sc_object* >::const_iterator l_it =
        l_tab_objs.begin();
        l_it != l_tab_objs.end();
        ++l_it)
        end_of_simulation_rec(**l_it);  // XXX Recursive CALL
}

  // ---------------------------------------------------------------------------
  // Output stream operator of energy object.
  std::ostream& operator<<(std::ostream& p_os,
			   const pw_stat_observer_base& p_observer)
  {
    std::ostream::sentry l_init(p_os);
    if(!l_init)
      {
        SC_REPORT_ERROR(PW_TRACING_OSTREAM_ERROR_TYPE_,
                        PW_TRACING_OSTREAM_ERROR_MSG_);
        return p_os;
      }
    
    return (p_os << p_observer.to_string());
  }


  // C++ API convenience function
  pw_stat_observer_base *pw_get_observer(sc_module *p_mod, sc_pwr::trace_t p_do_children)
  {
    return pw_get_observer((sc_object *)p_mod, p_do_children);
  }


  // C++ API convenience function
  pw_stat_observer_base *pw_get_observer(sc_object *p_mod, sc_pwr::trace_t p_do_children)
  {
    pw_assert(p_mod);
    const char *l_aid = (p_do_children == sc_pwr::sum_children) ? PW_STAT_OBSERVER_ATTRIBUTE_ID_CHILD_SUMMED: PW_STAT_OBSERVER_ATTRIBUTE_ID_SINGLE_MODULE;
    pw_stat_observer_base *r = (pw_stat_observer_base *)(p_mod->get_attribute(l_aid));
    if (!r)
      {

	// If no observer already (attach a stat_observer to this component if it is a module, otherwise just add an observer base).
	
	try
	  {
	    //pw_module_base& p_module =  dynamic_cast< pw_module_base& >(p_obj);
	    // Actually - only using bases in this release.
	    //l_observer_pt = new pw_stat_observer(p_module, l_aid);
	    r = new pw_stat_observer_base(*p_mod, l_aid);
	  }
        catch(std::bad_cast)
	  {
            r = new pw_stat_observer_base(*p_mod, l_aid);
	  }
	
        // [NOTE] Attach this observer to the power module and optionally to its children.
	attach_observer(*r, p_mod, p_do_children==sum_children);
      }
    return r;
  }


} // namespace sc_pwr
