// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_slk_trace.cpp,v 1.2 2011/07/25 15:32:45 my294 Exp $

/** @file pw_slk_trace.cpp
 * @brief Tracing facilities for generating statistics on power consumption.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 */

#include <cerrno>
#include <string>
#include <vector>
#include <utility>
#include <fstream>
#include <sstream>
#include <algorithm>

#include "pw_slk_trace.h"
#include "pw_observer_ids.h"
#include "pw_tracing_ids.h"
#include "pw_power.h"
#include "pw_energy.h"
#include "pw_physical_operators.h"
#include "pw_debug.h"



// ===========================================================================
namespace
{


// ---------------------------------------------------------------------------
// GLOBAL CONSTANT

#define _ENERGY_FORMAT_ "0.000E+00j"
#define _POWER_FORMAT_ "0.000E+00w"
#define _RATIO_FORMAT_ "0.000%"

static const char FILE_HEADER[] =
    "ID;TLM_POWER;N;E\n"
    "P;PGeneral\n"              // 0: Default Format (string)
    "P;P" _RATIO_FORMAT_ "\n"     // 1: Ratio Format
    "P;P" _ENERGY_FORMAT_ "\n"    // 2: Energy Value Format
    "P;P" _POWER_FORMAT_ "\n"     // 3: Power Value Format
    "P;FArial;M200\n"           // 1: String Font
    "P;FArial;M200;SB\n"        // 2: Bold String Font
    "P;EArial;M200\n"           // 3: Expression Font
    "P;EArial;M200;SB\n";       // 4 : Bold Expression Font

  static const unsigned TOT_MODULE_NAME_COL = 1;
  static const unsigned TOT_ENERGY_COL = 2;
  static const unsigned TOT_ACCT_COL = 3;
  static const unsigned TOT_COLS_PER_ACCT = 2;

  static const unsigned TOT_ACCT_ENERGY_COL = 0;
  static const unsigned TOT_ACCT_POWER_COL = 1;


  static const unsigned ACCT_COL_BASE = 1;
  static const unsigned COLS_PER_ACCT = 4;
  static const unsigned POWER_COL = 0;
  static const unsigned ENERGY_COL = 1;
  static const unsigned POWER_RATIO_COL = 2;
  static const unsigned ENERGY_RATIO_COL = 3;

  static std::string COL_LEGENDS[ACCT_COL_BASE + COLS_PER_ACCT * PW_ACCT_MAX_NO_ACCOUNTS];

  static const unsigned EXT_MODULE_NAME_COL = 1;
  static const unsigned EXT_MODE_NAME_COL = 2;
  static const unsigned EXT_PHASE_NAME_COL = 3;
  static const unsigned EXT_ACCT_COL_BASE = 4;
  static const unsigned EXT_COLS_PER_ACCT = 4;

static const unsigned EXT_ACCT_E_COL = 0;
static const unsigned EXT_ACCT_P_COL = 1;
static const unsigned EXT_ACCT_E_RATIO_COL = 2;
static const unsigned EXT_ACCT_P_RATIO_COL = 3;

  static std::string EXT_COL_LEGENDS[EXT_ACCT_COL_BASE + PW_ACCT_MAX_NO_ACCOUNTS * EXT_COLS_PER_ACCT];

static const char STRING_FORMAT_HEADER[] = "F;P0;FG0";
static const char STRING_FORMAT_TAILER[] = ";SM0;";
static const char BOLD_STRING_FORMAT_HEADER[] = "F;P0;FG0";
static const char BOLD_STRING_FORMAT_TAILER[] = ";SM1;";
static const char ENERGY_FORMAT[] = "F;P2;FE2R;SM2;";
static const char POWER_FORMAT[] = "F;P3;FE2R;SM2;";
static const char RATIO_FORMAT[] = "F;P1;F%2R;SM2;";

static unsigned COLUMN_NB =
    std::max(sizeof(COL_LEGENDS)/sizeof(char*),
             sizeof(EXT_COL_LEGENDS)/sizeof(char*));
static const unsigned COLUMN_MAX_SIZE = 30;

static const unsigned EXTRA_BLANK_LINES = 6;


// ---------------------------------------------------------------------------
//! Replace the char " by "" (" is an escape char in sylk).
static std::string string_normalize(const std::string& p_name)
{
    std::string l_res("");

    for(std::string::const_iterator l_it = p_name.begin();
        l_it != p_name.end();
        ++l_it)
    {
        l_res += *l_it;
        if(*l_it == '"') l_res += '"';
    }

    return l_res;
}


} // ANONYMOUS NAMESPACE (replace static declaration)


// ===========================================================================
namespace sc_pwr
{


// ---------------------------------------------------------------------------
  pw_slk_trace_file::pw_slk_trace_file(const char* p_name_pt):
    pw_observer_if(),
    pw_trace_file(),
    a_file((std::string(p_name_pt)+".slk").c_str(),
           std::ios_base::out | std::ios_base::trunc),
    a_file_name(p_name_pt),
    a_observers_pt(),
    a_observers_name(),
    a_update_called(false),
    a_extra_line(EXTRA_BLANK_LINES),
    a_columns_size(COLUMN_NB, 0),
    a_is_ext_legend_dumped(false)
{
    pw_info << "Creating new SYmbolic LinK (SYLK) trace file " << p_name_pt
            << ".slk..." << pw_endl;


    // Creation of the trace file
    if(not a_file)
    {
        std::ostringstream l_msg;
        l_msg << PW_SLK_TRACE_FILE_NAME_MSG_
              << " (" << p_name_pt << ") : "
              << std::strerror(errno) << std::endl;
        SC_REPORT_ERROR(PW_SLK_TRACE_FILE_NAME_TYPE_,
                        l_msg.str().c_str());
        return;
    }

    //! Initialization of the column legends.
    COL_LEGENDS[0] = "MODULE NAME";
    for (int l_acct=0; l_acct<PW_ACCT_MAX_NO_ACCOUNTS; l_acct++) if (pw_accounting_base::global_inuse(l_acct))
      {
	int base_col = ACCT_COL_BASE+l_acct*COLS_PER_ACCT;
	COL_LEGENDS[base_col + POWER_COL] = pw_accounting_base::get_global_name(l_acct) + "POWER";
	COL_LEGENDS[base_col + ENERGY_COL] = pw_accounting_base::get_global_name(l_acct) + "ENERGY";
	COL_LEGENDS[base_col + POWER_RATIO_COL] = pw_accounting_base::get_global_name(l_acct) + "POWER RATIO";
	COL_LEGENDS[base_col + ENERGY_RATIO_COL] = pw_accounting_base::get_global_name(l_acct) + "ENERGY RATIO";

      }

    //! Initialization of the column size

    for(unsigned l_cpt = 0;
        l_cpt < sizeof(COL_LEGENDS)/sizeof(char*);
        ++l_cpt)
              update_column_size(l_cpt+1,
                           std::string(COL_LEGENDS[l_cpt]).size());


    EXT_COL_LEGENDS[0] = "MODULE NAME";
    EXT_COL_LEGENDS[1] = "MODE NAME";
    EXT_COL_LEGENDS[2] = "PHASE NAME";

    for (int l_acct=0; l_acct<PW_ACCT_MAX_NO_ACCOUNTS; l_acct++) if (pw_accounting_base::global_inuse(l_acct))
      {
	int base_col = EXT_ACCT_COL_BASE+l_acct*COLS_PER_ACCT;
	EXT_COL_LEGENDS[base_col + POWER_COL] = pw_accounting_base::get_global_name(l_acct) + "POWER";
	EXT_COL_LEGENDS[base_col + ENERGY_COL] = pw_accounting_base::get_global_name(l_acct) + "ENERGY";
	EXT_COL_LEGENDS[base_col + POWER_RATIO_COL] = pw_accounting_base::get_global_name(l_acct) + "POWER RATIO";
	EXT_COL_LEGENDS[base_col + ENERGY_RATIO_COL] = pw_accounting_base::get_global_name(l_acct) + "ENERGY RATIO";

      }



    for(unsigned l_cpt = 0;
        l_cpt < sizeof(EXT_COL_LEGENDS)/sizeof(char*);
        ++l_cpt)
        update_column_size(l_cpt+1,
                           std::string(EXT_COL_LEGENDS[l_cpt]).size());

    dump_header();
}


  // ---------------------------------------------------------------------------
  pw_slk_trace_file::~pw_slk_trace_file(void)
  {
    pw_debug << __FILE__ " Destructor invoked\n";
    pw_info << "Close SYmbolic LinK (SYLK) trace file " << a_file_name
            << ".slk" << pw_endl;

    pw_assert(a_observers_pt.size() == a_observers_name.size()
              and "Oups, incoherent status of TXT trace file.");

    for(unsigned l_cpt = 0;
        l_cpt < a_observers_pt.size();
        ++l_cpt)
    {
        pw_stat_observer_base* l_observer_pt =
            a_observers_pt.at(l_cpt);
        // XXX Observer already dumped are marked with a NULL pointer
        if(l_observer_pt)
        {
            const std::string& l_name = a_observers_name.at(l_cpt);
            dump_observer(*l_observer_pt, l_name, l_cpt+2);
        }
    }

    dump_total();

    dump_tailer();

    // Dump End Of File
    a_file << "E";

    pw_debug  << __FILE__  " SLK file written. Destructor finished.\n";
}


// ---------------------------------------------------------------------------
  void pw_slk_trace_file::update(pw_subject& p_subject)
  {
    // The slk trace requires that a stat_observer_base is present for a component. Does it
    // share with txt trace or have its own? TODO explain.
    
    // XXX Store information before destruction of the parameter subject.
    try
    {
        pw_stat_observer_base& l_observer =
            dynamic_cast< pw_stat_observer_base& >(p_subject);

        std::vector< pw_stat_observer_base* >::iterator l_it =
            std::find(a_observers_pt.begin(),
                      a_observers_pt.end(),
                      &l_observer);

        pw_assert(l_it != a_observers_pt.end()
                  and "Oups, observers not recorded in this trace file.");

        const size_t l_pos = std::distance(a_observers_pt.begin(),
                                           l_it);
        dump_observer(l_observer,
                      a_observers_name.at(l_pos),
                      l_pos+2);

        // XXX Remove the l_observer from the list of the observer to dump.
        *l_it = NULL;

        // XXX It is not possible to trace a subject after the deletion of a 
        // statistic observer managed by this trace file.
        a_update_called = true;

    }
    catch(std::bad_cast)
    {
        pw_assert(false
                  and "Casting problem with the \"Observer\" design pattern");
    }
}


  // ---------------------------------------------------------------------------
  void pw_slk_trace_file::trace(sc_core::sc_object& p_obj,
				const std::string& p_str,
				trace_t p_do_children,
				plot_control_t p_plot_control
				)
  {
    // XXX It is not possible to trace a subject after the deletion of a 
    // statitisc observer managed by this trace file.
    if(a_update_called)
    {
        std::ostringstream l_msg;
        l_msg << PW_SLK_TRACE_LOCKED_MSG_
              << p_obj.name() << " (" << p_str << ")" << std::endl;
        SC_REPORT_WARNING(PW_SLK_TRACE_LOCKED_TYPE_,
                          l_msg.str().c_str());
        return;
    }

    pw_info << "Trace " << p_obj.name() << " (" << p_str
            << ") in " << a_file_name << ".slk..." << pw_endl;

    //sc_module *l_mod = dynamic_cast<sc_module *>(&p_obj);
    //pw_assert(l_mod and "trace should be applied to an sc_module");
    pw_stat_observer_base *l_observer_pt = pw_get_observer(&p_obj, p_do_children);

    a_observers_pt.push_back(l_observer_pt);
    a_observers_name.push_back(p_str);
    
    
    // Increment the line index for extra information
    ++a_extra_line;
    
    
    if (p_do_children == also_trace_children) this->trace_all_below(&p_obj, p_do_children);
}


// ---------------------------------------------------------------------------
void pw_slk_trace_file::set_time_unit(double p_value,
                                      sc_core::sc_time_unit p_unit)
{
    // Nothing todo
    return;
}


// ---------------------------------------------------------------------------
void pw_slk_trace_file::write_comment(const std::string& p_str)
{
    SC_REPORT_WARNING(PW_SLK_UNDEFINED_FUNCTION_TYPE_,
                      PW_SLK_UNDEFINED_FUNCTION_MSG_);
}


// ---------------------------------------------------------------------------
void pw_slk_trace_file::dump_string(unsigned p_pos_x,
                                    unsigned p_pos_y,
                                    const std::string& p_name,
                                    char p_alignment)
{
    update_column_size(p_pos_x, p_name.size());

    // Formating directives
    // p_alignment:
    //  - 'C' = Center
    //  - 'R' = Right
    //  - 'L' = Left
    a_file << STRING_FORMAT_HEADER << p_alignment << STRING_FORMAT_TAILER
           << "Y" << p_pos_y << ";"
           << "X" << p_pos_x << std::endl;

    // Module Name DATA
    a_file << "C;"
           << "Y" << p_pos_y << ";"
           << "X" << p_pos_x << ";"
           << "N;K\"" << string_normalize(p_name) << "\"" << std::endl;
}


// ---------------------------------------------------------------------------
void pw_slk_trace_file::dump_bold_string(unsigned p_pos_x,
                                         unsigned p_pos_y,
                                         const std::string& p_name,
                                         char p_alignment)
{
    update_column_size(p_pos_x, p_name.size());

    // Formating directives
    // p_alignment:
    //  - 'C' = Center
    //  - 'R' = Right
    //  - 'L' = Left
    a_file << BOLD_STRING_FORMAT_HEADER << p_alignment
           << BOLD_STRING_FORMAT_TAILER
           << "Y" << p_pos_y << ";"
           << "X" << p_pos_x << std::endl;

    // Module Name DATA
    a_file << "C;"
           << "Y" << p_pos_y << ";"
           << "X" << p_pos_x << ";"
           << "N;K\"" << string_normalize(p_name) << "\"" << std::endl;
}


// ---------------------------------------------------------------------------
void pw_slk_trace_file::dump_energy(unsigned p_pos_x,
                                    unsigned p_pos_y,
                                    const pw_energy& p_energy)
{
    update_column_size(p_pos_x, sizeof(_ENERGY_FORMAT_)-1);

    // Formating directives
    a_file << ENERGY_FORMAT
           << "Y" << p_pos_y << ";"
           << "X" << p_pos_x << std::endl;

    // Energy value
    a_file << "C;"
           << "Y" << p_pos_y << ";"
           << "X" << p_pos_x << ";"
           << "N;K" << std::scientific
           << p_energy.to_joules() << std::endl;
}


// ---------------------------------------------------------------------------
void pw_slk_trace_file::dump_power(unsigned p_pos_x,
                                   unsigned p_pos_y,
                                   const pw_power& p_power)
{
    update_column_size(p_pos_x, sizeof(_POWER_FORMAT_)-1);

    // Formating directives
    a_file << POWER_FORMAT
           << "Y" << p_pos_y << ";"
           << "X" << p_pos_x << std::endl;

    // Power value
    a_file << "C;"
           << "Y" << p_pos_y << ";"
           << "X" << p_pos_x << ";"
           << "N;K" << std::scientific
           << p_power.to_watts() << std::endl;
}


// ---------------------------------------------------------------------------
void pw_slk_trace_file::dump_ratio(unsigned p_pos_x,
                                   unsigned p_pos_y,
                                   double p_ratio)
{
    update_column_size(p_pos_x, sizeof(_RATIO_FORMAT_)-1);

    // Formating directives
    a_file << RATIO_FORMAT
           << "Y" << p_pos_y << ";"
           << "X" << p_pos_x << std::endl;

    // Ratio value
    a_file << "C;"
           << "Y" << p_pos_y << ";"
           << "X" << p_pos_x << ";"
           << "N;K" << std::scientific
           << p_ratio << std::endl;
}


// ---------------------------------------------------------------------------
void pw_slk_trace_file::dump_observer(pw_stat_observer_base& p_observer,
                                      const std::string& p_name,
                                      unsigned p_pos_y)
{
    // XXX We have to print information of unpowered module for preserving 
    // array size coherency
    /**
    if(p_observer.get_energy() == PW_ZERO_ENERGY)
        return;
        **/


    dump_string(EXT_MODULE_NAME_COL, p_pos_y, p_name);


    const sc_time l_global_last_update = p_observer.get_global_last_update();

    for (int acct=0; acct<PW_ACCT_MAX_NO_ACCOUNTS; acct++)
      {
	int base_col = EXT_ACCT_COL_BASE+acct*COLS_PER_ACCT;


	// Account Dynamic Energy Cell
	dump_energy(base_col+ENERGY_COL, p_pos_y,  p_observer.get_energy(acct));

	// Account Dynamic Power Cell
	const pw_power l_dynamic_power =
	  (p_observer.get_global_last_update() == sc_core::SC_ZERO_TIME)? \
	  (sc_pwr::PW_ZERO_POWER):					\
	  (p_observer.get_energy(acct) /
	   l_global_last_update);
	dump_power(base_col+POWER_COL, p_pos_y, l_dynamic_power);


	// Account Dynamic Energy Ratio Cell
	const double l_dynamic_ratio =
	  (pw_stat_observer_base::get_global_energy(acct)
	   == PW_ZERO_ENERGY)?				\
	  (0.):							\
	  (p_observer.get_energy(acct).to_double()
	   / pw_stat_observer_base::get_global_energy(acct).to_double());

	dump_ratio(base_col+ENERGY_RATIO_COL, p_pos_y, l_dynamic_ratio);

	// Account Power Ratio Cell
	const double l_static_ratio =
	  (pw_stat_observer_base::get_global_energy(acct)
	   == PW_ZERO_ENERGY)?			\
	  (0.):						\
	  (p_observer.get_energy(acct).to_double()
	   / pw_stat_observer_base::get_global_energy(acct).to_double());
	dump_ratio(base_col+POWER_RATIO_COL, p_pos_y,  l_static_ratio);
      }

    // Dump Extra information (mode and phase)
    #if 0
    try
    {
        pw_stat_observer& l_observer = dynamic_cast< pw_stat_observer& >(p_observer);
        dump_extra_infos(l_observer, p_name);
    }
    catch(std::bad_cast)
    {
        // Nothing todo
    }
    #endif
}



/*
// ---------------------------------------------------------------------------
void pw_slk_trace_file::dump_mode_infos(pw_stat_observer& p_observer,
                                        const std::string& p_mode,
                                        unsigned p_pos_y)
{
    dump_string(EXT_MODE_NAME_COL, p_pos_y, p_mode);

    dump_energy(EXT_DYNAMIC_ENERGY_COL, p_pos_y,
                p_observer.get_energy(p_mode));
    dump_energy(EXT_STATIC_ENERGY_COL, p_pos_y,
                p_observer.get_static_energy(p_mode));

    // Dynamic Power Cell
    const pw_power l_dynamic_power =
        (p_observer.get_elapsed_time(p_mode) == sc_core::SC_ZERO_TIME)?\
        (sc_pwr::PW_ZERO_POWER):\
        (p_observer.get_energy(p_mode) /
         p_observer.get_elapsed_time(p_mode));
    dump_power(EXT_DYNAMIC_POWER_COL, p_pos_y,
               l_dynamic_power);


    // Static Power Cell
    const pw_power l_static_power =
        (p_observer.get_elapsed_time(p_mode) == sc_core::SC_ZERO_TIME)?\
        (sc_pwr::PW_ZERO_POWER):\
        (p_observer.get_static_energy(p_mode) /
         p_observer.get_elapsed_time(p_mode));
    dump_power(EXT_STATIC_POWER_COL, p_pos_y,
               l_static_power);

    // Dynamic Ratio Cell
    const double l_dynamic_ratio =
        (p_observer.get_energy() == PW_ZERO_ENERGY)?\
        (0.):\
        (p_observer.get_energy(p_mode).to_double()
         / p_observer.get_energy().to_double());
    dump_ratio(EXT_DYNAMIC_RATIO_COL, p_pos_y,
               l_dynamic_ratio);

    // Static Ratio Cell
    const double l_static_ratio =
        (p_observer.get_static_energy() == PW_ZERO_ENERGY)?\
        (0.):\
        (p_observer.get_static_energy(p_mode).to_double()
         / p_observer.get_static_energy().to_double());
    dump_ratio(EXT_STATIC_RATIO_COL, p_pos_y,
               l_static_ratio);
}


// ---------------------------------------------------------------------------
void pw_slk_trace_file::dump_phase_infos(pw_stat_observer& p_observer,
                                         const std::string& p_mode,
                                         const std::string& p_phase,
                                         unsigned p_pos_y)
{
    dump_string(EXT_PHASE_NAME_COL, p_pos_y, p_phase);

    dump_energy(EXT_DYNAMIC_ENERGY_COL, p_pos_y,
                p_observer.get_energy(p_mode, p_phase));
    dump_energy(EXT_STATIC_ENERGY_COL, p_pos_y,
                p_observer.get_static_energy(p_mode, p_phase));

    // Dynamic Power Cell
    const pw_power l_dynamic_power =
        (p_observer.get_elapsed_time(p_mode, p_phase)
         == sc_core::SC_ZERO_TIME)?\
        (sc_pwr::PW_ZERO_POWER):\
        (p_observer.get_energy(p_mode, p_phase) /
         p_observer.get_elapsed_time(p_mode, p_phase));
    dump_power(EXT_DYNAMIC_POWER_COL, p_pos_y,
               l_dynamic_power);


    // Static Power Cell
    const pw_power l_static_power =
        (p_observer.get_elapsed_time(p_mode, p_phase)
         == sc_core::SC_ZERO_TIME)?\
        (sc_pwr::PW_ZERO_POWER):\
        (p_observer.get_static_energy(p_mode, p_phase) /
         p_observer.get_elapsed_time(p_mode, p_phase));
    dump_power(EXT_STATIC_POWER_COL, p_pos_y,
               l_static_power);

    // Dynamic Ratio Cell
    const double l_dynamic_ratio =
        (p_observer.get_energy(p_mode) == PW_ZERO_ENERGY)?\
        (0.):\
        (p_observer.get_energy(p_mode, p_phase).to_double()
         / p_observer.get_energy(p_mode).to_double());
    dump_ratio(EXT_DYNAMIC_RATIO_COL, p_pos_y,
               l_dynamic_ratio);

    // Static Ratio Cell
    const double l_static_ratio =
        (p_observer.get_static_energy(p_mode) == PW_ZERO_ENERGY)?\
        (0.):\
        (p_observer.get_static_energy(p_mode, p_phase).to_double()
         / p_observer.get_static_energy(p_mode).to_double());
    dump_ratio(EXT_STATIC_RATIO_COL, p_pos_y,
               l_static_ratio);
}
*/

// ---------------------------------------------------------------------------
void pw_slk_trace_file::dump_total(void)
{
    const unsigned l_pos_y = a_observers_pt.size() + 2;

    dump_bold_string(TOT_MODULE_NAME_COL, l_pos_y, "TOTAL");

    dump_energy(TOT_ENERGY_COL, l_pos_y, pw_stat_observer_base::get_global_energy());

    const sc_time l_global_last_update = pw_stat_observer_base::get_global_last_update();

    for (int acct=0;acct<PW_ACCT_MAX_NO_ACCOUNTS;acct++)
      {
	int base_col = TOT_ACCT_COL+acct*TOT_COLS_PER_ACCT;

	// Energy Cell
	dump_energy(base_col+TOT_ACCT_ENERGY_COL, l_pos_y, pw_stat_observer_base::get_global_energy(acct));
	
	// Power Cell
	const pw_power l_power =
	  (l_global_last_update  ==  sc_core::SC_ZERO_TIME)?		\
	  (sc_pwr::PW_ZERO_POWER):				\
	  (pw_stat_observer_base::get_global_energy(acct) / l_global_last_update);
	dump_power(base_col+TOT_ACCT_POWER_COL, l_pos_y, l_power);

	//dump_ratio(TOT_DYNAMIC_RATIO_COL, l_pos_y, 1.);//FUDGE  - should add up!
	//dump_ratio(TOT_STATIC_RATIO_COL, l_pos_y, 1.);
      }
}

// ---------------------------------------------------------------------------
void pw_slk_trace_file::dump_header(void)
{
    a_file << FILE_HEADER;
    
    // Dump the Legends of the Array
    for(unsigned l_cpt = 0;
        l_cpt < sizeof(COL_LEGENDS)/sizeof(char*);
        ++l_cpt)
        dump_bold_string(l_cpt+1, 1, COL_LEGENDS[l_cpt], 'C');
}


// ---------------------------------------------------------------------------
void pw_slk_trace_file::dump_tailer(void)
{
    pw_assert(a_columns_size.size() == COLUMN_NB
              and "Incoherent size of array");

    // Dump column length
    for(unsigned l_cpt = 0;
        l_cpt < COLUMN_NB;
        ++l_cpt)
        a_file << "F;W" << l_cpt+1 << " " << l_cpt+1 << " "
               << std::min(( a_columns_size.at(l_cpt) * 3 ) / 2,
                           COLUMN_MAX_SIZE)
               << std::endl;

    // Dump Size of the spreasheet
    const unsigned l_array_height =
        (a_is_ext_legend_dumped)?(a_extra_line-1):(a_observers_pt.size()+2);
    a_file << "B;Y" << l_array_height
           << ";X" << COLUMN_NB << std::endl;
}


// ---------------------------------------------------------------------------
void pw_slk_trace_file::update_column_size(unsigned p_pos_x,
                                           unsigned p_size)
{
  pw_assert(1 <= p_pos_x and "Incoherent position in array");
  
  
  a_columns_size.at(p_pos_x-1) =
    (p_pos_x >= a_columns_size.size()) ? p_size:
        std::max(a_columns_size.at(p_pos_x-1), p_size);
}


// ---------------------------------------------------------------------------
pw_trace_file* pw_create_slk_trace_file(const char* p_name_pt)
{
  pw_trace_file *r = new pw_slk_trace_file(p_name_pt);
  // TODO: is this missing?
  //if (p_do_children != pw_trace_file::no_children) r->trace_all_below(0, p_do_children);
  return r;
}


// ---------------------------------------------------------------------------
void pw_close_slk_trace_file(pw_trace_file* p_file_pt)
{
    delete p_file_pt;
}


} // namespace sc_pwr
