// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************/


#include <boost/property_tree/ptree.hpp>
#include <boost/property_tree/xml_parser.hpp>
#include <boost/foreach.hpp>
#include <string>
#include <set>
#include <exception>
#include <iostream>

#include <sys/stat.h>

struct debug_settings
{
  std::string m_file;               // log filename
  int m_level;                      // debug level
  std::set<std::string> m_modules;  // modules where logging is enabled
  void load(const std::string &filename);
  void save(const std::string &filename);
};

void debug_settings::load(const std::string &filename)
{

  // Create empty property tree object
  using boost::property_tree::ptree;
  ptree pt;

  // Load XML file and put its contents in property tree. 
  // No namespace qualification is needed, because of Koenig 
  // lookup on the second argument. If reading fails, exception
  // is thrown.
  read_xml(filename, pt);

  // Get filename and store it in m_file variable. Note that 
  // we specify a path to the value using notation where keys 
  // are separated with dots (different separator may be used 
  // if keys themselves contain dots). If debug.filename key is 
  // not found, exception is thrown.
  m_file = pt.get<std::string>("debug.filename");
    
  // Get debug level and store it in m_level variable. This is 
  // another version of get method: if debug.level key is not 
  // found, it will return default value (specified by second 
  // parameter) instead of throwing. Type of the value extracted 
  // is determined by type of second parameter, so we can simply 
  // write get(...) instead of get<int>(...).
  m_level = pt.get("debug.level", 0);

  // Iterate over debug.modules section and store all found 
  // modules in m_modules set. get_child() function returns a 
  // reference to child at specified path; if there is no such 
  // child, it throws. Property tree iterator can be used in 
  // the same way as standard container iterator. Category 
  // is bidirectional_iterator.
  BOOST_FOREACH(ptree::value_type &v, pt.get_child("debug.modules"))
    m_modules.insert(v.second.data());

}

void debug_settings::save(const std::string &filename)
{

  // Create empty property tree object
  using boost::property_tree::ptree;
  ptree pt;

  // Put log filename in property tree
  pt.put("debug.filename", m_file);

  // Put debug level in property tree
  pt.put("debug.level", m_level);

  // Iterate over modules in set and put them in property
  // tree. Note that the add function places new key at the
  // end of list of keys. This is fine in most of the
  // situations. If you want to place item at some other
  // place (i.e. at front or somewhere in the middle),
  // this can be achieved using a combination of the insert
  // and put_value functions
  BOOST_FOREACH(const std::string &name, m_modules)
    pt.add("debug.modules.module", name);
    
  // Write property tree to XML file
  write_xml(filename, pt);

}

int settings_test()
{
    try
      {
        debug_settings ds;
	// load it only if file exists
	struct stat f;
	if(stat("debug.xml", &f) == 0) {
	  ds.load("debug.xml");
	  ds.save("debug_settings_out.xml");
	  std::cout << "Success\n";
	}
      }
    catch (std::exception &e)
      {
	std::cout << "Error: " << e.what() << "\n";
      }
    return 0;
}


/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_power_map.cpp,v 1.2 2011/07/25 15:32:45 my294 Exp $

/** @file pw_power_map.cpp
 * @brief Associate power value to mode and phase.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 */

#include <iostream>
#include <cerrno>
#include <fstream>
#include <sstream>
#include <algorithm>

#include "pw_power_map.h"
#include "pw_utils_ids.h"
#include "pw_debug.h"
#include "pw_mode.h"

using namespace std;
// ===========================================================================
namespace
{

// ---------------------------------------------------------------------------
// GLOBAL CONSTANT
static const unsigned LINE_SIZE = 5;

} // ANONYMOUS NAMESPACE (replace static declaration)

// ===========================================================================
namespace sc_pwr
{

// ---------------------------------------------------------------------------
// Constructor
pw_power_map::pw_power_map(const char* p_file_name):
    a_file_name(p_file_name),
//    a_size(PW_ZERO_LENGTH, PW_ZERO_LENGTH),
    a_length(PW_ZERO_LENGTH), a_width(PW_ZERO_LENGTH),
//    a_location(PW_ZERO_LENGTH, PW_ZERO_LENGTH),
    a_loc_x(PW_ZERO_LENGTH), a_loc_y(PW_ZERO_LENGTH),
    a_scaled_static_power(PW_ZERO_POWER)
{
  //  settings_test();

  a_nominal_vcc = pw_voltage(1.0, PW_VOLT);

    // Parse configuration file
    std::ifstream l_file_is(p_file_name,
                            std::ios_base::in);
    if(not l_file_is)
    {
        std::ostringstream l_msg;
        l_msg << PW_MAP_CFG_FILE_NAME_MSG_
              << a_file_name << " ("
              << std::strerror(errno) << ")." << std::endl;
	//        SC_REPORT_WARNING(PW_MAP_CFG_FILE_NAME_TYPE_,   l_msg.str().c_str()); // TODO for now
        return;
    }
   
    pw_info << "Parsing Power Configuration file "
            << p_file_name << "..." << pw_endl;

    std::string l_line;
    unsigned l_cpt = 1;
    while(std::getline(l_file_is,
                       l_line))
    {
        parse_line(l_line,
                   l_cpt);
        ++l_cpt;
    }

   return;

   pw_info << "{";
   pw_info << p_file_name << " "; 

   // TODO - only print layout if fully set up!
   pw_info << " ( length = " << a_length
	   << ", width = " << a_width
	   << ", location x = " << a_loc_x
	   << ", location y = " << a_loc_y
	   << ", Nominal Vcc = " << a_nominal_vcc 
	   << ", Scaled Static Power = " << a_scaled_static_power  << ") }" << pw_endl;
}

  // ---------------------------------------------------------------------------
  bool pw_power_map::is_defined(const std::string& p_mode,
				const std::string& p_phase) const
  {
    std::string l_mode = (p_mode == "")?	\
      (PW_MODE_UNDEF):				\
      (p_mode);
    std::string l_phase = (p_phase == "")?	\
      (PW_PHASE_UNDEF):				\
      (p_phase);
    
    return a_power_map.count(map_key(l_mode,
                                     l_phase)) > 0;
  }




// ---------------------------------------------------------------------------
pw_power_map::~pw_power_map(void)
{
    // That's all folks!!!
}
// ---------------------------------------------------------------------------
#include <sys/stat.h>
#define MAXFLENGTH 1024
#define DIRECTORY_SEPARATOR_CHAR '/' // Un*x form

int fexist( char *filename ) 
{
  struct stat buffer ;
  if ( stat( filename, &buffer ) ) return 1 ;
  return 0 ;
}

class pathopen// Searh a path for a file.
{
  char pathseq[MAXFLENGTH];
public:
  FILE *open_r(const char *name, const char *path);
};


FILE *pathopen::open_r(const char *name, const char *path)
{
  FILE *r;
  const char *mode = "r";
  if (path == NULL || *name == DIRECTORY_SEPARATOR_CHAR || *name == '.')
    {
      //      strncpy(pathseq, name);                                                                                                           
      return fopen(name, mode);
    }
  
  while (path && *path)
  {
    int c = 0;
    char * d = pathseq;
    const char * n = name;
    while (c<MAXFLENGTH && *path && *path != ':')
    {
      *(d++) = *(path++);
      c++;
    }
    if (*path) path++; /* skip colon */
    if (c) *(d++) = DIRECTORY_SEPARATOR_CHAR;
    while (c<MAXFLENGTH && *n)
    {
      *(d++) = *(n++);
      c++;
    }
    *d = (char) 0;
    //    printf("Try %s: %p\n", pathseq, r);
    r = fopen(pathseq,  mode);
    if (r)     return r;
  }
  return 0;
}


// ---------------------------------------------------------------------------
const pw_length& pw_power_map::get_length(void) const
{
    return a_length; //a_size->first;
}

// ---------------------------------------------------------------------------
const pw_length& pw_power_map::get_width(void) const
{
    return a_width; //a_size->second;
}

// ---------------------------------------------------------------------------
const pw_length& pw_power_map::get_loc_x(void) const
{
    return a_loc_x; //a_location->first;
}

// ---------------------------------------------------------------------------
const pw_length& pw_power_map::get_loc_y(void) const
{
    return a_loc_y; //a_location->second;
}

// ---------------------------------------------------------------------------
const pw_voltage& pw_power_map::get_nominal_vcc(void) const
{
    return a_nominal_vcc;
}

// ---------------------------------------------------------------------------
const pw_power& pw_power_map::get_scaled_static_power(void) const
{
    return a_scaled_static_power;
}

// ---------------------------------------------------------------------------
std::string pw_power_map::to_string(void) const
{
    std::ostringstream l_os;

    l_os << "{";
    l_os << " ( length = " << a_length
         << ", width = " << a_width
         << ", location x = " << a_loc_x
         << ", location y = " << a_loc_y
         << ", Nominal Vcc = " << a_nominal_vcc
         << ", Scaled Static Power = " << a_scaled_static_power << ") ";
    l_os << "}";

    return l_os.str();
}


// ---------------------------------------------------------------------------
void pw_power_map::parse_line(const std::string& p_str,
                              unsigned p_line_nb)
{
    std::istringstream l_is(p_str);
    std::list< std::string > l_tokens;
    char token_type = 0;

    while(l_tokens.size() < LINE_SIZE
          and l_is)
    {
        char l_char = 0;
        l_is >> l_char;
        if(not l_is) break;

        // Skip comment.
        if(l_char == '#') break;
        if(l_char == '%') l_is >> token_type;
        else l_is.putback(l_char);

        std::string l_str;
        l_is >> l_str;
//       pw_assert(l_is
//                 and "Unexpected malformated input stream.");
        l_tokens.push_back(l_str);
    }
//        pw_assert(l_tokens.size() >= 4 and "Unexpected malformated input stream.");
    parse_tokens(l_tokens,
                 p_line_nb, token_type);
}


// ---------------------------------------------------------------------------
void pw_power_map::parse_tokens(std::list< std::string >& p_tokens,
                                unsigned p_line_nb, char token_type)
{
    pw_assert(p_tokens.size() <= LINE_SIZE
              and "Incoherent size of tokens vectors.");
    if(p_tokens.size() == 0) return;
    switch(token_type)
    {
      case 's': case 'S':
 /*     std::pair< pw_length, pw_length > l_size;
        l_size.second = extract_length(p_tokens.back(),
                                     p_line_nb);
        l_size.first = extract_length(p_tokens.back(),
                                     p_line_nb);
        a_size = l_size;
*/
        a_width = extract_length(p_tokens.back(),
                                     p_line_nb);
        p_tokens.pop_back();
        a_length = extract_length(p_tokens.back(),
                                     p_line_nb);    
        p_tokens.pop_back();
        break;

      case 'l': case 'L':
/*      std::pair< pw_length, pw_length > l_loc;
        l_size.second = extract_length(p_tokens.back(),
                                     p_line_nb);
        l_size.first = extract_length(p_tokens.back(),
                                     p_line_nb);
        a_location = l_loc;
*/
        a_loc_y = extract_length(p_tokens.back(),
                                     p_line_nb);
        p_tokens.pop_back();
        a_loc_x = extract_length(p_tokens.back(),
                                     p_line_nb);
        p_tokens.pop_back();
        break;

      case 'v': case 'V':
        {
	  pw_voltage l_vcc = extract_voltage(p_tokens.back(),
                              p_line_nb);
	  a_nominal_vcc = l_vcc;
	  p_tokens.pop_back();
	  break;
	}

      case 'p': case 'P':
	{
	  pw_power l_power = extract_power(p_tokens.back(),
                                         p_line_nb);
	  a_scaled_static_power = l_power;
	  p_tokens.pop_back();
	  break; 
	}

      default:;
  }
}

// ---------------------------------------------------------------------------
pw_power pw_power_map::extract_power(const std::string& p_str,
                                     unsigned p_line_nb) const
{
    std::istringstream l_is(p_str);
    pw_power l_power(PW_ZERO_POWER);
    l_is >> l_power;
    if(not l_is)
    {
        std::ostringstream l_msg;
        l_msg << PW_MAP_MALFORMED_POWER_VALUE_MSG_
              << "in file " << a_file_name
              << " at line " << p_line_nb << "." << std::endl;
        SC_REPORT_ERROR(PW_MAP_MALFORMED_POWER_VALUE_TYPE_,
                        l_msg.str().c_str());
    }

    return l_power;
}

// ---------------------------------------------------------------------------
pw_voltage pw_power_map::extract_voltage(const std::string& p_str,
                                     unsigned p_line_nb) const
{
    std::istringstream l_is(p_str);
    pw_voltage l_voltage(PW_ZERO_VOLT);
    l_is >> l_voltage;
    if(not l_is)
    {
        std::ostringstream l_msg;
        l_msg << PW_MAP_MALFORMED_VOLTAGE_VALUE_MSG_
              << "in file " << a_file_name
              << " at line " << p_line_nb << "." << std::endl;
        SC_REPORT_ERROR(PW_MAP_MALFORMED_VOLTAGE_VALUE_TYPE_,
                        l_msg.str().c_str());
    }

    return l_voltage;
}

// ---------------------------------------------------------------------------
pw_length pw_power_map::extract_length(const std::string& p_str,
                                     unsigned p_line_nb) const
{
    std::istringstream l_is(p_str);
    pw_length l_length(PW_ZERO_LENGTH);
    l_is >> l_length;
    if(not l_is)
    {
        std::ostringstream l_msg;
        l_msg << PW_MAP_MALFORMED_LENGTH_VALUE_MSG_
              << "in file " << a_file_name
              << " at line " << p_line_nb << "." << std::endl;
        SC_REPORT_ERROR(PW_MAP_MALFORMED_LENGTH_VALUE_TYPE_,
                        l_msg.str().c_str());
    }

    return l_length;
}



// ---------------------------------------------------------------------------
pw_energy pw_power_map::extract_energy(const std::string& p_str,
                                     unsigned p_line_nb) const
{
    std::istringstream l_is(p_str);
    pw_energy l_energy(PW_ZERO_ENERGY);
    l_is >> l_energy;
    if(not l_is)
    {
        std::ostringstream l_msg;
        l_msg << PW_MAP_MALFORMED_ENERGY_VALUE_MSG_
              << "in file " << a_file_name
              << " at line " << p_line_nb << "." << std::endl;
        SC_REPORT_ERROR(PW_MAP_MALFORMED_ENERGY_VALUE_TYPE_,
                        l_msg.str().c_str());
    }

    return l_energy;
}

// ---------------------------------------------------------------------------
// Output stream operator of a power map record.
std::ostream& operator<<(std::ostream& p_os,
                         const pw_power_map& p_map)
{
    std::ostream::sentry l_init(p_os);
    if(!l_init)
    {
        SC_REPORT_ERROR(PW_MAP_OSTREAM_ERROR_TYPE_,
                        PW_MAP_OSTREAM_ERROR_MSG_);
        return p_os;
    }

    return (p_os << p_map.to_string());

}

  // ---------------------------------------------------------------------------
  pw_power_map::map_key::map_key(const std::string& p_mode,
				 const std::string& p_phase):
    a_mode(p_mode),
    a_phase(p_phase)
  {
    // That's all folks!!!
  }


  // ---------------------------------------------------------------------------
  bool operator<(const pw_power_map::map_key& p_key1,
		 const pw_power_map::map_key& p_key2)
  {
    return (p_key1.a_mode < p_key2.a_mode
            or (p_key1.a_mode == p_key2.a_mode
                and p_key1.a_phase < p_key2.a_phase));
  }


  // ---------------------------------------------------------------------------
  bool operator==(const pw_power_map::map_key& p_key1,
		  const pw_power_map::map_key& p_key2)
  {
    return (p_key1.a_mode == p_key2.a_mode
            and p_key1.a_phase == p_key2.a_phase);
  }

} // namespace sc_pwr
