// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_physical_operators.cpp,v 1.1 2011/06/30 11:10:38 my294 Exp $

/** @file pw_physical_operators.cpp
 * @brief Define Physical operators between power, energy and time.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 */

#include "pw_physical_operators.h"
#include "pw_kernel_ids.h"
#include "pw_debug.h"
#include "pw_common.h"



// ===========================================================================
namespace sc_pwr
{

// ---------------------------------------------------------------------------
pw_energy operator*(const pw_power& p_power,
                    const sc_core::sc_time& p_time)
{

    const pw_energy l_energy(p_power.to_watts() * p_time.to_seconds(),
                             PW_JOULE);
    if(l_energy == PW_ZERO_ENERGY and p_time != sc_core::SC_ZERO_TIME
       and p_power != PW_ZERO_POWER)
    {
        SC_REPORT_WARNING(PW_ENERGY_UNDERFLOW_TYPE_,
                          PW_ENERGY_UNDERFLOW_MSG_);
    }
    return l_energy;
}

// ---------------------------------------------------------------------------
pw_energy operator*(const sc_core::sc_time& p_time,
                    const pw_power& p_power)
{
    return p_power * p_time;
}

// ---------------------------------------------------------------------------
pw_power operator/(const pw_energy& p_energy,
                   const sc_core::sc_time& p_time)
{
    if(p_time.value() == 0)
    {
      pw_debug << "Divide by zero";
      SC_REPORT_ERROR(PW_PHYS_DIVISION_BY_ZERO_TYPE_,
		      PW_PHYS_DIVISION_BY_ZERO_MSG_);
      return pw_power::max();
    }

    return pw_power(p_energy.to_joules() / p_time.to_seconds(),
                    PW_WATT);
}

// ---------------------------------------------------------------------------
sc_core::sc_time operator/(const pw_energy& p_energy,
                           const pw_power& p_power)
{
    if(p_power.value() == 0)
    {
        SC_REPORT_ERROR(PW_PHYS_DIVISION_BY_ZERO_TYPE_,
                        PW_PHYS_DIVISION_BY_ZERO_MSG_);
        return PW_UINT64_MAX * sc_core::sc_get_time_resolution();
    }

    return sc_core::sc_time(p_energy.to_joules() / p_power.to_watts(),
                            sc_core::SC_SEC);
}


} // namespace sc_pwr
