// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_observer_base.cpp,v 1.2 2011/07/25 15:32:45 my294 Exp $

/** @file pw_observer_base.cpp
 * @brief Makes statistic on power variation.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 *
 * Base class of the tracing observer.
 */

#include <systemc>
#include "pw_observer_base.h"
#include "pw_module_base.h"
#include "pw_kernel_ids.h"
#include "pw_debug.h"
#include "pw_common.h"
#include "pw_tracing_ids.h"
#include "pw_observer_ids.h"

// ===========================================================================
namespace sc_pwr
{

  //---------------------------------------------------------------------------
  pw_observer_base::pw_observer_base(sc_core::sc_object& p_obj,
				     const std::string& p_att_name):
    pw_observer_if(),
    sc_core::sc_attr_base(p_att_name),
    a_obj(p_obj),
    a_att_name(p_att_name)
  { }
  
  // ---------------------------------------------------------------------------
  pw_observer_base::~pw_observer_base(void)
  {
    if(a_obj.remove_attribute(a_att_name) != ((sc_core::sc_attr_base*) this))
      {
        const std::string l_warning_msg =
	  std::string(PW_UNRECORDED_OBSERVER_MSG_)
	  + "\"" + a_att_name + "\".";
        SC_REPORT_ERROR(PW_UNRECORDED_OBSERVER_TYPE_,
                        l_warning_msg.c_str());
      }
  }


  // ---------------------------------------------------------------------------
  std::string pw_observer_base::to_string(void) const
  {
    std::ostringstream l_os;
    
    l_os << "(" << a_obj.name()
         << ", " << a_att_name << ")";
    
    return l_os.str();
  }

  // ---------------------------------------------------------------------------
  sc_core::sc_object& pw_observer_base::get_sc_object(void) const
  {
    return a_obj;
  }


  // ---------------------------------------------------------------------------
  std::string pw_observer_base::get_obj_name(void) const
  {
    return a_obj.name();
  }
  
  // ---------------------------------------------------------------------------
  std::string pw_observer_base::get_att_name(void) const
  {
    return a_att_name;
  }

  // ---------------------------------------------------------------------------
  //! Attach a parameter observer locally and optionally also to all lower monitored modules.
  //! If p_object is null then start at top.
  void attach_observer(pw_observer_base& p_observer,
		       sc_core::sc_object* p_pos,
		       bool p_attach_to_children)
  {
    typedef std::vector< sc_core::sc_object* > vect_const;
    typedef std::vector< sc_core::sc_object* >::const_iterator vect_const_iterator;

    if (!p_pos) // Seed top of tree walk.
      { 
	pw_assert(p_attach_to_children and "nonsence to pass in null and not request recursion");
	const vect_const & l_objs = sc_core::sc_get_top_level_objects();
	for(vect_const_iterator l_it = l_objs.begin();
	    l_it != l_objs.end();
	    ++l_it)
	  attach_observer(p_observer, *l_it, p_attach_to_children);
      }

    else // Attach this one.
      {
        pw_module_base* l_module = dynamic_cast< pw_module_base *>(p_pos);

	// This cross cast seems to fail sometimes: eg the busaccess component always fails.
	//std::cout << "dox: attach_to dynamic_cast problematic " << ((void *)p_pos) << "  " << ((void *)l_module) << "\n";

        if (l_module) 
	  {
	    const char *l_aid = (p_attach_to_children) ? PW_STAT_OBSERVER_ATTRIBUTE_ID_CHILD_SUMMED: PW_STAT_OBSERVER_ATTRIBUTE_ID_SINGLE_MODULE;
	    // [NOTE] Add the created observer to the attribute list of the traced SystemC object.
	    l_module->attach(p_observer, l_aid); // Reverse link
	  }
	if (p_attach_to_children)
	  {
	    // Recursively attach the child of this object to this observer.
	    const std::vector< sc_core::sc_object* >& l_tab_objs = p_pos->get_child_objects();
	    for(std::vector< sc_core::sc_object* >::const_iterator l_it =
		  l_tab_objs.begin();
		l_it != l_tab_objs.end();
		++l_it)
	      attach_observer(p_observer, *l_it, p_attach_to_children);  // Recursive CALL
	  }
      }
  }


  // ---------------------------------------------------------------------------
  // Output stream operator of energy object.
  std::ostream& operator<<(std::ostream& p_os,
			   const pw_observer_base& p_observer)
  {
    std::ostream::sentry l_init(p_os);
    if(!l_init)
      {
        SC_REPORT_ERROR(PW_KERNEL_OSTREAM_ERROR_TYPE_,
                        PW_KERNEL_OSTREAM_ERROR_MSG_);
        return p_os;
      }
    return (p_os << p_observer.to_string());
  }


} // namespace sc_pwr
