// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $


/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: $


/** @file pw_accounting_base.cpp
 * @brief Implements the main account type that holds power and energy.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 *
 * This class is used by the TXT and CSV trace file for generating stastics on 
 * the power variation.
 */


#include <sstream>
#include <systemc>
#include "pw_accounting_base.h"
#include "pw_tracing_ids.h"
#include "pw_physical_operators.h"
#include "pw_debug.h"
#include "pw_common.h"
#include "iostream"
// ===========================================================================
using std::string;
using sc_core::sc_time;
using sc_core::SC_ZERO_TIME;
using sc_core::sc_time_stamp;
using namespace std;

// ===========================================================================
namespace sc_pwr
{


// ---------------------------------------------------------------------------
//! Macros definition of the maximal time value.
#define SC_TIME_MAX sc_time_max()


// ---------------------------------------------------------------------------
  class pw_accounts_directory
  {
  public:
    
    class dirslot
    {
    public:
      const char *m_name;
      bool inuse() { return (m_name != 0); };
      void start(const char *l_name) { m_name = l_name; }
      dirslot() { m_name = 0; }; // Constructor
    };
    int n_accounts_in_use()
    {
      int r = 0;
      for (int l_acct=0;l_acct<PW_ACCT_MAX_NO_ACCOUNTS;l_acct++) if (a_accts[l_acct].inuse()) r+=1;
      return r;
    }
    dirslot a_accts[PW_ACCT_MAX_NO_ACCOUNTS];
    bool inuse(int p_acct) { return a_accts[p_acct].inuse(); }
    int find_account(const char *name);
  };
  

  //! Reference to singleton object which stores the accounts directory.
  static pw_accounts_directory* g_accounts_directory_pt = NULL;

  int pw_accounts_directory::find_account(const char *name)
  {
    int l_acct;
    for (l_acct=0;l_acct<PW_ACCT_MAX_NO_ACCOUNTS;l_acct++) if (a_accts[l_acct].inuse() && !strcmp(a_accts[l_acct].m_name, name)) break;
    if (l_acct==PW_ACCT_MAX_NO_ACCOUNTS)
      {
	// find unused
	for (l_acct=0;l_acct<PW_ACCT_MAX_NO_ACCOUNTS;l_acct++) if (!a_accts[l_acct].inuse()) break;
	if (l_acct==PW_ACCT_MAX_NO_ACCOUNTS)
	  {
	    SC_REPORT_FATAL(PW_STAT_OBSERVER_TOO_MANY_ACCOUNTS_TYPE_, PW_STAT_OBSERVER_TOO_MANY_ACCOUNTS_MSG_);
	    return -1;
	  }
	else
	  {
	    pw_debug << "Start global account " << this 
		     << " " << l_acct << " " << name << pw_endl;
	    a_accts[l_acct].start(name);
	  }
      }
    return l_acct;
  }

  static pw_accounts_directory *pw_accounts_directory_ensure_exists()
  {
    if (!g_accounts_directory_pt) g_accounts_directory_pt = new pw_accounts_directory();
    return g_accounts_directory_pt;
  }

  // ---------------------------------------------------------------------------
  bool pw_accounting_base::inuse(int p_acct) const
  {
    return a_accts[p_acct].inuse();
  }




  //
  // Accounts have a global naming system held in the accounts_directory singleton class.
  //

  int pw_accounting_base::start_acct(const char *name)
  {
    pw_accounts_directory *l_dir = pw_accounts_directory_ensure_exists();
    int acct = l_dir->find_account(name);
    // Find existing of that name or create new
    auto_start_acct(acct);
    return acct;
  }

  bool pw_accounting_base::global_inuse(int p_acct)
  {
    pw_accounts_directory *l_dir = pw_accounts_directory_ensure_exists();
    return l_dir->inuse(p_acct);
    //    pw_assert(g_global_power_pt);
    //return g_global_power_pt->inuse(p_acct);
  }


  int pw_accounting_base::global_n_accounts_in_use()
  {
    pw_accounts_directory *l_dir = pw_accounts_directory_ensure_exists();
    return l_dir->n_accounts_in_use();
  }


  void pw_accounting_base::auto_start_acct(int a)
  {
    pw_accounts_directory *l_dir = pw_accounts_directory_ensure_exists();
    pw_assert(l_dir);
    pw_assert(l_dir->a_accts[a].inuse() and "account not started/named");
    const char *name= l_dir->a_accts[a].m_name;
    if (!a_accts[a].inuse()) // autostart
      {
	pw_debug << "Start account " << get_sc_object().name()
		 << " " << a << " " << name << pw_endl;
	a_accts[a].start(this, name);
      }
  }
  
  void pw_accounting_base::acct::start(pw_accounting_base *parent, const char *name)
  {
    this->m_parent = parent;
    this->m_name = name;
    this->m_energy = PW_ZERO_ENERGY;
    this->m_last_baseline_update = sc_time_stamp();
    this->m_phasemode_power  = PW_ZERO_POWER;
  }
  

  // ---------------------------------------------------------------------------



  std::string pw_accounting_base::get_global_name(int acct)
  {
    pw_accounts_directory *l_dir = pw_accounts_directory_ensure_exists();
    return l_dir->a_accts[acct].m_name;
  }

  const std::string pw_accounting_base::get_name(int acct)
  {
    return a_accts[acct].m_name;
  }


  pw_energy pw_accounting_base::get_energy(int acct)
  {
#if 0
    if(sc_time_stamp() == SC_TIME_MAX)
    {
      check_end_of_simulation();
      pw_accounting_base::end_of_simulation();
      a_is_simulation_terminated = true;
    }
#endif
    return a_accts[acct].get_energy();
  }

  
  pw_energy pw_accounting_base::acct::get_energy() const
  {
    return m_energy;
  }


  pw_energy pw_accounting_base::get_energy() // All accounts
  {
    pw_energy l_sum = PW_ZERO_ENERGY;
    for (int a=0;a<PW_ACCT_MAX_NO_ACCOUNTS;a++)
      if (a_accts[a].inuse()) l_sum += a_accts[a].get_energy();
    return l_sum;
  }

  // ---------------------------------------------------------------------------

  void pw_accounting_base::start_default_accounts(int n)
  {
    if (global_n_accounts_in_use() > 0) return;
    start_acct("STATIC0");
    start_acct("DYNAMIC1");
    if (n > 2) start_acct("WIRING2");
    if (n > 3) start_acct("AUX3");
  }

  const sc_time pw_accounting_base::get_last_updatet(int acct) const
  {
    return a_accts[acct].get_last_update_at();
  }


  //! Find last update time over all accounts and fold time to that.
  const sc_time pw_accounting_base::get_last_updatet()
  {
    sc_time l_last = SC_ZERO_TIME;
    
    for (int a=0;a<PW_ACCT_MAX_NO_ACCOUNTS;a++)
      {
	if (a_accts[a].inuse()) 
	  { 
	    sc_time t = a_accts[a].get_last_update_at();
	    if (t > l_last) l_last = t; // Find max
	  }
      }
    
    pw_assert(l_last != SC_TIME_MAX);
    
  for (int a=0;a<PW_ACCT_MAX_NO_ACCOUNTS;a++) if (a_accts[a].inuse())  a_accts[a].fold_power_to_energy(l_last);
  
  return l_last;
  }




  // The phase/mode power in an account is (in general) the sum of the dissipations of many components. 
  // We need to increase or decrease this according to the deltas made by any component.
void pw_accounting_base::acct::update_power(const sc_time when, const pw_power& inc, const pw_power& dec)
  {
    assert(inuse());
    fold_power_to_energy(when);
    m_phasemode_power += inc;
    m_phasemode_power -= dec;
  }




  // ---------------------------------------------------------------------------
  //! Power - all accounts
  const pw_power pw_accounting_base::get_power() const
  {
    pw_power l_total = PW_ZERO_POWER;
    for (int a=0;a<PW_ACCT_MAX_NO_ACCOUNTS;a++)
      {
	if (a_accts[a].inuse()) l_total += a_accts[a].m_phasemode_power;
      }
    return l_total;
  }


  //! Power - specific account
  const pw_power pw_accounting_base::get_power(int acct) const
  {
    return a_accts[acct].m_phasemode_power;
  }





  // ---------------------------------------------------------------------------

  //! Return true if all accounts have no energy logged.
  bool pw_accounting_base::all_accounts_empty()
  {
    for (int a=0;a<PW_ACCT_MAX_NO_ACCOUNTS;a++)
      if (a_accts[a].inuse() && a_accts[a].m_energy != PW_ZERO_ENERGY) return true;
    return false;
  }

// ---------------------------------------------------------------------------
#if 0
  void pw_accounting_base::update_acct_energy(int acct, const sc_core::sc_time& p_time)
  {
    a_accts[acct].update_acct_energy(p_time);
  }
#endif

  void pw_accounting_base::acct::update_acct_energy(const sc_core::sc_time& p_time)
  {
    //printf("p_time %f  a_last_static_update %f\n", p_time.to_double(), a_last_static_update.to_double());
    fold_power_to_energy(p_time);
  }



  
  void pw_accounting_base::acct::record_energy(pw_energy &new_energy)
  {
    if (!inuse())
      {
	pw_error << " Account not in use " 
		 << /* m_parent-> get_obj_name() << */ pw_endl;
      }
    assert(inuse());
    m_energy += new_energy;
    pw_debug << m_name << " record_energy " << new_energy << ", total=" << m_energy << " (" << m_energy.round3sf() << ")\n";
  }

  // ---------------------------------------------------------------------------
  void pw_accounting_base::acct::fold_power_to_energy(sc_time p_time)
  {
    // The baseline power figure in this account is converted to an energy by multiplying by the
    // interval since the last such event.
    pw_assert(p_time >= m_last_baseline_update and "incoherent updating time.");
    const sc_time l_interval = p_time - m_last_baseline_update;
    const pw_energy l_energy = l_interval * m_phasemode_power;
    m_last_baseline_update = p_time;
    pw_debug << "Account fold " << m_name << " olde=" << m_energy << "\n";
    m_energy += l_energy;
    pw_debug << "Account fold " << m_name << " t=" << l_interval << " p=" << m_phasemode_power.round3sf() << " le=" << l_energy.round3sf() << " e=" << m_energy.round3sf() << " (" << m_energy << ")\n";
  }


  void pw_accounting_base::acct::flush_and_close(sc_time last_update)
  {
    fold_power_to_energy(last_update);
    m_phasemode_power = PW_ZERO_POWER;
  }


  // ---------------------------------------------------------------------------

  // Constructor.
  pw_accounting_base::pw_accounting_base(sc_core::sc_object& p_obj, const std::string& p_att_name):
    pw_observer_base(p_obj, p_att_name)
  {
    
  }
  

  void pw_accounting_base::acct::end_of_simulation(sc_time when)
  {
    // Check if the static power is not equal to zero at the end of an infinite simulation.
    if (when == SC_TIME_MAX and (m_phasemode_power != PW_ZERO_POWER))
      {
	const string l_warning_msg =
	  string(PW_POWERED_OBSERVER_STOPPED_ACTIVE_MSG_)
	  + " (module \"" + m_parent->a_obj.name() + "\").";
	SC_REPORT_WARNING(PW_POWERED_OBSERVER_STOPPED_ACTIVE_TYPE_,
			  l_warning_msg.c_str());
      }
    else 
      {
	fold_power_to_energy(when);
	pw_debug << "End of simulation, account " << m_parent->get_obj_name() << " acct=" << m_name << " at " << when << " E=" << m_energy.round3sf() << "\n";
	m_phasemode_power = PW_ZERO_POWER;
      }
  }



  //
  // End of simulation - tidy accounts - cannot do this if they have an enduring power consumption and an infinite simulation!
  //
  void pw_accounting_base::flush_and_close(sc_time p_when)
  {
    pw_debug <<   "destructor proglog at " << p_when << "\n";
    for (int a = 0; a<PW_ACCT_MAX_NO_ACCOUNTS; a++) if (a_accts[a].inuse()) a_accts[a].flush_and_close(p_when);
  }

  pw_accounting_base::~pw_accounting_base(void)
  {

  }



} // namespace sc_pwr
