// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_slk_trace.h,v 1.1 2011/06/30 08:53:15 my294 Exp $

/** @file pw_slk_trace.h
 * @brief Tracing facilities for generating statistics on power consumption.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 */

#ifndef _PW_SLK_TRACE_
#define _PW_SLK_TRACE_

#include <vector>
#include <string>
#include <utility>
#include <fstream>

#include <systemc>

#include "pw_trace.h"
#include "pw_stat_observer_base.h"
//#include "pw_stat_observer.h"



// ===========================================================================
namespace sc_pwr
{


// ---------------------------------------------------------------------------
/**
 * @brief Trace file compatible with Excel recording power statistics.
 * 
 * Multiple sc_objects can be associated to this trace file for recording 
 * their power statistics.
 */
class pw_slk_trace_file:
    public pw_observer_if,
    public pw_trace_file
{
    public: // CONSTRUCTOR AND DESTRUCTOR
        /**
         * @brief SLK trace file constructor.
         * @param[in] p_name_pt Output trace file name.
         * @note This function should be called indirectly via the function 
         * pw_create_slk_trace_file.
         * @note ".slk" is automatically appended to p_name_pt.
         */
  pw_slk_trace_file(const char* p_name_pt);

        //! Virtual destructor for polymorphism.
        virtual ~pw_slk_trace_file(void);


    public: // SUBJECT MANAGEMENT FUNCTIONS
        /**
         * @brief Method called by an observer used by this trace file.
         * @param[in] p_subject Subject which called this observer.
         *
         * The subject monitored by this observer use this method for 
         * indicating its imminent destruction.
         */
        virtual void update(pw_subject& p_subject);


    public: // TRACING FUNCTIONS
        /**
         * @brief Trace the power consumption of a SystemC object.
         * @param[in,out] p_obj the SystemC object.
         * @param[in] p_str Identification string associated to p_obj.
	 * @param[in] p_children Automatically trace child power modules (included in total or separately).
	 * @param[in] p_plot_control how to include this object in the (gnu-)plot style output file (see VCD forms).
         */
        void trace(sc_core::sc_object& p_obj,
                   const std::string& p_str,
		   trace_t p_children = no_children,
		   plot_control_t p_plot_control = DoNotPlot
		   );


    public: // STANDARD MANAGEMENT FUNCTIONS
        /**
         * @brief Set the time unit use by a trace file for recording events.
         * @param[in] p_value Raw time value.
         * @param[in] p_unit Time unit multiple of the raw value.
         */
        virtual void set_time_unit(double p_value,
                                   sc_core::sc_time_unit p_unit);

        /**
         * @brief Not Yet Implemented.
         * @note This method is not yet implemented and throw a report of 
         * severity sc_warning when used.
         */
        virtual void write_comment(const std::string& p_str);


    private: // PRIVATE ATTRIBUTES
        //! Output tracing file.
        std::ofstream a_file;
        //! Output tracing file name.
        const std::string a_file_name;
        //! List of the observers used by this trace file.
        std::vector< pw_stat_observer_base* > a_observers_pt;
        //! List of the observers identifier string.
        std::vector< std::string > a_observers_name;
        //! True if the update method was called
        bool a_update_called;
        //! Line index for writing extra information
        unsigned a_extra_line;
        //! Size of the column in nb of characters
        std::vector< unsigned > a_columns_size;
        //! Indicate if the extra legend are dumped
        bool a_is_ext_legend_dumped;


    private: // PRIVATE DUMPING FUNCTIONS
        //! Dump a string cell
        void dump_string(unsigned,
                         unsigned,
                         const std::string&,
                         char p_alignment = 'L');
        //! Dump a string cell using a bold font
        void dump_bold_string(unsigned,
                              unsigned,
                              const std::string&,
                              char p_alignment = 'L');
        //! Dump the energy information
        void dump_energy(unsigned,
                         unsigned,
                         const pw_energy&);
        //! Dump the power information
        void dump_power(unsigned,
                        unsigned,
                        const pw_power&);
        //! Dump the informations recorded by a base statistical observer.
        void dump_ratio(unsigned,
                        unsigned,
                        double);
        //! Dump base information recorded by a statistical observer.
        void dump_observer(pw_stat_observer_base&,
                           const std::string&,
                           unsigned);
#if 0
        //! Dump base information in the extra informations array.
        void dump_extra_observer(pw_stat_observer_base&,
                                 const std::string&,
                                 unsigned);
        //! Dump extra informations of a statistical observer.
        void dump_extra_infos(pw_stat_observer&,
                              const std::string&);
        //! Dump mode informations of a statistical observer.
        void dump_mode_infos(pw_stat_observer&,
                             const std::string&,
                             unsigned);
        //! Dump phase informations of a statistical observer.
        void dump_phase_infos(pw_stat_observer&,
                              const std::string&,
                              const std::string&,
                              unsigned);
#endif

        //! Dump the total information
        void dump_total(void);
        //! dump header of a SYLK trace file
        void dump_header(void);
        //! dump formating directives at the end of the file
        void dump_tailer(void);


    private: // MISCELLANOUS FUNCTIONS
        //! Update the size of a column
        void update_column_size(unsigned,
                                 unsigned);


    private: // DISABLED FUNCTIONS
        //! DISABLE default constructor.
        pw_slk_trace_file(void);
        //! DISABLE copy constructor.
        pw_slk_trace_file(const pw_slk_trace_file&);
        //! DISABLE assignment operator.
        pw_slk_trace_file& operator=(const pw_slk_trace_file&);
}; // pw_slk_trace_file


// ---------------------------------------------------------------------------
/**
 * @brief Create a new SLK trace file recording power variations.
 * @param[in] p_name_pt Output SLK trace file name.
 * @return New SLK trace file.
 */
    pw_trace_file* pw_create_slk_trace_file(const char* p_name_pt);


// ---------------------------------------------------------------------------
/**
 * @brief Close a SLK trace file recording power variations.
 * @param[in] p_file_pt Output SLK trace file.
 */
void pw_close_slk_trace_file(pw_trace_file* p_file_pt);


} // namespace sc_pwr


#endif // _PW_SLK_TRACE_
