// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_energy.h,v 1.1 2011/06/30 08:53:15 my294 Exp $

/** @file pw_energy.h
 * @brief Define energy data type.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 */

#ifndef _PW_ENERGY_
#define _PW_ENERGY_

#include <iostream>
#include <string>

#include <systemc>



// ===========================================================================
namespace sc_pwr
{


// ---------------------------------------------------------------------------
//! Standard physical energy unit.
enum pw_energy_unit
{
    //! Femto Joule = 10e-15
    PW_fJ   = 0,
    //! Pico Joule  = 10e-12
    PW_pJ   = 1,
    //! Nano Joule  = 10e-9
    PW_nJ   = 2,
    //! Micro Joule = 10e-6
    PW_uJ   = 3,
    //! Milli Joule = 10e-3
    PW_mJ   = 4,
    //! Joule = 10e0
    PW_JOULE = 5,
};


// ---------------------------------------------------------------------------
// INCREMENT and DECREMENT OPERATORS

//! Prefix INCREMENT of energy unit.
pw_energy_unit& operator++(pw_energy_unit&);
//! Postfix INCREMENT of energy unit.
pw_energy_unit operator++(pw_energy_unit&,
                          int);
//! Prefix DECREMENT of energy unit.
pw_energy_unit& operator--(pw_energy_unit&);
//! Postfix DECREMENT of energy unit.
pw_energy_unit operator--(pw_energy_unit&,
                          int);


// ---------------------------------------------------------------------------
//! Output stream operator of energy unit object.
std::ostream& operator<<(std::ostream&,
                         const pw_energy_unit&);


// ---------------------------------------------------------------------------
//! Intput stream operator of energy unit object.
std::istream& operator>>(std::istream&,
                         pw_energy_unit&);


// ---------------------------------------------------------------------------
/**
 * @brief Represent a energy value using the standard unit (JOULE).
 * @note Arithmetic operations generate a report of severity PW_WARNING or 
 * PW_ERROR (division by zero) when an overflow is detected.  In this case it 
 * is initialized to 0 or to the max energy value (according to the performed 
 * arithmetic operation).
 */
class pw_energy
{
    public: // CONSTRUCTORS
        /** @brief Default constructor.
         * @note
         * - Initialized to a value of PW_ZERO_ENERGY.
         */
        pw_energy(void);

        /**
         * Constructor of a energy value.
         * @param[in] p_value Raw energy value.
         * @param[in] p_unit Energy unit multiple of the raw value.
         * @pre The parameter energy value should be equal to 0, or >= to the 
         * energy resoultion and <= to the max energy value, otherwise a report 
         * of severity PW_WARNING or PW_ERROR is thrown.  In this last case, 
         * the constructed energy value will be equal to 0 or to the maximal 
         * energy value (according to the parameters value).
         */
        pw_energy(double p_value,
                  pw_energy_unit p_unit);

        //! Copy constructor.
        pw_energy(const pw_energy&);

        //! Assignment operator.
        pw_energy& operator=(const pw_energy&);

    public:  // ARITHMETIC ASSIGNMENT OPERATORS
        //! Assignment by SUM.
        pw_energy& operator+=(const pw_energy&);
        //! Assignment by DIFFERENCE.
        pw_energy& operator-=(const pw_energy&);
        //! Assignment by MULTIPLICATION.
        pw_energy& operator*=(double);
        //! Assignment by DIVISION.
        pw_energy& operator/=(double);

        //! Prefix INCREMENT.
        pw_energy& operator++(void);
        //! Postfix INCREMENT.
        pw_energy operator++(int);
        //! Prefix DECREMENT.
        pw_energy& operator--(void);
        //! Postfix DECREMENT
        pw_energy operator--(int);

    private: // RELATIONAL OPERATORS
        //! Relational EQUAL operator.
        friend bool operator==(const pw_energy&,
                               const pw_energy&);
        //! Relational NOT EQUAL operator.
        friend bool operator!=(const pw_energy&,
                               const pw_energy&);
        //! Relational LESS THAN operator.
        friend bool operator<(const pw_energy&,
                              const pw_energy&);
        //! Relational GREATER THAN operator.
        friend bool operator>(const pw_energy&,
                              const pw_energy&);
        //! Relational LESS THAN OR EQUAL operator.
        friend bool operator<=(const pw_energy&,
                               const pw_energy&);
        //! Relational GREATER THAN OR EQUAL operator.
        friend bool operator>=(const pw_energy&,
                               const pw_energy&);



 public: // MISCELLANOUS METHODS
	//! Round to two three sig figures
	pw_energy round3sf() const;
	

        //! Return a string representing its pysical value.
        const std::string to_string(void) const;

        //! Return it raw value (multiple of the energy resolution).
        sc_dt::uint64 value(void) const;

        //! Return it raw value (multiple of the energy resolution).
        double to_double(void) const;

        //! Return its value in joules
        double to_joules(void) const;

        //! Return the maximal energy value.
        static pw_energy max(void);

    private: // MSICELLANOUS OPERATORS AND FUNCTIONS
        //! Set the energy resolution
        friend void pw_set_energy_resolution(double,
                                             const pw_energy_unit);

        //! Return the current energy resolution.
        friend pw_energy pw_get_energy_resolution(void);

    private: // PRIVATE MENBERS
        //! Current energy value.
        sc_dt::uint64 a_value;

        //! Energy resolution in femto joule.
        static double a_resolution_value;
        //! True if a pw_energy different of 0 have been construted.
        static bool a_is_resolution_fixed;
}; // class pw_energy


// ---------------------------------------------------------------------------
//! Relational EQUAL operator.
bool operator==(const pw_energy&,
                const pw_energy&);

// ---------------------------------------------------------------------------
//! Relational NOT EQUAL operator.
bool operator!=(const pw_energy&,
                const pw_energy&);

// ---------------------------------------------------------------------------
//! Relational LESS THAN operator.
bool operator<(const pw_energy&,
               const pw_energy&);

// ---------------------------------------------------------------------------
//! Relational GREATER THAN operator.
bool operator>(const pw_energy&,
               const pw_energy&);

// ---------------------------------------------------------------------------
//! Relational LESS THAN OR EQUAL operator.
bool operator<=(const pw_energy&,
                const pw_energy&);

// ---------------------------------------------------------------------------
//! Relational GREATER THAN OR EQUAL operator.
bool operator>=(const pw_energy&,
                const pw_energy&);

// ---------------------------------------------------------------------------
//! ADDITION operator.
pw_energy operator+(const pw_energy&,
                    const pw_energy&);

// ---------------------------------------------------------------------------
//! SUBSTRACTION operator.
pw_energy operator-(const pw_energy&,
                    const pw_energy&);

// ---------------------------------------------------------------------------
//! MULTIPLICATION with constant operator.
pw_energy operator*(double,
                    const pw_energy&);

// ---------------------------------------------------------------------------
//! MULTIPLICATION with constant operator.
pw_energy operator*(const pw_energy&,
                    double);

// ---------------------------------------------------------------------------
//! DIVISION operator.
double operator/(const pw_energy&,
                 const pw_energy&);

// ---------------------------------------------------------------------------
//! DIVISION with constant operator.
pw_energy operator/(const pw_energy&,
                    double);


// ---------------------------------------------------------------------------
// Output stream operator of energy object.
std::ostream& operator<<(std::ostream&,
                         const pw_energy&);

// ---------------------------------------------------------------------------
// Intput stream operator of energy object.
std::istream& operator>>(std::istream&,
                         pw_energy&);


// ---------------------------------------------------------------------------
/**
 * @brief Set the energy resoultion.
 * @param[in] p_value Raw value of the energy resolution.
 * @param[in] p_unit Energy unit of the energy resolution.
 * @pre This function has to be called during the elaboration phase, otherwise 
 * a report of severity PW_ERROR is thrown.
 * @pre This function can only be called one time, otherwise a report of 
 * severity PW_ERROR is thrown.
 * @pre This function can only be called before any pw_energy different from 0 
 * are constructed, otherwise a report of severity PW_ERROR is thrown.
 * @pre The p_value value should be a energy of 10, otherwise a report of 
 * severity PW_WARNING is thrown.
 * @note The default energy resolution is equal to 1 pico joule. 
 */
void pw_set_energy_resolution(double p_value,
                              pw_energy_unit p_unit);


// ---------------------------------------------------------------------------
//! Return the current energy resolution.
pw_energy pw_get_energy_resolution(void);


// ---------------------------------------------------------------------------
//! Constant for representing a energy of 0 joule.
extern const pw_energy PW_ZERO_ENERGY;


} // namespace sc_pwr


#endif // _PW_ENERGY_
