// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_area.h,v 1.1 2011/08/22 13:39:59 my294 Exp $

/** @file pw_area.h
 * @brief Define area data type.
 * @author adapted from pw_area.h Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 */

#ifndef _PW_AREA_
#define _PW_AREA_

#include <iostream>
#include <string>

#include <systemc>

#include "pw_length.h"


// ===========================================================================
namespace sc_pwr
{


// ---------------------------------------------------------------------------
//! Standard physical area unit.
enum pw_area_unit
{
    //! Square nanometer = 10e-18
  PW_sqnm   = 0,
    //! Square micron = 10e-12
  PW_squm   = 1,
    //! Square millimetre = 10-e6
  PW_sqmm   = 2,
    //! Square centimeter = 10-e4
  PW_sqcm   = 3,
  //! Square metre = 10e0
  PW_sqm   = 4,
};


// ---------------------------------------------------------------------------
// INCREMENT and DECREMENT OPERATORS

//! Prefix INCREMENT of area unit.
pw_area_unit& operator++(pw_area_unit&);
//! Postfix INCREMENT of area unit.
pw_area_unit operator++(pw_area_unit&,
                          int);
//! Prefix DECREMENT of area unit.
pw_area_unit& operator--(pw_area_unit&);
//! Postfix DECREMENT of area unit.
pw_area_unit operator--(pw_area_unit&,
                          int);


// ---------------------------------------------------------------------------
//! Output stream operator of area unit object.
std::ostream& operator<<(std::ostream&,
                         const pw_area_unit&);


// ---------------------------------------------------------------------------
//! Intput stream operator of area unit object.
std::istream& operator>>(std::istream&,
                         pw_area_unit&);


// ---------------------------------------------------------------------------
/**
 * @brief Represent a area value using the standard unit (METER).
 * @note Arithmetic operations generate a report of severity PW_WARNING or 
 * PW_ERROR (division by zero) when an overflow is detected.  In this case it 
 * is initialized to 0 or to the max area value (according to the performed 
 * arithmetic operation).
 */
class pw_area
{
    public: // CONSTRUCTORS
        /** @brief Default constructor.
         * @note
         * - Initialized to a value of PW_ZERO_METER.
         */
        pw_area(void);

        /**
         * Constructor of a area value.
         * @param[in] p_value Raw area value.
         * @param[in] p_unit Area unit multiple of the raw value.
         * @pre The parameter area value should be equal to 0, or >= to the 
         * area resoultion and <= to the max area value, otherwise a report 
         * of severity PW_WARNING or PW_ERROR is thrown.  In this last case, 
         * the constructed area value will be equal to 0 or to the maximal 
         * area value (according to the parameters value).
         */
        pw_area(double p_value, pw_area_unit p_unit);

        //! Copy constructor.
        pw_area(const pw_area&);

        //! Assignment operator.
        pw_area& operator=(const pw_area&);

    public:  // ARITHMETIC ASSIGNMENT OPERATORS
        //! Assignment by SUM.
        pw_area& operator+=(const pw_area&);
        //! Assignment by DIFFERENCE.
        pw_area& operator-=(const pw_area&);
        //! Assignment by MULTIPLICATION.
        pw_area& operator*=(double);
        //! Assignment by DIVISION.
        pw_area& operator/=(double);

        //! Prefix INCREMENT.
        pw_area& operator++(void);
        //! Postfix INCREMENT.
        pw_area operator++(int);
        //! Prefix DECREMENT.
        pw_area& operator--(void);
        //! Postfix DECREMENT
        pw_area operator--(int);

    private: // RELATIONAL OPERATORS
        //! Relational EQUAL operator.
        friend bool operator==(const pw_area&,
                               const pw_area&);
        //! Relational NOT EQUAL operator.
        friend bool operator!=(const pw_area&,
                               const pw_area&);
        //! Relational LESS THAN operator.
        friend bool operator<(const pw_area&,
                              const pw_area&);
        //! Relational GREATER THAN operator.
        friend bool operator>(const pw_area&,
                              const pw_area&);
        //! Relational LESS THAN OR EQUAL operator.
        friend bool operator<=(const pw_area&,
                               const pw_area&);
        //! Relational GREATER THAN OR EQUAL operator.
        friend bool operator>=(const pw_area&,
                               const pw_area&);



    public: // MISCELLANOUS METHODS
        //! Return a string representing its pysical value.
        const std::string to_string(void) const;

        //! Return it raw value (multiple of the area resolution).
        sc_dt::uint64 value(void) const;

        //! Return it raw value (multiple of the area resolution).
        double to_double(void) const;

        //! Return its value in sq_metres
        double to_sqmeters(void) const;

        //! Return the maximal area value.
        static pw_area max(void);

	//! Round to two three sig figures
	pw_area round3sf() const;

    private: // MSICELLANOUS OPERATORS AND FUNCTIONS
        //! Set the area resolution
        friend void pw_set_area_resolution(double, const pw_area_unit);

        //! Return the current area resolution.
        friend pw_area pw_get_area_resolution(void);

    private: // PRIVATE MEMBERS

        //! Current area value.
        sc_dt::uint64 a_value;

 public: // needed outside class in pw_area.cpp (djg tmp hack)
        //! Area resolution in femto meter.
        static double a_resolution_value;
        //! True if a pw_area different of 0 have been constructed.
        static bool a_is_resolution_fixed;
}; // class pw_area


// ---------------------------------------------------------------------------
//! Relational EQUAL operator.
bool operator==(const pw_area&,
                const pw_area&);

// ---------------------------------------------------------------------------
//! Relational NOT EQUAL operator.
bool operator!=(const pw_area&,
                const pw_area&);

// ---------------------------------------------------------------------------
//! Relational LESS THAN operator.
bool operator<(const pw_area&,
               const pw_area&);

// ---------------------------------------------------------------------------
//! Relational GREATER THAN operator.
bool operator>(const pw_area&,
               const pw_area&);

// ---------------------------------------------------------------------------
//! Relational LESS THAN OR EQUAL operator.
bool operator<=(const pw_area&,
                const pw_area&);

// ---------------------------------------------------------------------------
//! Relational GREATER THAN OR EQUAL operator.
bool operator>=(const pw_area&,
                const pw_area&);

// ---------------------------------------------------------------------------
//! ADDITION operator.
pw_area operator+(const pw_area&,
                    const pw_area&);

// ---------------------------------------------------------------------------
//! SUBSTRACTION operator.
pw_area operator-(const pw_area&,
                    const pw_area&);

// ---------------------------------------------------------------------------
//! MULTIPLICATION with constant operator.
pw_area operator*(double,
                    const pw_area&);

// ---------------------------------------------------------------------------
//! MULTIPLICATION with constant operator.
pw_area operator*(const pw_area&,
                    double);

// ---------------------------------------------------------------------------
//! DIVISION operator.
double operator/(const pw_area&,
                 const pw_area&);

// ---------------------------------------------------------------------------
//! DIVISION operator.
pw_length operator/(const pw_area&,
                 const pw_length&);

// ---------------------------------------------------------------------------
//! DIVISION with constant operator.
pw_area operator/(const pw_area&,
                    double);


// ---------------------------------------------------------------------------
// Output stream operator of area object.
std::ostream& operator<<(std::ostream&,
                         const pw_area&);

// ---------------------------------------------------------------------------
// Intput stream operator of area object.
std::istream& operator>>(std::istream&,
                         pw_area&);


// ---------------------------------------------------------------------------
/**
 * @brief Set the area resoultion.
 * @param[in] p_value Raw value of the area resolution.
 * @param[in] p_unit Area unit of the area resolution.
 * @pre This function has to be called during the elaboration phase, otherwise 
 * a report of severity PW_ERROR is thrown.
 * @pre This function can only be called one time, otherwise a report of 
 * severity PW_ERROR is thrown.
 * @pre This function can only be called before any pw_area different from 0 
 * are constructed, otherwise a report of severity PW_ERROR is thrown.
 * @pre The p_value value should be a area of 10, otherwise a report of 
 * severity PW_WARNING is thrown.
 * @note The default area resolution is equal to 1 pico meter. 
 */
void pw_set_area_resolution(double p_value,
                              pw_area_unit p_unit);


// ---------------------------------------------------------------------------
//! Return the current area resolution.
pw_area pw_get_area_resolution(void);


// ---------------------------------------------------------------------------
//! Constant for representing a area of 0.
extern const pw_area PW_ZERO_AREA;

// ---------------------------------------------------------------------------
//! MULT of a pair of lenths operator.
class pw_area operator*(const pw_length&, const pw_length&);


} // namespace sc_pwr


#endif // _PW_AREA_
