// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_voltage.h,v 1.1 2011/06/30 08:53:16 my294 Exp $

/** @file pw_voltage.h
 * @brief Define voltage data type.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 */

#ifndef _PW_VOLTAGE_
#define _PW_VOLTAGE_

#include <iostream>
#include <string>

#include <systemc>



// ===========================================================================
namespace sc_pwr
{


// ---------------------------------------------------------------------------
//! Standard physical voltage unit.
enum pw_voltage_unit
{
    //! Femto Volt = 10e-15
    PW_fV   = 0,
    //! Pico Volt  = 10e-12
    PW_pV   = 1,
    //! Nano Volt  = 10e-9
    PW_nV   = 2,
    //! Micro Volt = 10e-6
    PW_uV   = 3,
    //! Milli Volt = 10e-3
    PW_mV   = 4,
    //! Volt = 10e0
    PW_VOLT = 5,
};


// ---------------------------------------------------------------------------
// INCREMENT and DECREMENT OPERATORS

//! Prefix INCREMENT of voltage unit.
pw_voltage_unit& operator++(pw_voltage_unit&);
//! Postfix INCREMENT of voltage unit.
pw_voltage_unit operator++(pw_voltage_unit&,
                          int);
//! Prefix DECREMENT of voltage unit.
pw_voltage_unit& operator--(pw_voltage_unit&);
//! Postfix DECREMENT of voltage unit.
pw_voltage_unit operator--(pw_voltage_unit&,
                          int);


// ---------------------------------------------------------------------------
//! Output stream operator of voltage unit object.
std::ostream& operator<<(std::ostream&,
                         const pw_voltage_unit&);


// ---------------------------------------------------------------------------
//! Intput stream operator of voltage unit object.
std::istream& operator>>(std::istream&,
                         pw_voltage_unit&);


// ---------------------------------------------------------------------------
/**
 * @brief Represent a voltage value using the standard unit (Volt).
 * @note Arithmetic operations generate a report of severity PW_WARNING or 
 * PW_ERROR (division by zero) when an overflow is detected.  In this case it 
 * is initialized to 0 or to the max voltage value (according to the performed 
 * arithmetic operation).
 */
class pw_voltage
{
    public: // CONSTRUCTORS
        /** @brief Default constructor.
         * @note
         * - Initialized to a value of PW_ZERO_VOLT.
         */
        pw_voltage(void);

        /**
         * Constructor of a voltage value.
         * @param[in] p_value Raw voltage value.
         * @param[in] p_unit voltage unit multiple of the raw value.
         * @pre The parameter voltage value should be equal to 0, or >= to the 
         * voltage resolution and <= to the max voltage value, otherwise a report 
         * of severity PW_WARNING or PW_ERROR is thrown.  In this last case, 
         * the constructed voltage value will be equal to 0 or to the maximal 
         * voltage value (according to the parameters value).
         */
        pw_voltage(double p_value,
                  pw_voltage_unit p_unit);

        //! Copy constructor.
        pw_voltage(const pw_voltage&);

        //! Assignment operator.
        pw_voltage& operator=(const pw_voltage&);

    public:  // ARITHMETIC ASSIGNMENT OPERATORS
        //! Assignment by SUM.
        pw_voltage& operator+=(const pw_voltage&);
        //! Assignment by DIFFERENCE.
        pw_voltage& operator-=(const pw_voltage&);
        //! Assignment by MULTIPLICATION.
        pw_voltage& operator*=(double);
        //! Assignment by DIVISION.
        pw_voltage& operator/=(double);

        //! Prefix INCREMENT.
        pw_voltage& operator++(void);
        //! Postfix INCREMENT.
        pw_voltage operator++(int);
        //! Prefix DECREMENT.
        pw_voltage& operator--(void);
        //! Postfix DECREMENT
        pw_voltage operator--(int);

    private: // RELATIONAL OPERATORS
        //! Relational EQUAL operator.
        friend bool operator==(const pw_voltage&,
                               const pw_voltage&);
        //! Relational NOT EQUAL operator.
        friend bool operator!=(const pw_voltage&,
                               const pw_voltage&);
        //! Relational LESS THAN operator.
        friend bool operator<(const pw_voltage&,
                              const pw_voltage&);
        //! Relational GREATER THAN operator.
        friend bool operator>(const pw_voltage&,
                              const pw_voltage&);
        //! Relational LESS THAN OR EQUAL operator.
        friend bool operator<=(const pw_voltage&,
                               const pw_voltage&);
        //! Relational GREATER THAN OR EQUAL operator.
        friend bool operator>=(const pw_voltage&,
                               const pw_voltage&);

    public: // MISCELLANOUS METHODS
        //! Return a string representing its pysical value.
        const std::string to_string(void) const;

        //! Return it raw value (multiple of the voltage resolution).
        sc_dt::uint64 value(void) const;

        //! Return it raw value (multiple of the voltage resolution).
        double to_double(void) const;

        //! Return its value in Volts
        double to_volts(void) const;

        //! Return the maximal voltage value.
        static pw_voltage max(void);

    private: // MSICELLANOUS OPERATORS AND FUNCTIONS
        //! Set the voltage resolution
        friend void pw_set_voltage_resolution(double,
                                             const pw_voltage_unit);

        //! Return the current voltage resolution.
        friend pw_voltage pw_get_voltage_resolution(void);

    private: // PRIVATE MENBERS
        //! Current voltage value.
        sc_dt::uint64 a_value;

        //! voltage resolution in femto Volt.
        static double a_resolution_value;
        //! True if a pw_voltage different of 0 have been construted.
        static bool a_is_resolution_fixed;
}; // class pw_voltage


// ---------------------------------------------------------------------------
//! Relational EQUAL operator.
bool operator==(const pw_voltage&,
                const pw_voltage&);

// ---------------------------------------------------------------------------
//! Relational NOT EQUAL operator.
bool operator!=(const pw_voltage&,
                const pw_voltage&);

// ---------------------------------------------------------------------------
//! Relational LESS THAN operator.
bool operator<(const pw_voltage&,
               const pw_voltage&);

// ---------------------------------------------------------------------------
//! Relational GREATER THAN operator.
bool operator>(const pw_voltage&,
               const pw_voltage&);

// ---------------------------------------------------------------------------
//! Relational LESS THAN OR EQUAL operator.
bool operator<=(const pw_voltage&,
                const pw_voltage&);

// ---------------------------------------------------------------------------
//! Relational GREATER THAN OR EQUAL operator.
bool operator>=(const pw_voltage&,
                const pw_voltage&);

// ---------------------------------------------------------------------------
//! ADDITION operator.
pw_voltage operator+(const pw_voltage&,
                    const pw_voltage&);

// ---------------------------------------------------------------------------
//! SUBSTRACTION operator.
pw_voltage operator-(const pw_voltage&,
                    const pw_voltage&);

// ---------------------------------------------------------------------------
//! MULTIPLICATION with constant operator.
pw_voltage operator*(double,
                    const pw_voltage&);

// ---------------------------------------------------------------------------
//! MULTIPLICATION with constant operator.
pw_voltage operator*(const pw_voltage&,
                    double);

// ---------------------------------------------------------------------------
//! DIVISION operator.
double operator/(const pw_voltage&,
                 const pw_voltage&);

// ---------------------------------------------------------------------------
//! DIVISION with constant operator.
pw_voltage operator/(const pw_voltage&,
                    double);


// ---------------------------------------------------------------------------
// Output stream operator of energy object.
std::ostream& operator<<(std::ostream&,
                         const pw_voltage&);

// ---------------------------------------------------------------------------
// Intput stream operator of energy object.
std::istream& operator>>(std::istream&,
                         pw_voltage&);


// ---------------------------------------------------------------------------
/**
 * @brief Set the voltage resolution.
 * @param[in] p_value Raw value of the voltage resolution.
 * @param[in] p_unit voltage unit of the voltage resolution.
 * @pre This function has to be called during the elaboration phase, otherwise 
 * a report of severity PW_ERROR is thrown.
 * @pre This function can only be called one time, otherwise a report of 
 * severity PW_ERROR is thrown.
 * @pre This function can only be called before any pw_voltage different from 0 
 * are constructed, otherwise a report of severity PW_ERROR is thrown.
 * @pre The p_value value should be a energy of 10, otherwise a report of 
 * severity PW_WARNING is thrown.
 * @note The default voltage resolution is equal to 1 pico Volt. 
 */
void pw_set_voltage_resolution(double p_value,
                              pw_voltage_unit p_unit);


// ---------------------------------------------------------------------------
//! Return the current voltage resolution.
pw_voltage pw_get_voltage_resolution(void);


// ---------------------------------------------------------------------------
//! Constant for representing a energy of 0 Volt.
extern const pw_voltage PW_ZERO_VOLT;


} // namespace sc_pwr


#endif // _pw_voltage_
