// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_vcd_trace.h,v 1.1 2011/06/30 08:53:16 my294 Exp $

/** @file pw_vcd_trace.h
 * @brief Tracing facilities for generating power VCD consumption reports.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 */

#ifndef _PW_VCD_TRACE_
#define _PW_VCD_TRACE_

#include <string>

#include <systemc>

#include "pw_trace.h"

// ===========================================================================
namespace sc_pwr
{

#define PW_MAX_PLOT_COLS 32 // Maximum number of columns to have in a single (gnu-)plot trace file.
  // Generally a user will want to have multiple 'vcd' trace files rather than increasing this per-file limit. 

  // ---------------------------------------------------------------------------
  /**
   * @brief VCD trace file recording power and utlisation variations in the time domain.
   * 
   * Multiple sc_objects can be associated to this trace file for recording 
   * their power variations.  
   */
  class pw_vcd_trace_file:
      public pw_trace_file
  {
      public: // CONSTRUCTOR AND DESTRUCTOR
	  /**
	   * @brief VCD trace file constructor.
	   * @param[in] p_name_pt Output VCD trace file name.
	   * @note This function should be called indirectly via the function 
	   * @param[in] p_plot_type decaying or average style.
	   * pw_create_vcd_trace_file.
	   */

    pw_vcd_trace_file(const char* p_vcd_name_pt, const char *p_log_name_pt=0, const plot_style_t p_plot_type = PlotStyleDecaying);

	  //! Virtual destructor for polymorphism.
	  virtual ~pw_vcd_trace_file(void);

	  
  public: // TRACING FUNCTIONS
	  /**
	   * @brief Trace the power consumption of a SystemC object.
	   * @param[in,out] p_obj the SystemC object.
	   * @param[in] p_str Identification string associated to p_obj.
	   * @param[in] p_children Automatically trace child power modules (included in total or separately).
	   * @param[in] p_plot_control how to include this object in the (gnu-)plot style output file (see VCD forms).
	   */
	  void trace(sc_core::sc_object& p_obj,
		     const std::string& p_str,
		     trace_t p_children = no_children,
		     plot_control_t p_plot_control = DoNotPlot
		     );
	  
	  
  public: // STANDARD MANAGEMENT FUNCTIONS
	  /**
	   * @brief Set the time unit use by a trace file for recording events.
	   * @param[in] p_value Raw time value.
	   * @param[in] p_unit Time unit multiple of the raw value.
	   */
	  void set_time_unit(double p_value,
			     sc_core::sc_time_unit p_unit);
	  
        /**
         * @brief Write a comment in a trace file.
         * @param[in] p_str The comment to insert.
         */
	  void write_comment(const std::string& p_str);
	  
	  /**
         * @brief Close the file
         */
	  void end_vcd_files();
  protected: // PRIVATE ATTRIBUTES AND METHODS
	  friend class pw_vcd_observer;
	  int alloc_next_plot_column(std::string name);

	  class traced_variable
	  {
	    sc_dt::uint64 lasttime;
	    float vale;
	    traced_variable *next; 
	    int a_plot_col;
	  protected:
	    friend class pw_vcd_observer;
	    friend class pw_vcd_trace_file;
	    bool dirty();
	    const char *vcd_string;
	    std::string name;
	    pw_vcd_trace_file *parent;
	  void log(sc_time, pw_energy);
	  traced_variable(pw_vcd_trace_file *parent, const std::string name, int plot_col);
	  sc_dt::uint64 next_time_request(sc_dt::uint64);
	  float compute(sc_dt::uint64);
	  };
	  
  private: // PRIVATE ATTRIBUTES AND METHODS
	  struct plot_column
	  {
	    std::string name;
	  } Plot_cols[PW_MAX_PLOT_COLS];
	  int n_plot_cols_in_use;

	  plot_style_t a_plot_style;
	  //int limit;
	  sc_dt::uint64 decay_interval;
	  float decay_rate;
	  bool warmed;
	  sc_dt::uint64 local_time;
	  char a_trocnamerr[132];
	  FILE *a_plot_fd;
	  FILE *a_vcd_fd;
	  class traced_variable *a_traces;
	  
	  //! Trace file name
	  const std::string a_vcd_file_name;
	  //! Plot file name
	  const std::string a_plot_file_name;
	  
	  traced_variable *add_trace(int util, pw_energy startingvale, const std::string name, int plot_col);
	  char *inc_trocnamer();
	  void rdump(traced_variable *trace, float vale);
	  void start_vcd_files();
	  void warmup_vcd_files();
	  void yupdate(sc_dt::uint64);
	  int findscope(char *scope, const char *name);
	  void osci_code_for_timescale();
	
	
  private: // DISABLED FUNCTIONS
        //! DISABLE default constructor.
        pw_vcd_trace_file(void);
        //! DISABLE copy constructor.
        pw_vcd_trace_file(const pw_vcd_trace_file&);
        //! DISABLE assignment operator.
        pw_vcd_trace_file& operator=(const pw_vcd_trace_file&);
  }; // pw_vcd_trace_file
      

      // ---------------------------------------------------------------------------
      /**
       * @brief Create a new VCD trace file recording power variations.
       * @param[in] p_name_pt Output VCD trace file name (or null pointer).
       * @param[in] p_name_pt Output (gnu-)plot trace file name (or null pointer).
       * @return New VCD trace file.
       */
      pw_trace_file* pw_create_vcd_trace_file(const char* p_vcd_name_pt, const char* p_plot_name_pt=0, plot_style_t p_plot_style=PlotStyleAveraging);
      
      
      // ---------------------------------------------------------------------------
      /**
       * @brief Close a VCD trace file recording power variations.
       * @param[in] p_file_pt Output VCD trace file.
       */
      void pw_close_vcd_trace_file(pw_trace_file* p_file_pt);
      
      
} // namespace sc_pwr


#endif // _PW_VCD_TRACE_
