// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_txt_trace.h,v 1.2 2011/07/25 15:32:39 my294 Exp $

/** @file pw_txt_trace.h
 * @brief Tracing facilities for generating statistics on power consumption.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 */

#ifndef _PW_TXT_TRACE_
#define _PW_TXT_TRACE_

#include <list>
#include <vector>
#include <string>
#include <utility>
#include <fstream>
#include <systemc>
#include "pw_trace.h"
#include "pw_stat_observer_base.h"
//#include "pw_stat_observer.h"

// ===========================================================================
namespace sc_pwr
{

// ---------------------------------------------------------------------------
/**
 * @brief Textual trace file recording power statistics.
 * 
 * Multiple sc_objects can be associated to this trace file for recording 
 * their power statistics.
 */
class pw_txt_trace_file:
    public pw_observer_if,
    public pw_trace_file
{
    public: // CONSTRUCTOR AND DESTRUCTOR
        /**
         * @brief Textual trace file constructor.
         * @param[in] p_name_pt Output trace file name.
         * @note This function should be called indirectly via the function 
         * pw_create_txt_trace_file.
         * @note ".txt" is automatically appended to p_name_pt.
         */
  pw_txt_trace_file(const char* p_name_pt);

        //! Virtual destructor for polymorphism.
        virtual ~pw_txt_trace_file(void);

    public: // SUBJECT MANAGEMENT FUNCTIONS
        /**
         * @brief Method called by an observer used by this trace file.
         * @param[in] p_subject Subject which called this observer.
         *
         * The subject monitored by this observer use this method for 
         * indicating its eminent destruction.
         */
        virtual void update(pw_subject& p_subject);

    public: // TRACING FUNCTIONS
        /**
         * @brief Trace the power consumption of a SystemC object.
         * @param[in,out] p_obj the SystemC object.
         * @param[in] p_str Identification string associated to p_obj.
	 * @param[in] p_do_children - should be sum_children or also_trace_children.
	 * @param[in] p_plot_control how to include this object in the (gnu-)plot style output file (see VCD forms).
         */
        void trace(sc_core::sc_object& p_obj,
                   const std::string& p_str,
		   trace_t p_do_all_children,  // summed or separately
		   plot_control_t p_plot_control = DoNotPlot
		   );

    public: // STANDARD MANAGEMENT FUNCTIONS
        /**
         * @brief Set the time unit use by a trace file for recording events.
         * @param[in] p_value Raw time value.
         * @param[in] p_unit Time unit multiple of the raw value.
         */
        virtual void set_time_unit(double p_value,
                                   sc_core::sc_time_unit p_unit);
        /**
         * @brief Write a comment in a trace file.
         * @param[in] p_str The comment to insert.
         */
        virtual void write_comment(const std::string& p_str);

    private: // PRIVATE DUMPING FUNCTIONS
        //! Dump the informations recorded by a base statistical observer.
        void dump_observer(sc_time, pw_stat_observer_record& pw_stat_observer_record);

        //! Dump ratio information.
        std::string dump_ratio(void) const;

        //! Dump extra information recorded by a statistical observer.
        //void dump_observer(pw_stat_observer&, const std::string&);


    private: // PRIVATE ATTRIBUTES
        //! Output tracing file.
        std::ofstream a_file;
        //! Output tracing file name.
        const std::string a_file_name;
        //! List of the observers used by this trace file.
        std::vector< pw_stat_observer_record* > a_observers_pt;


        //! Formatted energy statistics
	std::ostringstream a_energy_stat;
        //! Formatted power statistics
        std::ostringstream a_power_stat;
        //! List of the formatted statisitics.
        std::list< std::string > a_observer_stat;

        //! Comment to show
        std::string a_comment;

	//! Table horiz bar
        std::string a_hline_sep;

	//! Are power summaries to be printed?  Defaults true - but is turned off by exceptions.
	bool a_do_powers;
	

    private: // DISABLED FUNCTIONS
        //! DISABLE default constructor.
        pw_txt_trace_file(void);
        //! DISABLE copy constructor.
        pw_txt_trace_file(const pw_txt_trace_file&);
        //! DISABLE assignment operator.
        pw_txt_trace_file& operator=(const pw_txt_trace_file&);
}; // pw_txt_trace_file

// ---------------------------------------------------------------------------
/**
 * @brief Create a new TXT trace file recording power variations.
 * @param[in] p_name_pt Output TXT trace file name.
 * @return New TXT trace file.
 */
    pw_trace_file* pw_create_txt_trace_file(const char* p_name_pt);

// ---------------------------------------------------------------------------
/**
 * @brief Close a TXT trace file recording power variations.
 * @param[in] p_file_pt Output TXT trace file.
 */
void pw_close_txt_trace_file(pw_trace_file* p_file_pt);

} // namespace sc_pwr


#endif // _PW_TXT_TRACE_
