// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_trace.h,v 1.1 2011/06/30 08:53:15 my294 Exp $

/** @file pw_trace.h
 * @brief Tracing facilities for generating power consumption reports.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 */

#ifndef _PW_TRACE_
#define _PW_TRACE_

#include <string>
#include <vector>

#include <systemc>

#include "pw_module_base.h"



// ===========================================================================
namespace sc_pwr
{


  /**
   * @brief  Set or recall the title for the current simulation.
   */
  std::string simulation_title(const char *set = 0);

  /**
   * @brief  Two styles of graphical rendition of dirac pulses are possible: low-pass impulse response (decaying) or averaged over interval.
   */
   typedef enum { PlotStyleAveraging, PlotStyleDecaying } plot_style_t;

  // As well as writing a VCD file, a textual plot file can be written
  // for use with gnuplot etc which includes selected traces accoding to this
  // flag: they may have all accounts as separate fields or all accounts summed.


  /**
   * @brief Auxiliary graphical plot mode
   */
 
  // Gnuplot expects whitespace-separated columns of data and specific columms can be extracted using commands such as 
  // gnuplot>  plot "plot.dat" using 1:2 title 'DRAM', "plot.dat" using 1:3 title 'CPU'
  
  typedef enum { DoNotPlot, PlotEachAccount, PlotCombinedAccounts } plot_control_t;

  
  /**
   * @brief How to trace children. They can be disregarded, included in the sum for this module or included in their own reporting lines.  
   */
  typedef enum { no_children, sum_children, also_trace_children } trace_t;



  
  // ---------------------------------------------------------------------------
  //! Base class for generating different kinds of consumption reports.
  class pw_trace_file
  {
  public: // CONSTRUCTOR AND DESTRUCTOR
 
    
    //! Default constructor
    pw_trace_file();
    
    
    //! Virtual destructor for polymorphism.
    virtual ~pw_trace_file(void);
    
    
  public: // TRACING FUNCTIONS
    
    /**
     * @brief Trace all power records as single line entries in the plot file.
     * @param[in] p_plot_control Set this to create a results column in the (gnu-)plot style file if provided to the VCD tracer.
     */
    void pw_trace_all(pw_trace_file* p_file_pt, plot_control_t p_plot_control = DoNotPlot);
    
    /**
     * @brief Trace the power consumption of a SystemC object.
     * @param[in,out] p_obj the SystemC object.
     * @param[in] p_str Identification string associated to p_obj.
     * @param[in] p_plot_control Set this to create a results column in the (gnu-)plot style file if provided to the VCD tracer.
     */
    virtual void trace(sc_core::sc_object& p_obj,
		       const std::string& p_str,
		       trace_t p_all_children,
		       plot_control_t p_plot_control = DoNotPlot) = 0;

    
  public: // STANDARD MANAGEMENT FUNCTIONS
    /**
         * @brief Set the time unit use by a trace file for recording events.
         * @param[in] p_value Raw time value.
         * @param[in] p_unit Time unit multiple of the raw value.
         */
    virtual void set_time_unit(double p_value,
			       sc_core::sc_time_unit p_unit) = 0;
    
    /**
     * @brief Write a comment in a trace file.
     * @param[in] p_str The comment to insert.
     */
    virtual void write_comment(const std::string& p_str) = 0;
    
    
        /**
         * @brief  Trace all modules (start from root if Null is passed in).
         * @param[in] p_obj (start from root if Null is passed in).
         * @param[in] p_do_children - should be sum_children or also_trace_children.
         */
	//! 
    void trace_all_below(sc_object *p_obj, trace_t p_do_children, int p_depth=0);
    
    
  private: // DISABLED FUNCTIONS
    //! DISABLE copy constructor.
    pw_trace_file(const pw_trace_file&);
    //! DISABLE assignment operator.
    pw_trace_file& operator=(const pw_trace_file&);
  };
  

// ---------------------------------------------------------------------------
/**
 * @brief Trace the power consumption of a SystemC object.
 * @param[in,out] p_file_pt The target trace file.
 * @param[in,out] p_obj The traced SystemC object.
 * @param[in] p_do_children how to include child objects in this trace record.
 * @param[in] p_str Identification string if do not wish to use object's own name.
 * @param[in] p_plot_control how to include this object in the (gnu-)plot style output file (see VCD forms).
 */
void pw_trace(pw_trace_file* p_file_pt,
              sc_core::sc_object& p_obj,
              const std::string& p_str,
	      trace_t p_do_children=no_children,
	      plot_control_t p_plot_control = DoNotPlot
	      );

/**
 * @brief Trace the power consumption of a SystemC object.
 * @param[in,out] p_file_pt The target trace file.
 * @param[in,out] p_obj The traced SystemC object.
 * @param[in] p_do_children how to include child objects in this trace record.
 * @param[in] p_plot_control how to include this object in the (gnu-)plot style output file (see VCD forms).
 */

void pw_trace(pw_trace_file* p_file_pt,
              sc_core::sc_object& p_obj,
	      trace_t p_do_children=no_children,
	      plot_control_t p_plot_control = DoNotPlot
	      );


// ---------------------------------------------------------------------------
/**
 * @brief Set the time unit use by a trace file for recording events.
 * @param[in,out] p_file_pt The target trace file.
 * @param[in] p_value Raw time value.
 * @param[in] p_unit Time unit multiple of the raw value.
 */
void pw_set_time_unit(pw_trace_file* p_file_pt,
                      double p_value,
                      sc_core::sc_time_unit p_unit);


// ---------------------------------------------------------------------------
/**
 * @brief Write a comment in a trace file.
 * @param[in,out] p_file_pt The target trace file.
 * @param[in] p_str The comment to insert.
 */
void pw_write_comment(pw_trace_file* p_file_pt,
                      const std::string& p_str);


} // namespace sc_pwr


#endif // _PW_TRACE_
