// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_stat_observer_base.h,v 1.2 2011/07/25 15:32:39 my294 Exp $

/** @file pw_stat_observer_base.h
 * @brief Makes statistic on power variation.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 *
 * This class is used by the TXT and CSV trace file for generating stastics on 
 * the power variation.
 */

#ifndef _PW_STAT_OBSERVER_BASE_
#define _PW_STAT_OBSERVER_BASE_

#include <list>
#include <string>
#include <utility>
#if __GNUC__ == 4
#   include <tr1/unordered_map>
#else
#   include <map>
#endif

#include <systemc>
#include "pw_power.h"
#include "pw_energy.h"
#include "pw_subject.h"
#include "pw_module_base.h"
#include "pw_observer_base.h"
#include "pw_accounting_base.h"
#include "pw_trace.h"

// ===========================================================================
namespace sc_pwr
{

// ---------------------------------------------------------------------------
// Pre-declaration for friendship definition.
class sc_power_flush_callback;


// ---------------------------------------------------------------------------
/**
 * @brief Record statics on the power variation of a SystemC component.
 */
class pw_stat_observer_base: 
   public pw_accounting_base
{
    public: // CONSTRUCTOR AND DESTRUCTOR
        /**
         * @brief Constructor of a statistic observer.
         * @param[in] p_obj SystemC object monitored by this observer.
         * @param[in] p_name SystemC attribute name.
         */
        pw_stat_observer_base(sc_core::sc_object& p_obj,
                              const std::string& p_name);

        // virtual destructor
        virtual ~pw_stat_observer_base(void);



    public: // SUBJECT MANAGEMENT FUNCTIONS
        /**
         * @brief Method called by a subject monitored by this observer.
         * @param[in] p_subject Subject which called this observer.
         *
         * The subject monitored by this observer use this method for 
         * indicating a change of its state.
         */
        virtual void update(pw_subject& p_subject);

    public: // ACCESS FUNCTIONS
 
        //! True if the simulation is terminated
        bool is_simulation_terminated(void) const;


        //! Return the global energy consumed by the system.
        static pw_energy get_global_energy(int);
        //! Return the global energy consumed by the system - all accounts.
        static pw_energy get_global_energy();

        //! Return the last update time globally (and fold global account).
	static const sc_time get_global_last_update();


    public: // MISCELLANOUS FUNCTIONS
        //! Return an informative string about this observer.
        virtual std::string to_string(void) const;

    protected: // POWER UPDATE FUNCTIONS

        //! Update transaction energy/power - read transaction event from pw_module base.
        void update_energy_infos(const pw_module_base&);

        //! Update transaction energy/power - read transaction event from pw_module base.
        void update_transaction_infos(const pw_module_base&);

        //! Update the static power value.
        void update_static_power(const pw_module_base&);
        //! Update the static informations (power+energy).
        void update_static_infos(const pw_module_base&);

        //! set the last global update
        static void set_global_last_update(const sc_core::sc_time&);

    protected: // MISCELLANOUS FUNCTIONS
        //! Return an informative string about this observer.
        virtual void end_of_simulation(void);
        //! Check if the simulation properly ended.
        void check_end_of_simulation(void);
        //! Update information before destruction.
        virtual void destructor_prologue(void);

    protected: // PROTECTED ATTRIBUTES
        //! Predicate for proper managing of object destruction
        bool a_is_dumped;

        //! Lock indication the end of the simulation.
        bool a_is_simulation_terminated;

        //! Local for allowing only one warning message.
        static bool g_lock_warning;

 protected: // PRIVATE ATTRIBUTES
	friend class global_power;

        sc_core::sc_time a_utilization;

    private: // DISABLED FUNCTIONS
        //! DISABLE default constructor
        pw_stat_observer_base(void);
        //! DISABLE copy constructor.
        pw_stat_observer_base(const pw_stat_observer_base&);
        //! DISABLE assignment operator.
        pw_stat_observer_base& operator=(const pw_stat_observer_base&);

    private: // FRIEND DECLARATION
        friend class sc_power_flush_callback;
}; // pw_stat_observer_base

    // ---------------------------------------------------------------------------
    // Output stream operator of statiscal oberserver
    std::ostream& operator<<(std::ostream&,
                         const pw_stat_observer_base&);


    
    //! C++ API - function for user code to get an observer.
    pw_stat_observer_base *pw_get_observer(sc_object *p_mod, sc_pwr::trace_t p_do_children);
    //! C++ API - function for user code to get an observer.
    pw_stat_observer_base *pw_get_observer(sc_module *p_mod, sc_pwr::trace_t p_do_children);


  // ---------------------------------------------------------------------------
  //! 

  class pw_stat_observer_record
  {
    public:
    trace_t m_do_children;
    pw_stat_observer_base &m_observer;
    const std::string m_name;

    // Constructor
  pw_stat_observer_record(trace_t p_do_children, pw_stat_observer_base &p_observer, const std::string p_name):
    m_do_children(p_do_children),   
      m_observer(p_observer),
      m_name(p_name)
      {}
  };
  
  

} // namespace sc_pwr


#endif // _PW_STAT_OBSERVER_BASE_
