// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/********************************************************************
 * Copyright (C) 2010 by Verimag                                    *
 * Initial author: Matthieu Moy                                     *
 ********************************************************************/

/*!
  \file pw_param.h
  \brief Manage power-consumption through a set of parameters.

  This is an more flexible alternative to the mode/phase provided by
  TLM_Power:

  - Instead of having only two parameters (mode and phase), this
    mechanism allows an arbitrary number of them, named by strings.

  - Instead of having only a finite number of possible values for each
    parameter, we allow different data-types (like integers, strings)
    for parameter values.

  - The computation of static and dynamic power are no longer limited
    to static text files, but is done through actual C++ code.

  From the user's point of view, one should:

  - Derive a class X from sc_pwr::power_params. This class should:
    - define its parameters (name and type) in its constructors by
      calling sc_pwr::power_params::new_param for each parameter.
    - implement the computation of power consumption by overloading
      sc_pwr::power_params::compute_static_power() and
      sc_pwr::power_params::compute_dynamic_power().

  - Inherit from sc_pwr::pw_param_module<X> where X is the class
    defined above.

  - Use sc_pwr::pw_param_module::set_value() to change the value of a
    parameter, and sc_pwr::pw_param_module::update_power() to
    re-compute the power consumption after such parameter change.
*/
#ifndef PW_PARAM_H
#define PW_PARAM_H

#include <systemc>
#include <map>
#include <pw_module_base.h>
#include <pw_kernel_ids.h>

/*! Converter to string for anything convertible */
template <class T>
std::string string_from(const T &val,
                        std::ios_base & (*f)(std::ios_base&) = std::dec)
{
        std::ostringstream oss(std::ostringstream::out);
        oss<<f<<val;
        return oss.str();
}

namespace sc_pwr
{

extern const char* PW_BAD_PARAM_TYPE_TYPE_;
extern const char* PW_BAD_PARAM_TYPE_MSG_;

extern const char* PW_BAD_PARAM_TYPE_;
extern const char* PW_BAD_PARAM_MSG_;

/*! \brief Base class for parameter value.
 *
 * A parameter value is a type wrapped into a class which inherits from
 * this class. It should have a public member "value" and a static
 * function "get_default()" returning a default value for the type.
 */
class power_param_value_base {
public:
	virtual std::string to_string() = 0;
};

/*! \brief Class for parameter values.
 *
 * This class is a template, it should be refined for each type. The
 * refinement should inherit sc_pwr::power_param_value_base.
 */
template <typename T>
class power_param_value;

template<>
class power_param_value<int> : public power_param_value_base {
public:
	typedef power_param_value<int> this_type;
	int value;
	power_param_value(int v) {
		value = v;
	}
	static this_type *get_default() {
		return new this_type(0);
	}
	std::string to_string() {return string_from(value);}
};

template<>
class power_param_value<std::string> : public power_param_value_base {
public:
	typedef power_param_value<std::string> this_type;
	power_param_value(std::string v) {
		value = v;
	}
	static this_type *get_default() {
		return new this_type("");
	}
	std::string value;
	std::string to_string() {return value;}
};

/*! Set of named parameters.
 *
 * The core of this class is a map associating a parameter value to
 * each parameter name. This class is meant to be derived to create
 * user-defined sets of parameters, with associated
 * compute_static_power()/compute_dynamic_power() functions.
 */
class power_params {
public:
	typedef std::map<std::string, sc_pwr::power_param_value_base *> map_type;
private:
	map_type m_map;
public:
	/*! Get the value of a parameter, given its name as a string.
	 *
	 * The value is returned as a reference, hence this function
	 * can also be used to change the parameter value.
	 */
	template<typename V>
	V& get_value(std::string name) {
		map_type::iterator i = m_map.find(name);
		if (i == m_map.end()) {
			SC_REPORT_ERROR(PW_BAD_PARAM_TYPE_,
					PW_BAD_PARAM_MSG_);
		}
		power_param_value<V> *val = dynamic_cast<power_param_value<V> *>(i->second);
		if (val == NULL) {
			SC_REPORT_ERROR(PW_BAD_PARAM_TYPE_TYPE_,
					PW_BAD_PARAM_TYPE_MSG_);
		}
		return dynamic_cast<power_param_value<V> *>(m_map[name])->value;
	}

	/*! \brief Create a new parameter.
	 *
	 * The argument is the name of the parameter, and the template
	 * argument is the type of the parameter. This function is
	 * meant to be called in the constructor of classes deriving
	 * from this class.
	 */
	template<typename M>
	void new_param(std::string name) {
		m_map[name] = M::get_default();
	}

	/*! \brief Compute static power consumption.
	 *
	 * This should use the current power parameters (with
	 * get_value()) to return the appropriate value.
	 */
	virtual pw_power compute_static_power() = 0;

	/*! \brief Compute dynamic power consumption.
	 *
	 * This should use the current power parameters (with
	 * get_value()) to return the appropriate value.
	 */
	virtual pw_power compute_dynamic_power() = 0;

	friend std::ostream& operator<<(std::ostream& p_os, const power_params& m);
};

/*! \brief Display an instance of power_params.
 *
 * The display looks like: param1 = ..., param2 = ..., param3 = ...
 */
std::ostream& operator<<(std::ostream& p_os,
                         const power_params& m);

/* Forward declaration needed for template operator */
template <typename M> class pw_param_module;
template<typename M>
std::ostream& operator<<(std::ostream& p_os,
                         const pw_param_module<M>& p_module);

/*! \brief Base class for power-managed modules.
 *
 * The template parameter M should derive from sc_pwr::power_params to
 * define the set of parameters and power consumption computation.
 */
template <typename M>
class pw_param_module : public pw_module_base {
	M m_param;
public:
	/*! Recompute static and dynamic power according to current
 	 * parameters value.
	 */
	void update_power() {
		set_static_power(m_param.compute_static_power());
		set_dynamic_power(m_param.compute_dynamic_power());
	}
	/*! \brief Get the value of a parameter.
	 *
	 * Simple wrapper around power_params::get_value().
	 */
	template<typename V>
	V& get_value(std::string name) {
		return m_param.get_value<V>(name);
	}
	/*! \brief Sets the value of a parameter.
	 *
	 * This does not recompute
	 * the power consumption, one should call update_power()
	 * afterwards.
	 */
	template<typename V>
	void set_value(std::string name, V v) {
		get_value<V>(name) = v;
	}

	/*! \brief Display the module as a string
	 *
	 * (same format as the << operator)
	 */
	const std::string to_string() const {
		return string_from(*this);
	}

	friend std::ostream& operator<< <>(std::ostream& p_os,
					   const pw_param_module<M>& p_module);
};

/*! Display a module with its name, static and dynamic power
 * consumption, and the value of all its power parameters.
 */
template<typename M>
std::ostream& operator<<(std::ostream& p_os,
			 const pw_param_module<M>& p_module) {
	p_os << "{ ";
	try
	{
		const sc_core::sc_object& l_object =
			dynamic_cast< const sc_core::sc_object& >(p_module);
		p_os << l_object.name();
	}
	catch(std::bad_cast)
	{
		SC_REPORT_ERROR(sc_pwr::PW_MODULE_DYNAMIC_CAST_FAILURE_TYPE_,
				sc_pwr::PW_MODULE_DYNAMIC_CAST_FAILURE_MSG_);
	}

	p_os// << ", ( "   << p_module.get_old_dynamic_power()
	     << " -> "   << p_module.get_dynamic_power()
	     << " ), ( " << p_module.get_old_static_power()
	  //<< " -> "   << p_module.get_static_power()
	     << " ), ";
	p_os << p_module.m_param;
	p_os << " }";
	return p_os;
}

} // end namespace sc_pwr

#endif // PW_PARAM_H
