(*

This is essentially the AST of L3 from SPL, 
but without records, tuples, inl, inr, and case. 
Implementing these (and objects) are left as exercises. 

*) 
datatype type_expr =
         TEint
       | TEunit
       | TEbool
       | TEfunc of type_expr * type_expr
       | TEref of type_expr 


type var = Temp.temp 

datatype oper = Plus 
              | Mult 
              | Subt 
              | GTEQ
              | EQ   (* New, not in Slang.1 *) 

datatype unary_oper = Neg | Not  

datatype expr = 
         Skip
       | Integer of int
       | Boolean of bool
       | UnaryOp of unary_oper * expr
       | Op of oper * expr * expr
       | Assign of expr * (type_expr option) * expr (* more general than Slang.1 *) 
       | Deref of expr                              (* more general than Slang.1 *) 
       | Seq of expr * expr
       | If of expr * expr * expr
       | While of expr * expr

       (* new constructors, not in Slang.1 *) 
       | Ref of expr 
       | Var of var
       | Fn of var * type_expr * expr 
       | App of expr * expr                          (* NOTE: single argument *) 
       | Let of var * type_expr * expr * expr        
       | Letrecfn of var * type_expr * var * type_expr * expr * expr


val empty_set = Binaryset.empty String.compare 
val member = Binaryset.member
val singleton = Binaryset.singleton String.compare 
fun delete(s, x) = (Binaryset.delete (s, x)) handle Binaryset.NotFound => s 
val union     = Binaryset.union 
val difference     = Binaryset.difference 
fun list_to_set l = Binaryset.addList(empty_set, l)

(* set of free variables *) 
fun free_vars (Skip)              = empty_set 
  | free_vars (Integer _)         = empty_set 
  | free_vars (Boolean _)         = empty_set 
  | free_vars (UnaryOp (_, e))    = free_vars e
  | free_vars (Op (_, e1, e2))    = union (free_vars e1, free_vars e2)
  | free_vars (Assign(e1, _, e2)) = union (free_vars e1, free_vars e2)
  | free_vars (Deref e)           = free_vars e
  | free_vars (Seq(e1, e2))       = union (free_vars e1, free_vars e2)
  | free_vars (If(e1, e2, e3))    = union (union (free_vars e1, free_vars e2), free_vars e3)
  | free_vars (While(e1, e2))     = union (free_vars e1, free_vars e2)
  | free_vars (Ref e)             = free_vars e
  | free_vars (Var v)             = singleton v 
  | free_vars (App(e1, e2))       = union (free_vars e1, free_vars e2) 
  | free_vars (Fn(x, _, e))       = delete (free_vars e, x)
  | free_vars (Let(x, _, e1, e2)) = union (free_vars e1, delete (free_vars e2, x))
  | free_vars (Letrecfn(f, _, x, _, e1, e2))    
                                  = union (delete(delete(free_vars e1, x), f), delete (free_vars e2, f))

fun type_expr_to_string TEint = "int" 
  | type_expr_to_string TEunit = "unit" 
  | type_expr_to_string TEbool = "bool" 
  | type_expr_to_string (TEfunc (TEfunc (t1, t2), t3)) = 
    "(" ^ (type_expr_to_string (TEfunc (t1, t2))) ^ ") -> " ^ (type_expr_to_string t3) 
  | type_expr_to_string (TEfunc (t1, t2)) = 
    (type_expr_to_string t1) ^ " -> " ^ (type_expr_to_string t1) 
  | type_expr_to_string (TEref t) = "ref(" ^ (type_expr_to_string t) ^ ")"

fun unary_to_string Neg = "-"
  | unary_to_string Not = "~"

fun op_to_string Plus = "+"
  | op_to_string Mult = "*"
  | op_to_string Subt = "-"
  | op_to_string GTEQ = ">="
  | op_to_string EQ   = "="


(* pretty printing using Mosml's PP module *) 

fun formal_to_string (v, t) = v ^ " : " ^ (type_expr_to_string t)

fun ppformal pps p = PP.add_string pps (formal_to_string p) 
                         
fun ppe pps (Integer n) = PP.add_string pps (Int.toString n)
  | ppe pps (Boolean b) = PP.add_string pps (Bool.toString b)
  | ppe pps (UnaryOp (uopr,e)) = 
      (
       PP.add_string pps "("; 
       PP.add_string pps (unary_to_string uopr); 
       ppe pps e;  
       PP.add_string pps ")"
      )
  | ppe pps (Op (opr,e1,e2)) = 
      (
       PP.add_string pps "("; 
       ppe pps e1; 
       PP.add_string pps (" " ^ (op_to_string opr) ^ " "); 
       ppe pps e2;  
       PP.add_string pps ")"
      )
  | ppe pps (If (e1, e2, e3)) = 
      (PP.add_string pps "if "; 
      ppe pps e1; 
      PP.add_break pps (0, 0); 
      PP.add_string pps " then "; 
      ppe pps e2; 
      PP.add_break pps (0, 0); 
      PP.add_string pps " else "; 
      ppe pps e3)
  | ppe pps (Deref e) = (PP.add_string pps "!"; ppe pps e)
  | ppe pps (Assign (e1,NONE, e2)) =  
       (
	ppe pps e1; 
        PP.add_string pps " := "; 
	ppe pps e2
       )
  | ppe pps (Assign (e1,SOME t, e2)) =  
       (
	ppe pps e1; 
        PP.add_string pps " : " ; 
        PP.add_string pps (type_expr_to_string t); 
        PP.add_string pps " := "; 
	ppe pps e2
       )
  | ppe pps (Skip) = PP.add_string pps "skip"	
  | ppe pps (Seq (e1,e2)) =  
       (PP.add_string pps "("; 		 
        ppe pps e1; 
        PP.add_string pps "; "; 
        PP.add_break pps (0, 0); 	
	ppe pps e2;
        PP.add_string pps ")")
  | ppe pps (While (e1,e2)) = 
      (PP.add_string pps "while "; 
       ppe pps e1; 
       PP.add_string pps " do "; 
       PP.add_break pps (0, 0); 
       PP.add_string pps "("; 
       ppe pps e2; 
       PP.add_string pps ")") 
  | ppe pps (Ref e) = (PP.add_string pps "ref("; ppe pps e; PP.add_string pps ")")
  | ppe pps (Var x) = PP.add_string pps x 
  | ppe pps (Fn(x, t, e)) = 
      (
       PP.add_string pps "(fn "; 
       PP.add_string pps (x ^ " : "); 
       PP.add_string pps (type_expr_to_string t); 
       PP.add_string pps " => "; 
       ppe pps e; 
       PP.add_string pps ")"
      ) 
  | ppe pps (App(e1, e2)) = 
      (ppe pps e1; 
       PP.add_string pps "("; 
       ppe pps e2; 
       PP.add_string pps ")")
  | ppe pps (Let(x, t, e1, e2)) = 
      (
       PP.add_string pps ("let val " ^ x ^ " : " ); 
       PP.add_string pps (type_expr_to_string t); 
       PP.add_string pps " = "; 
       ppe pps e1; 
       PP.add_break pps (10, 3); 
       PP.add_string pps " in "; 
       PP.add_break pps (10, 6); 
       ppe pps e2; 
       PP.add_break pps (10, 3); 
       PP.add_string pps " end"
      ) 
  | ppe pps (Letrecfn(f, t1, x, t2, e1, e2)) = 
      (
       PP.add_string pps ("let val rec " ^ f ^ " : "); 
       PP.add_string pps (type_expr_to_string t1); 
       PP.add_string pps " = "; 
       PP.add_break pps (10, 3); 
       PP.add_string pps "(fn "; 
       PP.add_string pps (x ^ " : "); 
       PP.add_string pps (type_expr_to_string t2); 
       PP.add_string pps " => "; 
       ppe pps e1; 
       PP.add_string pps ")"; 
       PP.add_break pps (10, 3); 
       PP.add_string pps " in "; 
       PP.add_break pps (10, 6); 
       ppe pps e2; 
       PP.add_break pps (10, 3); 
       PP.add_string pps " end"
      ) 

fun pp_expr expr = 
    let val ppstrm = PP.mk_ppstream {
                consumer  = fn s => TextIO.output(TextIO.stdOut, s), 
                linewidth = 80,               
                flush     = fn () => TextIO.flushOut TextIO.stdOut}
    in 
        PP.begin_block ppstrm PP.CONSISTENT 0; 
        ppe ppstrm expr ;
        PP.end_block ppstrm; 
        PP.flush_ppstream ppstrm 
    end 

(* this is used for debugging and error messages *) 
val expr_to_string = PP.pp_to_string 100 ppe 

