
type var = AST_expr.var

datatype expr = 
         Skip
       | Integer of int
       | Boolean of bool
       | UnaryOp of AST_expr.unary_oper * expr
       | Op of AST_expr.oper * expr * expr
       | Assign of expr * expr 
       | Deref of expr                              
       | Seq of expr * expr
       | If of expr * expr * expr
       | While of expr * expr
       | Ref of expr 
       | Var of var
       | Let of var * expr * expr        
       | App of expr * (expr list) 
       | Closure of var * (expr list)
       | Direct of var 
       | AppDir of var * (expr list) 
       | AppCls of var * (expr list) 

datatype fundef = FunDecl of 
         var          (* function name *) 
         * bool       (* is it recursive? *) 
         * var        (* formal parameter *) 
         * (var list) (* environment *) 
         * expr             (* body *) 

datatype prog = Prog of (fundef list) * Temp.label 

fun pptl pps [] = () 
  | pptl pps [t] = PP.add_string pps t 
  | pptl pps (t::rest) = (PP.add_string pps t; PP.add_string pps ", ";  pptl pps (t::rest))

fun ppe pps (Integer n) = PP.add_string pps (Int.toString n)
  | ppe pps (Boolean b) = PP.add_string pps (Bool.toString b)
  | ppe pps (UnaryOp (uopr,e)) = 
      (
       PP.add_string pps "("; 
       PP.add_string pps (AST_expr.unary_to_string uopr); 
       ppe pps e;  
       PP.add_string pps ")"
      )
  | ppe pps (Op (opr,e1,e2)) = 
      (
       PP.add_string pps "("; 
       ppe pps e1; 
       PP.add_string pps (" " ^ (AST_expr.op_to_string opr) ^ " "); 
       ppe pps e2;  
       PP.add_string pps ")"
      )
  | ppe pps (If (e1, e2, e3)) = 
      (PP.add_string pps "if "; 
      ppe pps e1; 
      PP.add_break pps (0, 0); 
      PP.add_string pps " then "; 
      ppe pps e2; 
      PP.add_break pps (0, 0); 
      PP.add_string pps " else "; 
      ppe pps e3)
  | ppe pps (Deref e) = (PP.add_string pps "!"; ppe pps e)
  | ppe pps (Assign (e1, e2)) =  
       (
	ppe pps e1; 
        PP.add_string pps " := "; 
	ppe pps e2
       )
  | ppe pps (Skip) = PP.add_string pps "skip"	
  | ppe pps (Seq (e1,e2)) =  
        (PP.add_string pps "("; 		 
        ppe pps e1; 
        PP.add_string pps "; "; 
        PP.add_break pps (0, 0); 	
	ppe pps e2;
        PP.add_string pps ")")
  | ppe pps (While (e1,e2)) = 
       (PP.add_string pps "while "; 
       ppe pps e1; 
       PP.add_string pps " do "; 
       PP.add_break pps (0, 0); 
       PP.add_string pps "("; 
       ppe pps e2; 
       PP.add_string pps ")" )
  | ppe pps (Ref e) = (PP.add_string pps "ref("; ppe pps e; PP.add_string pps ")")
  | ppe pps (Var x) = PP.add_string pps x 
  | ppe pps (AppDir(f, el)) = 
       (PP.add_string pps ("(calldirect " ^ f); 
        PP.add_string pps ", [" ;
        ppel pps el; 
        PP.add_string pps "])")
  | ppe pps (AppCls(f, el)) = 
       (PP.add_string pps ("(callclosure " ^ f); 
       PP.add_string pps ", [";
       ppel pps el; 
       PP.add_string pps "])")
  | ppe pps (App(e, el)) = 
      (PP.add_string pps "(call "; 
       ppe pps e; 
       PP.add_string pps ", [";
       ppel pps el; 
       PP.add_string pps "])")
  | ppe pps (Let(x, e1, e2)) = 
      (
       PP.add_string pps ("let val " ^ x ); 
       PP.add_string pps " = "; 
       ppe pps e1; 
       PP.add_break pps (10, 3); 
       PP.add_string pps "in "; 
       PP.add_break pps (10, 6); 
       ppe pps e2; 
       PP.add_break pps (10, 3); 
       PP.add_string pps "end"
      ) 
  | ppe pps (Closure(l, el)) = 
      (
       PP.add_string pps "CLOSURE("; 
       PP.add_string pps (l ^ ",  ["); 
       ppel pps el; 
       PP.add_string pps "]"; 
       PP.add_string pps ")"
      ) 
  | ppe pps (Direct x) = PP.add_string pps ("DIRECT(" ^ x ^ ")")


and ppel pps [] = () 
  | ppel pps [e] = ppe pps e 
  | ppel pps (e::rest) = 
      (
       ppe pps e;  
       PP.add_string pps ", "; 
       ppel pps rest
      )

fun pp_fun_decl pps (FunDecl(name, is_rec, arg, non_locals, body)) = 
    (
     PP.add_string pps ("function " ^ name); 
     (if is_rec then PP.add_string pps " (recursive) " else PP.add_string pps " (not recursive) ");
     PP.add_string pps ("(" ^ arg ^ ",  ["); 
     pptl pps non_locals; 
     PP.add_string pps "]) = "; 
     PP.add_break pps (10, 3); 
     ppe pps body; 
     PP.add_newline pps  
    ) 

fun pp_fun_list pps [] = () 
  | pp_fun_list pps [fun_decl] = pp_fun_decl pps fun_decl 
  | pp_fun_list pps (fun_decl ::rest) = 
     (
      pp_fun_decl pps fun_decl;  
      pp_fun_list pps rest
     ) 

val ppstrm = PP.mk_ppstream {
                consumer  = fn s => TextIO.output(TextIO.stdOut, s), 
                linewidth = 80,               
                flush     = fn () => TextIO.flushOut TextIO.stdOut}

fun pp_prog (Prog(fun_list, f)) = 
      (
        PP.begin_block ppstrm PP.CONSISTENT 0; 
        pp_fun_list ppstrm fun_list; 
        PP.end_block ppstrm; 
        PP.flush_ppstream ppstrm 
      )  


