
open AST_expr; 

exception TypeError of string 

fun type_error s = raise (TypeError s) 

fun find tenv x = (Library.lookup(tenv, x)) handle _ => type_error (x ^ " is unbound") 


fun match_application_type f ([], []) = [] 
  | match_application_type f ((x, t1)::formals, (t2, e)::te_list) = 
     if t1 = t2 
     then e::(match_application_type f (formals, te_list))
     else type_error (f ^ " expects argument " ^ x ^ " to be of type " ^ (type_expr_to_string t1) ^ " but found type " ^ (type_expr_to_string t2))
  | match_application_type f (_, []) = type_error ("application of " ^ f ^ " has too few arguments") 
  | match_application_type f ([], _) = type_error ("application of " ^ f ^ " has too many arguments") 

fun in_list x [] = false 
  | in_list x (y::rest) = (x = y) orelse (in_list x rest) 

fun has_duplicate_args fl = 
    let fun aux vars [] = false 
          | aux vars ((x,_)::rest) = 
            if in_list x vars then true else aux (x::vars) rest 
    in aux [] fl end 

fun css tenv (Boolean b) = (TEbool, Boolean b)  
  | css tenv (Integer n) = 
    (* 
      In vrm.0, constants must be in range –128 to +127  in order to fit into one (signed) byte. 
      However, the negation symbol "-" is a part of the Slang language, so any "naked" integer 
      constant will be between 0 and 127. 
    *) 
    if  0 <= n andalso n <= 127 
    then (TEint, Integer n)  
    else type_error("vrm.0 target requires all integer constants to be in range 0 to 127") 
  | css tenv (UnaryOp (Neg,Integer n)) = 
    if  0 <= n andalso n <= 128
    then (TEint, UnaryOp (Neg,Integer n))  
    else type_error("vrm.0 target requires all negated integer constants to be in range 0 to 128") 
  | css tenv (UnaryOp (Neg,e)) = 
     let val (t, e') = css tenv e
     in 
        if t = TEint 
        then (t, UnaryOp (Neg,e')) 
        else type_error ("negative applied to expression of type " ^ (type_expr_to_string t))
     end 

  | css tenv (UnaryOp (Not,e)) = 
     let val (t, e') = css tenv e
     in 
        if t = TEbool 
        then (t, UnaryOp (Not,e')) 
        else type_error ("negation applied to expression of type " ^ (type_expr_to_string t))
     end 
  | css tenv (Op (GTEQ,e1,e2)) = 
     let val (t1, e1') = css tenv e1 
         val (t2, e2') = css tenv e2
     in 
        if t1 = TEint 
        then if t2 = TEint 
             then (TEbool, Op (GTEQ, e1',e2'))
             else type_error ("second expression of >= has type " ^ (type_expr_to_string t1))
        else type_error ("first expression of >= has type " ^ (type_expr_to_string t2))
     end 
  | css tenv (Op (EQ,e1,e2)) = 
     let val (t1, e1') = css tenv e1 
         val (t2, e2') = css tenv e2
     in 
        if t1 = TEint 
        then if t2 = TEint 
             then (TEbool, Op (EQ, e1',e2'))
             else type_error ("second expression of = has type " ^ (type_expr_to_string t1))
        else type_error ("first expression of = has type " ^ (type_expr_to_string t2))
     end 
  | css tenv (Op (opr,e1,e2)) =
     let val (t1, e1') = css tenv e1 
         val (t2, e2') = css tenv e2
     in 
        if t1 = TEint 
        then if t2 = TEint 
             then (TEint, Op (opr, e1',e2'))
             else type_error ("second expression of " ^ (op_to_string opr) ^ " has type " ^ (type_expr_to_string t2))
        else type_error ("first expression of " ^ (op_to_string opr) ^ " has type " ^ (type_expr_to_string t1))
     end 
  | css tenv (If (e1,e2,e3)) = 
     let val (t1, e1') = css tenv e1 
         val (t2, e2') = css tenv e2
         val (t3, e3') = css tenv e3
     in 
        if t1 = TEbool 
        then if t2 = t3 
             then (t2, If (e1', e2', e3'))
             else type_error ("then branch of type " ^ (type_expr_to_string t3) ^ " while else branch of type " ^ (type_expr_to_string t3))
        else type_error ("condition of 'if' has type " ^ (type_expr_to_string t1))
     end 
  | css tenv (Deref e) = 
     let val (t, e') = css tenv e
     in 
       case t of 
         TEref t' => (t', Deref e')
       | _ => type_error ("expecting ref type, found " ^ (type_expr_to_string t))
     end 
  | css tenv (Assign (e1, t_opt, e2)) = 
     (* FIX : look at t_opt ?  Parser never puts a type there ... *) 
     let val (t', e1') = css tenv e1
     in 
      case t' of 
        TEref t1 => 
          let val (t2, e2') = css tenv e2 
          in 
             if t1 = t2 
             then (TEunit, Assign (e1', SOME t2, e2'))    (* NOTE: type annotation no longer used ! *) 
            else type_error ("right-hand side of assignment has type " ^ (type_expr_to_string t2))
          end 
      | _  => type_error ("expecting ref type on left-hand-side of assign, found " ^ (type_expr_to_string t'))
     end 
  | css tenv (Skip) = (TEunit, Skip) 
  (* 
     We are going to be very liberal in typing sequence, 
     more so than L3. 
  *) 
  | css tenv (Seq (e1,e2)) = 
     let val (_, e1') = css tenv e1 
         val (t, e2') = css tenv e2
     in 
        (t, Seq(e1', e2'))
     end 
  | css tenv (While (e1,e2)) = 
     let val (t1, e1') = css tenv e1 
         val (t2, e2') = css tenv e2
     in 
        if t1 = TEbool
        then if t2 = TEunit
             then (TEunit, While(e1', e2'))
             else type_error ("body of while has type " ^ (type_expr_to_string t2))
        else type_error ("condition of 'while' has type " ^ (type_expr_to_string t1))
     end 
  | css tenv (Ref e) = 
     let val (t, e') = css tenv e 
     in 
        (TEref t, Ref e')
     end 
   | css tenv (Var x) = (find tenv x, Var x)
   | css tenv (Fn (x, t1, e)) = 
     let val (t2, e') = css (Library.update(tenv, x, t1)) e 
     in 
        (TEfunc(t1,t2), Fn (x, t1, e'))
     end 
   | css tenv (App(Var "print", e)) = 
     let val (t, e') = css tenv e 
     in if t = TEint 
        then (TEunit, App(Var "print", e'))
        else type_error ("argument of print must be of type int, found " ^ (type_expr_to_string t))
     end 
   | css tenv (App(e1, e2)) = 
       let val (fun_t, e1') = css tenv e1 
           val (arg_t, e2') = css tenv e2 
       in 
	   case fun_t of 
             TEfunc(t1, t2) => 
               if t1 = arg_t 
               then (t2, App(e1', e2'))
               else type_error ("mismatch in application types. Function has type " ^ (type_expr_to_string fun_t) ^ ", while argument has type " ^ (type_expr_to_string arg_t))
           | _ => type_error ("expecting function type, found " ^ (type_expr_to_string fun_t))
       end 
   | css tenv (Let(x, t, e1, e2)) = 
       let val (t1, e1') = css tenv e1 
           val (t2, e2') = css (Library.update(tenv, x, t1)) e2
       in 
           if t = t1 
           then (t2, Let(x, t, e1', e2')) 
           else type_error (x ^ " has declared type " ^ (type_expr_to_string t) ^ " while the associated expression has type " ^ (type_expr_to_string t1))
       end          
   | css tenv (Letrecfn(f, TEfunc(t1,t2), x, arg_t, e1, e2)) = 
       if t1 = arg_t 
       then  let val tenv' = Library.update(tenv, f, TEfunc(t1, t2))
                 val (body_t, e1') = css (Library.update(tenv', x, t1)) e1 
                 val (result_t, e2') = css tenv' e2
             in 
                 if t2 = body_t 
                 then (result_t, Letrecfn(f, TEfunc(t1,t2), x, arg_t, e1', e2'))
                 else type_error (f ^ " has declared type " ^ (type_expr_to_string (TEfunc(t1,t2)))  ^ ", which does not match the type of the function, " ^ (type_expr_to_string (TEfunc(t1, body_t))))

             end 
       else type_error (f ^ " has declared type " ^ (type_expr_to_string (TEfunc(t1,t2))) ^ ", which does not match the type of " ^ x ^ ", which is " ^ (type_expr_to_string arg_t))
   | css tenv (Letrecfn(f, t, _, _, _, _)) = 
       type_error (f ^ " has declared type " ^ (type_expr_to_string t) ^ ", which is not a function type")




fun check_static_semantics e = 
    let val tenv = Library.empty_env 
        val (_, e') = css tenv e 
    in e' end 
    
   
