open AST_vsm_assembler; 

(* structure T = Tree *) 

fun not_yet s = Library.internal_error ("vsm_code_gen has not yet implmented " ^ s ) 

val new_offset = Library.new_counter(); 
val next_label = Library.new_counter(); 
fun new_label () = "_l" ^ (Int.toString (next_label ())) 

fun vsm_label_sequence [] = (new_label(), []) 
  | vsm_label_sequence ((VSM_Code oper)        :: rest        ) = 
    let val l = new_label() 
    in 
       (l, (VSM_Labelled(l, oper)) :: rest) 
    end 
  | vsm_label_sequence ((VSM_Labelled(l, oper)) :: rest) = 
       (l, (VSM_Labelled(l, oper)) :: rest) 

fun vsm_insert_remark_in_op new (VSM_Hlt remark)         = VSM_Hlt (new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Nop remark)         = VSM_Nop (new ^ " " ^ remark)                  
  | vsm_insert_remark_in_op new (VSM_Jmp (cl, remark))   = VSM_Jmp (cl, new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Push (c, remark))   = VSM_Push (c, new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Store (dl, remark)) = VSM_Store (dl, new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_StoreEnv (off1, off2, remark)) = VSM_StoreEnv (off1, off2, new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Load (dl, remark))  = VSM_Load (dl, new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_LoadEnv (off1, off2 , remark))  = VSM_LoadEnv (off1, off2, new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Pop remark)         = VSM_Pop (new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Add remark)         = VSM_Add (new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Sub remark)         = VSM_Sub (new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Mul remark)         = VSM_Mul (new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Ifz (cl, remark))   = VSM_Ifz (cl, new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Ifp (cl, remark))   = VSM_Ifp (cl, new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Ifn (cl, remark))   = VSM_Ifn (cl, new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Pri remark)         = VSM_Pri (new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Prb remark)         = VSM_Prb (new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_ReturnClosure remark) = VSM_ReturnClosure (new ^  " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_ReturnDirect remark)  = VSM_ReturnDirect (new ^  " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_CallClosure remark)   = VSM_CallClosure (new ^  " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_CallDirect remark)    = VSM_CallDirect (new ^  " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Call remark)          = VSM_Call (new ^  " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_HeapAllocateHeap (n, remark))          = VSM_HeapAllocateHeap (n, new ^  " " ^ remark)

fun vsm_insert_remark remark [] = [] 
  | vsm_insert_remark remark ((VSM_Code oper) :: rest) = (VSM_Code (vsm_insert_remark_in_op remark oper)) :: rest
  | vsm_insert_remark remark ((VSM_Labelled(l, oper)) :: rest) = ((VSM_Labelled(l, vsm_insert_remark_in_op remark oper)) :: rest) 

fun find a [] = NONE 
  | find a ((b, v) :: rest) = if a = b then SOME v else find a rest 

fun fenv_find f fenv = SOME(Library.lookup(fenv, f)) handle _ => NONE 


fun push_zero 0 = [] 
  | push_zero n = (VSM_Code(VSM_Push (SVconstant 0, ""))) :: (push_zero (n-1))

fun list_index t n [] = Library.internal_error ("list_index: did not find " ^ t)
  | list_index t n (v::rest) = if v = t then n else list_index t (n + 1) rest

fun member ([], _) = false 
  | member (a::rest, b) = if a = b then true else member (rest, b) 

fun find_temp_value (name, arg, non_locals, locals) t = 
    if t = arg 
    then VSM_Load(0 - 1, "load function argument " ^ t)
    else if member(non_locals, t) 
         then VSM_LoadEnv(0 - 2,  list_index t 1 non_locals, "load non-local " ^ t) 
         else if member(locals, t) 
              then VSM_Load(1 + (list_index t 1 locals), "load local " ^ t) 
              else Library.internal_error ("vsm_code_gen: unbound variable " ^ t)

val vsm_unit_value = 0 
val vsm_true_value = 1 
val vsm_false_value = 0


fun visit_vsm_data_loc env d = 
    case find d env 
    of NONE => (d, new_offset()) :: env 
     | SOME _ => env 

fun stm_to_vsm_code_list fenv env (Tree.SEQ(s1, s2)) = (stm_to_vsm_code_list fenv env s1) @ (stm_to_vsm_code_list fenv env s2)
  | stm_to_vsm_code_list fenv env (Tree.LABEL l) = [VSM_Labelled(l, VSM_Nop "")]
  | stm_to_vsm_code_list fenv env (Tree.MOVE(Tree.TEMP t, e)) = 
       let val cl = expr_to_vsm_code_list fenv env e
           val (name, arg, non_locals, locals)  = env 
           val code = 
                  if member(locals, t) 
                  then VSM_Store (2 + (list_index t 1 locals) , "store " ^ t ^ " in frame")
                  else if member(non_locals, t) 
                       then VSM_StoreEnv(0-2, 1 + (list_index t 1 locals) , "store " ^ t ^ " in frame")
                       else Library.internal_error ("stm_to_vsm_code : temp " ^ t ^ " is not local nor non_local")
       in 
          cl @ [VSM_Code code] 
       end 
  | stm_to_vsm_code_list fenv env (Tree.MOVE(Tree.MEM e1, e2)) = not_yet "MOVE-MEM"
  | stm_to_vsm_code_list fenv env (Tree.MOVE(_, _)) = Library.internal_error "stm_to_vsm_code_list fenv env: malformed MOVE" 
  | stm_to_vsm_code_list fenv env (Tree.EXP e) = expr_to_vsm_code_list fenv env e 
  | stm_to_vsm_code_list fenv env (Tree.JUMP (Tree.NAME l, ll)) = [VSM_Code(VSM_Jmp (l, ""))]
  | stm_to_vsm_code_list fenv env (Tree.JUMP (e, _)) = not_yet "JUMP"

  | stm_to_vsm_code_list fenv env (Tree.CJUMP(Tree.NE, e1, e2, true_label, false_label)) = not_yet "NE"
  | stm_to_vsm_code_list fenv env (Tree.CJUMP(Tree.LT, e1, e2, true_label, false_label)) = not_yet "LT"
  | stm_to_vsm_code_list fenv env (Tree.CJUMP(Tree.GT, e1, e2, true_label, false_label)) = not_yet "GT"
  | stm_to_vsm_code_list fenv env (Tree.CJUMP(Tree.LE, e1, e2, true_label, false_label)) = not_yet "LE"
  | stm_to_vsm_code_list fenv env (Tree.CJUMP(Tree.EQ, e1, e2, true_label, false_label)) = 
       let val cl1 = expr_to_vsm_code_list fenv env e1
           val cl2 = expr_to_vsm_code_list fenv env e2
           val bypass_label = new_label() 
      in  
          vsm_insert_remark "start EQ ... " (
          cl1
          @ cl2 
          @ [VSM_Code (VSM_Sub ""),       (* "e1 - e2 now on top of stack *) 
             VSM_Code(VSM_Ifz (bypass_label, "")), 
             VSM_Code(VSM_Push (SVconstant vsm_false_value, "push false")), 
             VSM_Code(VSM_Jmp (false_label, "")), 
             VSM_Labelled(bypass_label, VSM_Push (SVconstant vsm_true_value, "push true")), 
             VSM_Code(VSM_Jmp (true_label, " ... end EQ"))
            ])
      end 

  | stm_to_vsm_code_list fenv env (Tree.CJUMP(Tree.GE, e1, e2, true_label, false_label)) =
       let val cl1 = expr_to_vsm_code_list fenv env e1
           val cl2 = expr_to_vsm_code_list fenv env e2
           val bypass_label = new_label() 
      in  
          vsm_insert_remark "start >= ... " (
          cl1
          @ cl2 
          @ [VSM_Code (VSM_Sub ""),       (* "e1 - e2 now on top of stack *) 
             VSM_Code(VSM_Ifn (bypass_label, "")), 
             VSM_Code(VSM_Push (SVconstant vsm_true_value, "push true")), 
             VSM_Code(VSM_Jmp (true_label, "")), 
             VSM_Labelled(bypass_label, VSM_Push (SVconstant vsm_false_value, "push false")), 
             VSM_Code(VSM_Jmp (false_label, " ... end >="))
            ])
      end 
  | stm_to_vsm_code_list fenv env (Tree.CJUMP(Tree.ULT, e1, e2, true_label, false_label)) = not_yet "ULT"
  | stm_to_vsm_code_list fenv env (Tree.CJUMP(Tree.ULE, e1, e2, true_label, false_label)) = not_yet "ULE"
  | stm_to_vsm_code_list fenv env (Tree.CJUMP(Tree.UGT, e1, e2, true_label, false_label)) = not_yet "UGT"
  | stm_to_vsm_code_list fenv env (Tree.CJUMP(Tree.UGE, e1, e2, true_label, false_label)) = not_yet "UGE"

and expr_to_vsm_code_list fenv env (Tree.BINOP(Tree.PLUS, e1, e2)) = 
       let val cl1 = expr_to_vsm_code_list fenv env e1
           val cl2 = expr_to_vsm_code_list fenv env e2
       in 
          cl1 @ cl2 @ [VSM_Code (VSM_Add "")]
       end 
  (* note that "sub" semantics is "under-top-of-stack - top-of-stack", so this works! *) 
  | expr_to_vsm_code_list fenv env (Tree.BINOP(Tree.MINUS, e1, e2)) = 
       let val cl1 = expr_to_vsm_code_list fenv env e1
           val cl2 = expr_to_vsm_code_list fenv env e2
       in 
          cl1 @ cl2 @ [VSM_Code (VSM_Sub "")]
       end 
  | expr_to_vsm_code_list fenv env (Tree.BINOP(Tree.MUL, e1, e2)) = 
       let val cl1 = expr_to_vsm_code_list fenv env e1
           val cl2 = expr_to_vsm_code_list fenv env e2
       in 
          cl1 @ cl2 @ [VSM_Code (VSM_Mul "")]
       end 
  | expr_to_vsm_code_list fenv env (Tree.BINOP(Tree.DIV, e1, e2)) = not_yet "DIV"
  | expr_to_vsm_code_list fenv env (Tree.BINOP(Tree.AND, e1, e2)) = not_yet "AND"
  | expr_to_vsm_code_list fenv env (Tree.BINOP(Tree.OR, e1, e2)) = not_yet "OR"
  | expr_to_vsm_code_list fenv env (Tree.BINOP(Tree.LSHIFT, e1, e2)) = not_yet "LSHIFT"
  | expr_to_vsm_code_list fenv env (Tree.BINOP(Tree.RSHIFT, e1, e2)) = not_yet "RSHIFT"
  | expr_to_vsm_code_list fenv env (Tree.BINOP(Tree.ARSHIFT, e1, e2)) = not_yet "ARSHIFT"
  | expr_to_vsm_code_list fenv env (Tree.BINOP(Tree.XOR, e1, e2)) =  not_yet "XOR"
  | expr_to_vsm_code_list fenv env (Tree.MEM e) = not_yet "MEM"

  | expr_to_vsm_code_list fenv env (Tree.ESEQ(stm, e)) = (stm_to_vsm_code_list fenv env stm) @ (expr_to_vsm_code_list fenv env e )
  | expr_to_vsm_code_list fenv env (Tree.CONST n) = [VSM_Code(VSM_Push (SVconstant  n, ""))]

  | expr_to_vsm_code_list fenv env (Tree.TEMP t) = 
    (case  fenv_find t fenv of 
        NONE => [VSM_Code(find_temp_value env t)] 
      | SOME is_closure => if is_closure then [VSM_Code (VSM_Push(SVclosure t, ""))] else [VSM_Code (VSM_Push(SVfunction t, ""))])
  | expr_to_vsm_code_list fenv env (Tree.NAME l) = [VSM_Labelled(l, VSM_Nop "")]
  | expr_to_vsm_code_list fenv env (Tree.CALL (Tree.NAME "print", [e])) = 
      (expr_to_vsm_code_list fenv env e) @ [VSM_Code (VSM_Pri "")]
  | expr_to_vsm_code_list fenv env (Tree.CALL (Tree.NAME f, [e])) = 
     (case  fenv_find f fenv of 
        NONE => Library.internal_error ("vsm_code_gen : function not defined : " ^ f)
      | SOME is_closure => 
        if is_closure 
       then [VSM_Code (VSM_Push(SVclosure f, ""))] @ (expr_to_vsm_code_list fenv env e) @ [VSM_Code (VSM_Call ("call function " ^ f ))]
       else [VSM_Code (VSM_Push(SVfunction f, ""))] @ (expr_to_vsm_code_list fenv env e) @ [VSM_Code (VSM_Call ("call function " ^ f ))])
  | expr_to_vsm_code_list fenv env (Tree.CALL (e, el)) = 
      (expr_to_vsm_code_list fenv env e) @ (List.concat (List.map (expr_to_vsm_code_list fenv env) el)) @ [VSM_Code (VSM_Call "")]
  | expr_to_vsm_code_list fenv env (Tree.CLOSURE (f, el)) = 
    (List.concat (List.map (expr_to_vsm_code_list fenv env) el)) 
    @ [VSM_Code(VSM_Push(SVfunction f, "function of closure")), VSM_Code(VSM_HeapAllocateHeap(1 + (List.length el), "allocate closure for " ^ f))] 

fun vsm_fun_decl fenv (Tree.FunDecl(name, arg, non_locals, locals, stm)) = 
    let val env = (name, arg, non_locals, locals) 
        val cl = stm_to_vsm_code_list fenv env stm 
        val allocate_locals = push_zero (List.length locals) 
        val return = if [] = non_locals then VSM_ReturnDirect "" else VSM_ReturnClosure ""
    in 
        (VSM_Labelled (name, VSM_Nop "")) :: 
         (allocate_locals @ cl @ [VSM_Code return])
    end 

fun mk_fenv fenv [] = fenv 
  | mk_fenv fenv ((Tree.FunDecl(f, _, [], _ , _)):: rest) = mk_fenv (Library.update(fenv,f,false)) rest 
  | mk_fenv fenv ((Tree.FunDecl(f, _, _ , _ , _)):: rest) = mk_fenv (Library.update(fenv,f,true)) rest 

fun vsm_code_gen (Tree.Prog(fun_list, f)) = 
    let val init_code = 
         [VSM_Code (VSM_Push (SVfunction f, "")),
          VSM_Code (VSM_Call "call main "),
          VSM_Code (VSM_Hlt " that's all folks!")] 
        val fenv = mk_fenv Library.empty_env fun_list 
    in 
        init_code @ List.concat (List.map (vsm_fun_decl fenv) fun_list)
    end  

(* Our peep-hole optimization eliminates 

    push c
    pop 

  and simplifies 

    l : nop 
        inst 

  to 

    l : inst 

  We could attempt to preserve comments, but currently do not ...  

*) 
fun vsm_peep_hole ((VSM_Code(VSM_Push _)) :: ((VSM_Code(VSM_Pop _)) :: rest)) = 
       vsm_peep_hole rest 
  | vsm_peep_hole ((VSM_Labelled(l, VSM_Nop _)) :: ((VSM_Code(c)) :: rest)) = 
       (VSM_Labelled(l, c)) :: (vsm_peep_hole rest) 
  | vsm_peep_hole (c :: rest) = c :: (vsm_peep_hole rest)
  | vsm_peep_hole [] = [] 

