(* See comments in Tree.sig *) 


  type label=Temp.label

  datatype stm = SEQ of stm * stm
               | LABEL of label
               | JUMP of exp * label list
               | CJUMP of relop * exp * exp * label * label
               | MOVE of exp * exp
               | EXP of exp

       and exp = BINOP of binop * exp * exp
               | MEM of exp
               | TEMP of Temp.temp
               | ESEQ of stm * exp
               | NAME of label
               | CONST of int
               | CALL of exp * (exp list)
               | CLOSURE of Temp.label * (exp list)

       and binop = PLUS | MINUS | MUL | DIV 
                 | AND | OR | LSHIFT | RSHIFT | ARSHIFT | XOR

       and relop = EQ | NE | LT | GT | LE | GE 
                 | ULT | ULE | UGT | UGE


  datatype fundef = FunDecl of 
         Temp.label         (* name *) 
         * Temp.temp        (* formal parameter *) 
         * (Temp.temp list) (* environment *) 
         * (Temp.temp list) (* local variables *) 
         * stm              (* body *) 

  datatype prog = Prog of (fundef list) * Temp.label


datatype tree_rep  = Ex of exp
                   | Nx of stm
                   | Cx of Temp.label * Temp.label -> stm


(* this function and one and zero are duplicated in Translate.sml *) 
fun seq([]) = Library.internal_error "seq : given empty list!"
  | seq([s]) = s
  | seq(h::t) = SEQ(h, seq(t))

val zero = CONST 0
val one = CONST 1

  fun unEx(Ex e) = e
    | unEx(Cx mk_stm) =
        let
          val result      = TEMP (Temp.newtemp())
          val true_label  = Temp.newlabel() 
          val false_label = Temp.newlabel()
        in
            ESEQ(seq[MOVE(result, one),
                       mk_stm(true_label ,false_label),
                       LABEL false_label,
                       MOVE(result, zero),
                       LABEL true_label],
                   result)
        end
    | unEx(Nx s) = ESEQ(s, zero)
  
  fun unNx(Ex e) = EXP(e)
    | unNx(Cx mk_stm) = 
      let val join = Temp.newlabel()
      in
          SEQ(mk_stm(join, join), LABEL join)
      end
    | unNx(Nx s) = s
    
  fun unCx(Ex(CONST 0)) = (fn (t, f) => JUMP(NAME f, [f]))
    | unCx(Ex(CONST _)) = (fn (t, f) => JUMP(NAME t, [t]))
    | unCx(Ex e) = (fn (t, f) => CJUMP(NE, e, zero, t, f))
    | unCx(Cx f) = f
    | unCx(Nx _) = Library.internal_error "unCx: given an Nx!"


fun binop_to_string PLUS = "PLUS" 
  | binop_to_string MINUS = "MINUS" 
  | binop_to_string MUL = "MUL" 
  | binop_to_string DIV = "DIV" 
  | binop_to_string AND = "AND" 
  | binop_to_string OR = "OR" 
  | binop_to_string LSHIFT = "LSHIFT" 
  | binop_to_string RSHIFT = "RSHIFT" 
  | binop_to_string ARSHIFT = "ARSHIFT" 
  | binop_to_string XOR = "XOR" 

fun relop_to_string EQ = "EQ"
  | relop_to_string NE = "NE"
  | relop_to_string LT = "LT"
  | relop_to_string GT = "GT"
  | relop_to_string LE = "LE"
  | relop_to_string GE = "GE"
  | relop_to_string ULT = "ULT"
  | relop_to_string ULE = "ULE"
  | relop_to_string UGT = "UGT"
  | relop_to_string UGE = "UGE"
      

fun flatten_seq(SEQ (s1, s2)) = (flatten_seq s1) @ (flatten_seq s2)
  | flatten_seq e             = [e] 

fun ppm pps (SEQ (s1, s2)) = 
      (
       PP.add_string pps "("; 
       ppml pps ((flatten_seq s1) @ (flatten_seq s2)); 
       PP.add_string pps ")"
      )
  | ppm pps (LABEL l) = PP.add_string pps ("LABEL " ^ l )
  | ppm pps (JUMP (e, ll)) = 
      (
       PP.add_string pps "JUMP("; 
       ppe pps e;  
       PP.add_string pps ")"
      )
  | ppm pps (CJUMP(ro, e1, e2, l1, l2)) = 
      (
       PP.add_string pps "CJUMP("; 
       PP.add_string pps (relop_to_string ro); 
       PP.add_string pps ", "; 
       ppe pps e1;  
       PP.add_string pps ", "; 
       ppe pps e2;  
       PP.add_string pps ", "; 
       PP.add_string pps l1; 
       PP.add_string pps ", "; 
       PP.add_string pps l2; 
       PP.add_string pps ")"
      )
  | ppm pps (MOVE(e1, e2)) = 
      (
       PP.add_string pps "MOVE("; 
       ppe pps e1;  
       PP.add_string pps ", "; 
       ppe pps e2;  
       PP.add_string pps ")"
      )
  | ppm pps (EXP e) = 
      (
       PP.add_string pps "EXP("; 
       ppe pps e;  
       PP.add_string pps ")"
      )

and ppe pps (BINOP(bop, e1, e2)) = 
      (
       PP.add_string pps "("; 
       PP.add_string pps (binop_to_string bop); 
       PP.add_string pps ","; 
       ppe pps e1;  
       PP.add_string pps ","; 
       ppe pps e2;  
       PP.add_string pps ")"
      )
  | ppe pps (MEM e) = 
      (
       PP.add_string pps "MEM("; 
       ppe pps e;  
       PP.add_string pps ")"
      )
  | ppe pps (TEMP t) = PP.add_string pps ("TEMP " ^ t)
  | ppe pps (ESEQ (s, e)) = 
      (
       PP.add_string pps "ESEQ("; 
       ppml pps (flatten_seq s); 
       PP.add_string pps ","; 
       PP.add_break pps (10, 3); 
       ppe pps e;  
       PP.add_string pps ")"
      )
  | ppe pps (NAME l) = PP.add_string pps ("NAME " ^ l)
  | ppe pps (CONST n) = PP.add_string pps (Int.toString n)
  | ppe pps (CALL (e, el)) = 
      (
       PP.add_string pps "CALL("; 
       ppe pps e;  
       PP.add_string pps ", [" ; 
       ppel pps el ;  
       PP.add_string pps "])"
      )
  | ppe pps (CLOSURE (l, el)) = 
      (
       PP.add_string pps "CLOSURE("; 
       PP.add_string pps l; 
       PP.add_string pps ", [" ; 
       ppel pps el ;  
       PP.add_string pps "])"
      )

and ppel pps [e] = ppe pps e 
  | ppel pps (e::rest) = 
      (
       ppe pps e;  
       PP.add_string pps ", "; 
       ppel pps rest
      )
  | ppel pps [] = () 

and ppml pps [s] = (ppm pps s)
  | ppml pps (s::rest) = 
      (
       ppm pps s;  
       PP.add_string pps "; "; 
       PP.add_newline pps; 
       ppml pps rest
      )
  | ppml pps [] = () 

and pptl pps [t] = PP.add_string pps t 
  | pptl pps (t::rest) = 
      (
       PP.add_string pps t; 
       PP.add_string pps ", "; 
       pptl pps rest
      )
  | pptl pps [] = () 


fun pptr pps (Ex e) = 
      (
       PP.add_string pps "Ex("; 
       ppe pps e;  
       PP.add_string pps ")"
      )
  | pptr pps (Nx s) = 
      (
       PP.add_string pps "Nx("; 
       ppm pps s;  
       PP.add_string pps ")"
      )
  | pptr pps (Cx f) = 
    let val l1 = Temp.newlabel() 
        val l2 = Temp.newlabel() 
    in 
      (
       PP.add_string pps "Cx(fn "; 
       PP.add_string pps ("(" ^ l1 ^ ", " ^ l2 ^ ") => "); 
       ppm pps (f (l1, l2) );  
       PP.add_string pps ")"
      )
    end 

val ppstrm = PP.mk_ppstream {
       consumer  = fn s => TextIO.output(TextIO.stdOut, s), 
       linewidth = 80,               
       flush     = fn () => TextIO.flushOut TextIO.stdOut
    }

fun pp_exp exp = 
        (PP.begin_block ppstrm PP.CONSISTENT 0; 
        ppe ppstrm exp ;
        PP.end_block ppstrm; 
        PP.flush_ppstream ppstrm )
 

fun pp_stm stm = 
        (PP.begin_block ppstrm PP.CONSISTENT 0; 
        ppm ppstrm stm ;
        PP.end_block ppstrm; 
        PP.flush_ppstream ppstrm)

fun pp_tree_rep tr = 
        (PP.begin_block ppstrm PP.CONSISTENT 0; 
        pptr ppstrm tr ;
        PP.end_block ppstrm; 
        PP.flush_ppstream ppstrm )

fun pp_fun_decl pps (FunDecl(name, arg, non_locals, locals, body)) = 
    (
     PP.add_string pps ("function " ^ name ^ " ("); 
     PP.add_string pps (arg ^ ",  ["); 
     pptl pps non_locals; 
     PP.add_string pps "]) = "; 
       PP.add_break pps (10, 3); 
     PP.add_string pps ("locals : ["); 
     pptl pps locals; 
     PP.add_string pps "]"; 
       PP.add_break pps (10, 3); 
     ppm pps body 
    ) 


fun pp_fun_list pps [] = () 
  | pp_fun_list pps [fun_decl] = 
    (pp_fun_decl pps fun_decl; PP.add_newline pps) 
  | pp_fun_list pps (fun_decl ::rest) = 
     (
      pp_fun_decl pps fun_decl;  
      PP.add_newline pps ; 
      pp_fun_list pps rest
      ) 

fun pp_prog (Prog(fun_list, f)) = 
      (
        PP.begin_block ppstrm PP.CONSISTENT 0; 
        pp_fun_list ppstrm fun_list; 
        PP.end_block ppstrm; 
        PP.flush_ppstream ppstrm 
      )  

