
type data_loc = int 
type code_loc = string
type constant = int 
type vsm_comment = string (* for instructional purposes! *)

datatype vsm_operation = 
       (* data operations *) 
         VSM_Nop of vsm_comment 
       | VSM_Push of constant * vsm_comment 
       | VSM_Load of data_loc * vsm_comment 
       | VSM_Store of data_loc * vsm_comment 
       | VSM_Pop of vsm_comment 
       | VSM_Add of vsm_comment 
       | VSM_Sub of vsm_comment 
       | VSM_Mul of vsm_comment 
       (* control flow operations *) 
       | VSM_Hlt of vsm_comment 
       | VSM_Jmp of code_loc * vsm_comment 
       | VSM_Ifz of code_loc * vsm_comment 
       | VSM_Ifp of code_loc * vsm_comment 
       | VSM_Ifn of code_loc * vsm_comment 
       (* input/output *) 
       | VSM_Pri of vsm_comment 
       | VSM_Prb of vsm_comment 

datatype vsm_code = 
         VSM_Code of vsm_operation 
       | VSM_Labelled of code_loc * vsm_operation 

type vsm_program = vsm_code list 

fun vsm_operation_to_strings (VSM_Hlt remark)         = ("hlt",                      remark) 
  | vsm_operation_to_strings (VSM_Nop remark)         = ("nop",                      remark) 
  | vsm_operation_to_strings (VSM_Add remark)         = ("add",                      remark)  
  | vsm_operation_to_strings (VSM_Sub remark)         = ("sub",                      remark)  
  | vsm_operation_to_strings (VSM_Mul remark)         = ("mul",                      remark)  
  | vsm_operation_to_strings (VSM_Pop remark)         = ("pop",                      remark)  
  | vsm_operation_to_strings (VSM_Pri remark)         = ("pri",                      remark)  
  | vsm_operation_to_strings (VSM_Prb remark)         = ("prb",                      remark)  
  | vsm_operation_to_strings (VSM_Jmp (cl, remark))   = ("jmp " ^ cl,                remark)  
  | vsm_operation_to_strings (VSM_Ifz (cl, remark))   = ("ifz " ^ cl,                remark) 
  | vsm_operation_to_strings (VSM_Ifp (cl, remark))   = ("ifp " ^ cl,                remark) 
  | vsm_operation_to_strings (VSM_Ifn (cl, remark))   = ("ifn " ^ cl,                remark) 
  | vsm_operation_to_strings (VSM_Load (dl, remark))  = ("load " ^ (Int.toString dl),remark) 
  | vsm_operation_to_strings (VSM_Store (dl, remark)) = ("store "^ (Int.toString dl),remark) 
  | vsm_operation_to_strings (VSM_Push (c, remark))   = ("push " ^ (Int.toString c), remark) 


fun vsm_code_to_strings (VSM_Code oper) = ("", vsm_operation_to_strings oper) 
  | vsm_code_to_strings (VSM_Labelled (l,oper)) = (l ^ " : ", vsm_operation_to_strings oper)

fun vsm_program_to_strings [] = [] 
  | vsm_program_to_strings (c::rest) = (vsm_code_to_strings c) :: (vsm_program_to_strings rest)

fun vsm_inspect_program_strings ll lo [] = (ll, lo)
  | vsm_inspect_program_strings ll lo ((label, (oper, _))::rest) = 
       vsm_inspect_program_strings (Int.max(ll, String.size label)) (Int.max(lo, String.size oper)) rest 

fun vsm_program_with_nice_indentation ll lo [] = ""
  | vsm_program_with_nice_indentation ll lo ((label, (oper, remark)):: rest) = 
    (label ^ (Library.n_spaces (ll - (String.size label))) ^ oper ^ (Library.n_spaces (lo - (String.size oper))) ^ " %" ^ remark ^ "\n") 
    ^ (vsm_program_with_nice_indentation ll lo rest) 

fun vsm_program_to_string prog = 
    let val strs = vsm_program_to_strings prog 
        val (longest_label, longest_operation) = vsm_inspect_program_strings 0 0 strs 
    in 
        vsm_program_with_nice_indentation longest_label longest_operation strs 
    end 

