open AST_expr; 
open AST_normal_expr; 

val next = Library.new_counter(); 

fun new_location () = "_X" ^ (Int.toString (next ())) 

(* can_update(l, e) is true when e can possibly update the location l *) 
fun can_update (l, UnaryOp (_, e))   = can_update(l, e) 
  | can_update (l, Op (_,e1,e2))     = (can_update(l, e1)) orelse (can_update(l, e2)) 
  | can_update (l, If (e1, e2, e3))  = (can_update(l, e1)) orelse (can_update(l, e2)) orelse (can_update(l, e3)) 
  | can_update (l, Assign (l',_, e)) = (l = l') orelse (can_update(l, e)) 
  | can_update (l, Seq (e1,e2))      = (can_update(l, e1)) orelse (can_update(l, e2)) 
  | can_update (l, While (e1,e2))    = (can_update(l, e1)) orelse (can_update(l, e2)) 
  | can_update (l, Print (_, e))     = can_update(l, e) 
  | can_update _                     = false 

fun locs_to_init_code [] = [] 
  | locs_to_init_code (l ::rest) = (Normal_SetInteger(l, 0)) :: (locs_to_init_code rest) 

fun add_loc a [] = [a] 
  | add_loc a (b :: rest) = if a = b then (b::rest) else b::(add_loc a rest) 

fun all_locs locs (UnaryOp (_, e))   = all_locs locs e
  | all_locs locs (Op (_,e1,e2))     = all_locs (all_locs locs e1) e2
  | all_locs locs (If (e1, e2, e3))  = all_locs (all_locs (all_locs locs e1) e2) e3
  | all_locs locs (Assign (l,_, e))  = add_loc l (all_locs locs e)
  | all_locs locs (Deref l)          = add_loc l locs 
  | all_locs locs (Seq (e1,e2))      = all_locs (all_locs locs e1) e2
  | all_locs locs (While (e1,e2))    = all_locs (all_locs locs e1) e2
  | all_locs locs (Print (_, e))     = all_locs locs e
  | all_locs locs  _                 = locs 

(*
    normalise_expr : expr -> (normal_expr list) * loc
    
    If "normalise_expr e" results in (nel, l), then the 
    execution of "Normal_Seq nel" will leave the value of e in location l. 
    Can you prove it correct (or find a bug) using the methods of SPL? 
    Requires a formalization of the semantics of normal_expr, which is 
    here only informal. 
*) 
fun normalise_expr Skip = ([], "_Unit")  (* is this correct? Discuss! *) 
  | normalise_expr (Integer n) = 
    let val l = new_location() 
    in 
       ([Normal_SetInteger(l, n)], l) 
    end 
  | normalise_expr (Boolean b) = 
    let val l = new_location() 
    in 
       ([Normal_SetBoolean(l, b)], l) 
    end 
  | normalise_expr (UnaryOp (uop, e)) = 
    let val (el, l) = normalise_expr e 
        and l' = new_location() 
    in 
        (el @ [Normal_UnaryOp(uop, l', l)], l') 
    end 
  | normalise_expr (Op (bop, e1, e2)) = 
    let val (el1, l1) = normalise_expr e1 
        and (el2, l2) = normalise_expr e2 
        and l3 = new_location() 
    in 
       (* be careful! running el2 could change the value "saved" in location l1 *) 
       if can_update(l1, e2) 
       then let val l4 = new_location() 
            in 
               (el1 @ [Normal_Assign(l4, l1)] @ el2 @ [Normal_Op(bop, l3, l4, l2)], l3)		
            end 
       else (el1 @ el2 @ [Normal_Op(bop, l3, l1, l2)], l3)  
    end 
  | normalise_expr (Assign (l, _, Integer n)) =  ([Normal_SetInteger(l, n)], "_Unit") 
  | normalise_expr (Assign (l, _, Boolean b)) = ([Normal_SetBoolean(l, b)], "_Unit") 
  | normalise_expr (Assign (l, _, e)) = 
    let val (el, l') = normalise_expr e 
    in 
       (el @ [Normal_Assign(l, l')], "_Unit")  
    end 
  | normalise_expr (Deref l) = ([], l)   (* Is this correct?  Discuss! *) 
  | normalise_expr (Seq (e1, e2)) = 
    let val (el1, _) = normalise_expr e1 
        and (el2, l) = normalise_expr e2 
    in 
        (el1 @ el2, l) 
    end 
  | normalise_expr (If(e1, e2, e3)) = 
    let val (el1, l1) = normalise_expr e1 
        and (el2, l2) = normalise_expr e2 
        and (el3, l3) = normalise_expr e3 
    in 
       if l2 = l3
       then ([Normal_If(Into(Normal_Seq el1, l1), Normal_Seq el2, Normal_Seq el3)], l2)
       else (* could we be more clever?  Discuss! *) 
            let val l4 = new_location() 
                val new_el2 = Normal_Seq (el2 @ [Normal_Assign(l4, l2)]) 
                val new_el3 = Normal_Seq (el3 @ [Normal_Assign(l4, l3)]) 
            in 
               ([Normal_If(Into(Normal_Seq el1, l1), new_el2, new_el3)], l4)
            end 
    end 
  | normalise_expr (While (e1, e2)) = 
    let val (el1, l1) = normalise_expr e1 
        and (el2, _ ) = normalise_expr e2 
    in 
       ([Normal_While(Into (Normal_Seq el1, l1), Normal_Seq el2)], "_Unit") 
    end 
  | normalise_expr (Print (t_opt, e)) = 
    let val (el, l) = normalise_expr e 
    in 
        (el @ [Normal_Print(t_opt, l)], "_Unit") 
    end 

fun normalise e = 
    let val (el, _) = normalise_expr e 
        and init_code = locs_to_init_code ("_Unit" :: (all_locs [] e))
    in 
      Normal_Seq (init_code @ el)
    end     


