open AST_normal_expr; 
open AST_vsm_assembler; 


val new_offset = Library.new_counter(); 
val next_label = Library.new_counter(); 
fun new_label () = "_l" ^ (Int.toString (next_label ())) 


fun vsm_label_sequence [] = (new_label(), []) 
  | vsm_label_sequence ((VSM_Code oper)        :: rest        ) = 
    let val l = new_label() 
    in 
       (l, (VSM_Labelled(l, oper)) :: rest) 
    end 
  | vsm_label_sequence ((VSM_Labelled(l, oper)) :: rest) = 
       (l, (VSM_Labelled(l, oper)) :: rest) 

fun vsm_insert_remark_in_op new (VSM_Hlt remark)         = VSM_Hlt (new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Nop remark)         = VSM_Nop (new ^ " " ^ remark)                  
  | vsm_insert_remark_in_op new (VSM_Jmp (cl, remark))   = VSM_Jmp (cl, new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Push (c, remark))   = VSM_Push (c, new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Store (dl, remark)) = VSM_Store (dl, new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Load (dl, remark))  = VSM_Load (dl, new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Pop remark)         = VSM_Pop (new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Add remark)         = VSM_Add (new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Sub remark)         = VSM_Sub (new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Mul remark)         = VSM_Mul (new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Ifz (cl, remark))   = VSM_Ifz (cl, new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Ifp (cl, remark))   = VSM_Ifp (cl, new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Ifn (cl, remark))   = VSM_Ifn (cl, new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Pri remark)         = VSM_Pri (new ^ " " ^ remark)
  | vsm_insert_remark_in_op new (VSM_Prb remark)         = VSM_Prb (new ^ " " ^ remark)

fun vsm_insert_remark remark [] = [] 
  | vsm_insert_remark remark ((VSM_Code oper) :: rest) = (VSM_Code (vsm_insert_remark_in_op remark oper)) :: rest
  | vsm_insert_remark remark ((VSM_Labelled(l, oper)) :: rest) = ((VSM_Labelled(l, vsm_insert_remark_in_op remark oper)) :: rest) 


fun find a [] = NONE 
  | find a ((b, v) :: rest) = if a = b then SOME v else find a rest 

fun visit_vsm_data_loc env d = 
    case find d env 
    of NONE => (d, new_offset()) :: env 
     | SOME _ => env 

val vsm_unit_value = 0 
val vsm_true_value = 1 
val vsm_false_value = 0
 
fun expr_to_vsm_code_list env Skip            = 
       (env, [VSM_Code(VSM_Push (vsm_unit_value, "push unit value"))])
  | expr_to_vsm_code_list env (Integer n)     = 
       (env, [VSM_Code(VSM_Push (n, ""))])
  | expr_to_vsm_code_list env (Boolean true)  = 
       (env, [VSM_Code(VSM_Push (vsm_true_value, "push true value"))])
  | expr_to_vsm_code_list env (Boolean false) = 
       (env, [VSM_Code(VSM_Push (vsm_false_value, "push false value"))])
  | expr_to_vsm_code_list env (UnaryOp (Neg, e)) = 
       let val (env1, cl) = expr_to_vsm_code_list env e
       in 
          (env1, [VSM_Code(VSM_Push (0, "start neg ..."))] @ cl @ [VSM_Code (VSM_Sub "... end neg")])
       end 
  | expr_to_vsm_code_list env (UnaryOp (Not, e)) = 
       let val (env1, cl) = expr_to_vsm_code_list env e
           and l_true = new_label() 
           and l_end = new_label() 
       in 
          (env1, 
           (vsm_insert_remark "start not ... " cl) 
           @ [VSM_Code(VSM_Ifz (l_true, "start boolean negation ...")), 
              VSM_Code(VSM_Push (vsm_false_value, "push false")),          (* Ifz consumed top-of-stack! *) 
              VSM_Code(VSM_Jmp (l_end, "")), 
              VSM_Labelled(l_true, VSM_Push (vsm_true_value, "push true")), 
              VSM_Labelled(l_end, VSM_Nop " ... end not")
             ])
      end 
  (* for of of + or * we have "a op b = b op a", so we really don't care about order on stack. 
     Note that L1 semantics dictates left-to-right evaluation. *)  
  | expr_to_vsm_code_list env (Op (Plus, e1, e2)) = 
       let val (env1, cl1) = expr_to_vsm_code_list env e1
           val (env2, cl2) = expr_to_vsm_code_list env1 e2
       in 
          (env2, cl1 @ cl2 @ [VSM_Code (VSM_Add "")]) 
       end 
  | expr_to_vsm_code_list env (Op (Mult, e1, e2)) = 
       let val (env1, cl1) = expr_to_vsm_code_list env e1
           val (env2, cl2) = expr_to_vsm_code_list env1 e2
       in 
          (env2, cl1 @ cl2 @ [VSM_Code (VSM_Mul "")]) 
       end 
  (* note that "sub" semantics is "under-top-of-stack - top-of-stack", so this works! *) 
  | expr_to_vsm_code_list env (Op (Subt, e1, e2)) =     
       let val (env1, cl1) = expr_to_vsm_code_list env e1
           val (env2, cl2) = expr_to_vsm_code_list env1 e2
       in 
          (env2, cl1 @ cl2 @ [VSM_Code (VSM_Sub "")]) 
       end 
  | expr_to_vsm_code_list env (Op (GTEQ, e1, e2)) = 
       let val (env1, cl1) = expr_to_vsm_code_list env e1
           val (env2, cl2) = expr_to_vsm_code_list env1 e2
           val l_false = new_label() 
           val l_end = new_label() 
      in  (env2, 
          (vsm_insert_remark "start >= ... " cl1)
          @ cl2 
          @ [VSM_Code (VSM_Sub ""),       (* "e1 - e2 now on top of stack *) 
             VSM_Code(VSM_Ifn (l_false, "")), 
             VSM_Code(VSM_Push (vsm_true_value, "push true")), 
             VSM_Code(VSM_Jmp (l_end, "")), 
             VSM_Labelled(l_false, VSM_Push (vsm_false_value, "push false")), 
             VSM_Labelled(l_end,  VSM_Nop " ... end >=")
            ])
      end 
  | expr_to_vsm_code_list env (Assign(l, _, e)) = 
       let val (env1, cl) = expr_to_vsm_code_list env e
           val env2 = visit_vsm_data_loc env1 l 
       in case find l env2
            of SOME offset => 
	        (env2, cl 
                       @ [VSM_Code (VSM_Store (offset, "store " ^ l)), 
                          VSM_Code(VSM_Push (vsm_unit_value, "push unit value"))])
             | NONE => Library.internal_error "expr_to_vsm_code_list(1) : This cannot happen!"
       end 
  | expr_to_vsm_code_list env (Deref l) = 
       (* all locations are implicitly initialized to 0 *) 
       let val env2 = visit_vsm_data_loc env l 
       in case find l env2
          of SOME offset => (env2, [VSM_Code (VSM_Load (offset, "load " ^ l))])
           | NONE => Library.internal_error "expr_to_vsm_code_list(2) : This cannot happen!"	
       end
  | expr_to_vsm_code_list env (Seq (e1, e2)) = 
       let val (env1, cl1) = expr_to_vsm_code_list env e1
           val (env2, cl2) = expr_to_vsm_code_list env1 e2
       in 
          (env2, cl1 @ [VSM_Code (VSM_Pop "sequence pop")] @ cl2) 
       end 
  | expr_to_vsm_code_list env (If(e0, e1, e2))        =  
    let val (env0, cl0) = expr_to_vsm_code_list env e0 
        val (env1, cl1) = expr_to_vsm_code_list env0 e1 
        val (env2, cl2) = expr_to_vsm_code_list env1 e2 
    in 
        let val (l2, cl3) = vsm_label_sequence cl2 
            and l3 = new_label() 
        in 
           (env2, VSM_Code(VSM_Ifz (l2, " start if ...")) 
                  :: (cl1 
                      @ [VSM_Code(VSM_Jmp (l3, " ... end then ..."))] 
                      @ (vsm_insert_remark " start else ... " cl3 )
                      @ [VSM_Labelled(l3, VSM_Nop "... end if")]))
        end 
    end  
  | expr_to_vsm_code_list env (While(e1, e2))     = 
    let val (env1, cl1) = expr_to_vsm_code_list env e1
        val (env2, cl2) = expr_to_vsm_code_list env1 e2
        val l2 = new_label()   
    in 
        let val (l3, cl3) = vsm_label_sequence cl1
        in 
	   (env2, (vsm_insert_remark " start while ... " cl3) 
                  @ [VSM_Code(VSM_Ifz (l2, " test of while ..."))] 
                  @ cl2 
                  @ [VSM_Code (VSM_Pop "end-of-while-body pop"), 
                     VSM_Code(VSM_Jmp (l3, "... jump to while condition ")), 
                     VSM_Labelled(l2, VSM_Nop  "... end while")])
        end 
    end  
  | expr_to_vsm_code_list env (Print(SOME TEint, e))  = 
    let val (env1, cl) = expr_to_vsm_code_list env e
    in 
       (env1, cl 
              @ [VSM_Code (VSM_Pri ""), 
                 VSM_Code(VSM_Push (vsm_unit_value, "push unit value from print"))])
    end 
  | expr_to_vsm_code_list env (Print(NONE, l))        = 
       Library.internal_error "expr_to_vsm_code_list env : Print has no type"
  | expr_to_vsm_code_list env (Print(SOME t , l))     = 
       Library.internal_error ("expr_to_vsm_code_list env : Print has type " ^ (type_expr_to_string t))


fun env_to_init_code env = 
    let fun aux [] = [] 
          | aux ((l, _) :: rest ) = 
            (VSM_Code(VSM_Push (0, "slot for " ^ l))) :: (aux rest) 
    in aux (List.rev env) end 

fun vsm_code_gen e = 
    let val (env', cl) = expr_to_vsm_code_list [] e
    in 
       (env_to_init_code env') 
        @ cl 
        @ [VSM_Code (VSM_Hlt " that's all folks!")] 
    end 


(* Our peep-hole optimization eliminates 

    push c
    pop 

  and simplifies 

    l : nop 
        inst 

  to 

    l : inst 

  We could attempt to preserve comments, but currently do not ...  

*) 
fun vsm_peep_hole ((VSM_Code(VSM_Push _)) :: ((VSM_Code(VSM_Pop _)) :: rest)) = 
       vsm_peep_hole rest 
  | vsm_peep_hole ((VSM_Labelled(l, VSM_Nop _)) :: ((VSM_Code(c)) :: rest)) = 
       (VSM_Labelled(l, c)) :: (vsm_peep_hole rest) 
  | vsm_peep_hole (c :: rest) = c :: (vsm_peep_hole rest)
  | vsm_peep_hole [] = [] 

